package com.hupu.tv.player.app.base

import android.graphics.Color

import android.view.View
import android.widget.ImageView
import android.widget.LinearLayout
import android.widget.TextView
import androidx.annotation.ColorRes
import androidx.appcompat.app.AppCompatActivity
import androidx.core.content.ContextCompat
import androidx.recyclerview.widget.OrientationHelper
import androidx.recyclerview.widget.RecyclerView
import androidx.recyclerview.widget.SimpleItemAnimator
import com.chad.library.adapter.base.BaseQuickAdapter
import com.chad.library.adapter.base.BaseQuickAdapter.RequestLoadMoreListener
import com.scwang.smartrefresh.layout.SmartRefreshLayout
import com.softgarden.baselibrary.base.BaseLazyFragment
import com.softgarden.baselibrary.base.IBasePresenter
import com.softgarden.baselibrary.utils.DisplayUtil
import com.softgarden.baselibrary.utils.EmptyUtil
import com.softgarden.baselibrary.utils.ScreenUtil
import com.softgarden.baselibrary.widget.ColorDividerDecoration
import com.softgarden.baselibrary.widget.GridDividerDecoration
import com.hupu.tv.player.app.R
import com.hupu.tv.player.app.utils.EmptyConfig

/**
 * @author by DELL
 * @date on 2020/06/11
 * @describe 通用的列表刷新Fragment
 * 上拉刷新用的是 SmartRefreshLayout
 * 下拉加载用的是 BaseAdapter带的
 * 注意事项：
 * 刷新控件 id必须为：mRefreshLayout
 * RecyclerView id必须为：mRecyclerView
 * 当然也可以自己调整
 */
abstract class RefreshFragment<P : IBasePresenter?> : BaseLazyFragment<P>(),
    RequestLoadMoreListener {
    /*** 每页请求数量  */
    var PAGE_COUNT = 10

    /*** 页码，默认从1开始  */
    protected var PageStart = 1
    protected var mPage = 1
    protected var isRefresh = false

    /*** 空布局类型 可以在这里设置默认值  */
    protected var mEmptyType = 0

    /*** 是否启用空布局  */
    private var enableEmptyView = true
    protected var mRefreshLayout: SmartRefreshLayout? = null
    protected var mRecyclerView: RecyclerView? = null
    private var mAdapter: BaseQuickAdapter<*, *>? = null
//    public var headerView: CustomClassicsHeader? = null

    /**
     * 设置刷新框架，需要时调用即可
     */
    protected fun initRefreshLayout() {
        mRefreshLayout = view?.findViewById(R.id.mRefreshLayout)
        mRefreshLayout?.let {
//            headerView = CustomClassicsHeader(context)
            it.isEnabled = true
            it.setEnableLoadMore(false)
            it.setEnableRefresh(true)
//            it.setRefreshHeader(headerView!!)
            it.setDisableContentWhenLoading(true)
            it.setDisableContentWhenRefresh(true)
            it.setOnRefreshListener { onRefresh() }
        }
//        headerView?.setDrawableSize(10f)

    }

    fun setPageCount(pageCount: Int) {
        PAGE_COUNT = pageCount
    }

    override fun onFirstUserVisible() {
        mPage = PageStart
        super.onFirstUserVisible()
    }


    /**
     * 开关刷新
     *
     * @param enableRefresh
     */
    fun setEnableRefresh(enableRefresh: Boolean) {
        mRefreshLayout?.setEnableRefresh(enableRefresh)
    }

    /**
     * 初始化列表控件
     */
    protected fun initRecyclerView() {
        mRecyclerView = requireView().findViewById<View>(R.id.mRecyclerView) as RecyclerView
        (mRecyclerView!!.itemAnimator as SimpleItemAnimator).supportsChangeAnimations = false
    }


    /**
     * 设置分割线
     *
     * @param id
     */
    protected fun addItemDecoration(@ColorRes id: Int) {
        mRecyclerView?.addItemDecoration(
            ColorDividerDecoration(
                ContextCompat.getColor(
                    requireActivity(),
                    id
                )
            )
        )
    }

    protected fun addItemDecoration(@ColorRes id: Int, dp: Int) {
        mRecyclerView?.addItemDecoration(
            ColorDividerDecoration(
                ContextCompat.getColor(
                    requireActivity(),
                    id
                ), DisplayUtil.dip2px(requireActivity(), dp.toFloat())
            )
        )
    }

    protected fun addItemDecoration(
        recyclerView: RecyclerView?,
        @ColorRes id: Int,
        dp: Int,
        mOrientation: Int = OrientationHelper.VERTICAL
    ) {
        recyclerView?.addItemDecoration(
            ColorDividerDecoration(
                mOrientation,
                ContextCompat.getColor(requireActivity(), id),
                DisplayUtil.dip2px(requireActivity(), dp.toFloat())
            )
        )
    }

    protected fun addGridItemDecoration(@ColorRes id: Int, dp: Int) {
        mRecyclerView?.addItemDecoration(
            GridDividerDecoration(
                DisplayUtil.dip2px(
                    requireActivity(),
                    dp.toFloat()
                ), ContextCompat.getColor(requireActivity(), id)
            )
        )
    }

    protected fun addGridItemDecoration(recyclerView: RecyclerView?, @ColorRes id: Int, dp: Int) {
        recyclerView?.addItemDecoration(
            GridDividerDecoration(
                DisplayUtil.dip2px(
                    requireActivity(),
                    dp.toFloat()
                ), ContextCompat.getColor(requireActivity(), id)
            )
        )
    }

    protected fun addItemDecoration() {
        //设置默认分割线
        mRecyclerView?.addItemDecoration(ColorDividerDecoration(Color.parseColor("#CCCCCC")))
    }

    /**
     * 自动刷新
     */
    fun autoRefresh() {
        if (mRefreshLayout != null) mRefreshLayout!!.autoRefresh(0)
    }

    /**
     * 结束刷新
     */
    fun finishRefresh() {
        if (mRefreshLayout != null) mRefreshLayout!!.finishRefresh(0)
    }

    /***
     * 是否启用空布局
     * @param enableEmptyView
     */
    fun setEnableEmptyView(enableEmptyView: Boolean) {
        this.enableEmptyView = enableEmptyView
    }

    fun setEnableEmptyView(enableEmptyView: Boolean, emptyType: Int) {
        this.enableEmptyView = enableEmptyView
        this.mEmptyType = emptyType
    }

    fun setLoadData(
        adapter: BaseQuickAdapter<*, *>,
        list: List<*>?,
        showEmptyView: Boolean = true
    ) {
        setLoadData(adapter, list, mEmptyType, showEmptyView) //emptyType 可以给一个默认值,作为统一设置
    }

    fun setLoadData(adapter: BaseQuickAdapter<*, *>, list: List<*>?, emptyType: Int) {
        setLoadData(adapter, list, emptyType, true) //emptyType 可以给一个默认值,作为统一设置
    }

    /**
     * 不分页的 设置数据
     *
     * @param adapter
     * @param list
     * @param emptyType
     */
    fun setLoadData(
        adapter: BaseQuickAdapter<*, *>,
        list: List<*>?,
        emptyType: Int,
        showEmptyView: Boolean
    ) {
        mAdapter = adapter
        this.mEmptyType = emptyType
        adapter.setHeaderFooterEmpty(true, true) //是否显示头部和底部
        finishRefresh()
        adapter.setNewData(list as List<Nothing>?)
        if (EmptyUtil.isEmpty(list)) {
            setEmptyView(adapter, showEmptyView)
        }
    }

    fun setLoadMore(adapter: BaseQuickAdapter<*, *>, list: List<*>?) {
        setLoadMore(adapter, list, mEmptyType, true)
    }


    /**
     * 如果使用的是当前的 mRecyclerView
     *
     * @param adapter
     * @param list
     * @param emptyType
     */
    fun setLoadMore(
        adapter: BaseQuickAdapter<*, *>,
        list: List<*>?,
        emptyType: Int,
        showEmptyView: Boolean = true
    ) {
        if (mRecyclerView != null) {
            setLoadMore(mRecyclerView, adapter, list, emptyType, showEmptyView)
        }
    }

    fun setLoadMore(
        adapter: BaseQuickAdapter<*, *>,
        list: List<*>?,
        showEmptyView: Boolean = true
    ) {
        if (mRecyclerView != null) {
            setLoadMore(mRecyclerView, adapter, list, mEmptyType, showEmptyView)
        }
    }


    /**
     * 结束刷新
     * 自动设置上拉更多
     *
     * @param adapter
     * @param list
     */
    fun setLoadMore(
        recyclerView: RecyclerView?,
        adapter: BaseQuickAdapter<*, *>,
        list: List<*>?,
        emptyType: Int = mEmptyType,
        showEmptyView: Boolean = true
    ) {
        mAdapter = adapter
        this.mEmptyType = emptyType
        adapter.setHeaderFooterEmpty(true, true) //是否显示头部和底部
        finishRefresh() //结束刷新
        if (mPage == PageStart) {
            adapter.setNewData(list as List<Nothing>?)
            if (EmptyUtil.isEmpty(list)) {
                setEmptyView(adapter, showEmptyView)
            }
        } else {
            list?.run {
                adapter.addData(this as List<Nothing>)
            }
        }
        handleLoadMore(recyclerView, adapter, list)
    }


    fun handleLoadMore(
        recyclerView: RecyclerView?,
        adapter: BaseQuickAdapter<*, *>,
        list: List<*>?
    ) {
        if (list == null || list.size < PAGE_COUNT) {
            adapter.loadMoreEnd()
        } else {
            adapter.setEnableLoadMore(true)
            adapter.setOnLoadMoreListener(this, recyclerView)
            adapter.loadMoreComplete()
        }
    }

    /**
     * 设置空状态
     *
     *
     * 根据项目情况 自由定制
     * @param adapter
     */

    protected fun setEmptyView(adapter: BaseQuickAdapter<*, *>?, showEmptyView: Boolean = true) {
        if (showEmptyView) {

            val emptyView = View.inflate(activity, R.layout.layout_empty, null) as LinearLayout
            val ivEmpty = emptyView.findViewById<ImageView>(R.id.iv_empty)
            val tvEmpty = emptyView.findViewById<TextView>(R.id.tv_empty)
            val tvEmptyHint = emptyView.findViewById<TextView>(R.id.tv_empty_hint)
            var hintDef = "暂无数据"
            var hintDefDescribe = "暂无数据"
            var imgResId = R.mipmap.place_empty
            when (mEmptyType) {
                EmptyConfig.NO_HISTORY_DATA -> {
                    hintDef = EmptyConfig.NO_HISTORY_DATA_TEXT
                    hintDefDescribe = EmptyConfig.NO_HISTORY_DATA_TEXT_HINT
//                    var params = tvEmptyHint.layoutParams as RelativeLayout.LayoutParams
//                    params.bottomMargin = ((ScreenUtil.getScreenRealHeight(activity as AppCompatActivity) - emptyView.height) / 2f).toInt()
                }
                EmptyConfig.NO_DOWNLOAD_DATA -> {
                    hintDef = EmptyConfig.NO_DOWNLOAD_DATA_TEXT
                    hintDefDescribe = EmptyConfig.NO_DOWNLOAD_DATA_TEXT_HINT
//                    var params = tvEmptyHint.layoutParams as RelativeLayout.LayoutParams
//                    params.bottomMargin = ((ScreenUtil.getScreenRealHeight(activity as AppCompatActivity) - emptyView.height) / 2f).toInt()
                }
                EmptyConfig.NO_COLLECTION -> {
                    imgResId = R.mipmap.place_empty
                    hintDefDescribe = EmptyConfig.NO_COLLECTION_TEXT_HINT
                }

                EmptyConfig.NO_TIK_TOK -> {
                    imgResId = R.mipmap.place_empty
                    hintDefDescribe = EmptyConfig.NO_TIK_TOK_TEXT_HINT
                }

                EmptyConfig.NO_FOLLOW_ANCHOR -> {

                    hintDef = EmptyConfig.NO_FOLLOW_ANCHOR_TEXT
                    hintDefDescribe = EmptyConfig.NO_FOLLOW_ANCHOR_HINT
                }

                EmptyConfig.NO_FOLLOW_MATCH -> {

                    hintDef = EmptyConfig.NO_FOLLOW_MATCH_TEXT
                    hintDefDescribe = EmptyConfig.NO_FOLLOW_MATCH_HINT
                }
                EmptyConfig.NO_DETAIL_LIVE -> {

                    hintDef = EmptyConfig.NO_DETAIL_LIVE_TEXT
                    hintDefDescribe = EmptyConfig.NO_DETAIL_LIVE_HINT
                }
                EmptyConfig.NO_DETAIL_RED -> {

                    hintDef = EmptyConfig.NO_DETAIL_RED_TEXT
                    hintDefDescribe = EmptyConfig.NO_DETAIL_RED_HINT
                }
                else -> {
                    //这里是其他布局，主页等
//                    setEmptyLayouParams(emptyView)
                }

            }
            setEmptyLayoutParams(emptyView)
            ivEmpty.setImageResource(imgResId)
            tvEmpty.text = hintDef
            tvEmptyHint.text = hintDefDescribe
            adapter?.emptyView = emptyView
        }
    }

    private fun setEmptyLayoutParams(emptyView: LinearLayout) {
        emptyView.layoutParams = LinearLayout.LayoutParams(
            LinearLayout.LayoutParams.MATCH_PARENT,
            LinearLayout.LayoutParams.MATCH_PARENT
        )
        val layoutParams = emptyView.layoutParams as LinearLayout.LayoutParams
        //获取不到父布局的padding，默认设置的是18dp(左右），转换为px，所以固定写死
        val paddingStart = mRecyclerView?.paddingStart ?: 0
        val paddingEnd = mRecyclerView?.paddingEnd ?: 0
        layoutParams.width =
            ScreenUtil.getScreenWidth(activity as AppCompatActivity) - paddingStart - paddingEnd
        //                    layoutParams.width = ScreenUtil.getScreenWidth(activity as AppCompatActivity)
        emptyView.requestLayout()
    }


    /**
     * 结束刷新
     */
    override fun onRequestFinish() {
        super.onRequestFinish()
        finishRefresh()
    }

    open fun onRefresh() {
        isRefresh = true
        mPage = 1
        lazyLoad()
    }

    /**
     * 如需上拉更多 请重写该方法
     */
    override fun onLoadMoreRequested() {
        mPage++
        lazyLoad()
    }
}