package com.hupu.tv.player.app.utils

import android.Manifest
import android.content.Context
import android.content.DialogInterface
import android.content.Intent
import android.graphics.Bitmap
import android.net.Uri
import android.widget.ImageView
import androidx.appcompat.app.AppCompatActivity
import com.bumptech.glide.Glide
import com.bumptech.glide.load.engine.DiskCacheStrategy
import com.bumptech.glide.request.RequestOptions
import com.bumptech.glide.request.target.Target
import com.softgarden.baselibrary.base.BaseActivity
import com.softgarden.baselibrary.dialog.LoadingDialog
import com.softgarden.baselibrary.utils.*
import com.trello.rxlifecycle4.android.ActivityEvent
import io.reactivex.rxjava3.core.Observable
import com.hupu.tv.player.app.R
import io.reactivex.rxjava3.android.schedulers.AndroidSchedulers
import io.reactivex.rxjava3.core.ObservableEmitter
import io.reactivex.rxjava3.core.ObservableOnSubscribe
import io.reactivex.rxjava3.disposables.Disposable
import io.reactivex.rxjava3.schedulers.Schedulers

import java.io.File
import java.io.FileNotFoundException
import java.io.FileOutputStream
import java.io.IOException
import java.util.*

/**
 * @author by DELL
 * @date on 2020/06/11
 * @describe
 */
object ImageUtil {
    val TAG = ImageUtil::class.java.name


    fun getImageUrl(url: String?): String {
        return "自定义路径$url"
    }

    /**
     * 检查url 是否拼接
     *
     * @param url
     * @return
     */
    fun checkUrl(url: String?): String? {
        //有完整路径的url不用拼接 本地图片也不用
        var url = url
        if (url != null && (url.startsWith("http://")
                        || url.startsWith("https://")
                        || url.startsWith("file:///android_asset/")
                        || url.startsWith("file:///android_asset/")
                        || url.startsWith(FileUtil.appRootDir!!.absolutePath))) {
        } else {
            url = getImageUrl(url)
        }
        return url
    }

    /**
     * 检查url集合 是否拼接
     *
     * @param urlList
     * @return
     */
    fun checkUrl(urlList: List<String?>): List<String?> {
        val imgList: MutableList<String?> = ArrayList()
        if (EmptyUtil.isEmpty(urlList)) {
            return imgList
        }
        for (path in urlList) {
            imgList.add(checkUrl(path))
        }
        return imgList
    }

    /**
     * 加载图片正方形图片
     * 默认图是正方形
     *
     * @param imageView
     * @param url
     */
    fun load(imageView: ImageView, url: String?) {
        Glide.with(imageView.context)
                .load(checkUrl(url))
                .apply(RequestOptions.placeholderOf(R.mipmap.ic_launcher)
                        .error(R.mipmap.ic_launcher)
                        .dontAnimate()
                        .diskCacheStrategy(DiskCacheStrategy.ALL))
                .into(imageView)
    }

    /**
     * 加载drawable资源
     *
     * @param imageView
     * @param resId
     */
    fun loadSrc(imageView: ImageView, resId: Int) {
        Glide.with(imageView.context)
                .load(resId)
                .apply(RequestOptions.noAnimation()
                        .diskCacheStrategy(DiskCacheStrategy.ALL))
                .into(imageView)
    }

    /**
     * 加载本地视频 自动获取缩略图
     *
     * @param imageView
     * @param path
     */
    fun loadLocalVideo(imageView: ImageView, path: String?) {
        Glide.with(imageView.context)
                .load(path)
                .apply(RequestOptions.placeholderOf(R.mipmap.ic_launcher)
                        .error(R.mipmap.ic_launcher)
                        .dontAnimate()
                        .diskCacheStrategy(DiskCacheStrategy.ALL))
                .into(imageView)
    }

    /**
     * 加载本地图片 最好override一下宽高，不然滑动会卡顿
     *
     * @param imageView
     * @param url
     * @param width
     */
    fun loadLocalSize(imageView: ImageView, url: String?, width: Int) {
        Glide.with(imageView.context)
                .load(url)
                .apply(RequestOptions.placeholderOf(R.mipmap.ic_launcher)
                        .error(R.mipmap.ic_launcher)
                        .override(width, width)
                        .dontAnimate()
                        .diskCacheStrategy(DiskCacheStrategy.ALL))
                .into(imageView)
    }

    /**
     * 加载头像
     *
     * @param imageView
     * @param url
     */
    fun loadHeader(imageView: ImageView, url: String?) {
        Glide.with(imageView.context)
                .load(checkUrl(url))
                .apply(RequestOptions.placeholderOf(R.mipmap.ic_launcher)
                        .error(R.mipmap.ic_launcher)
                        .dontAnimate()
                        .diskCacheStrategy(DiskCacheStrategy.ALL))
                .into(imageView)
    }

    /**
     * 加载大头像
     * 只是默认图不同
     *
     * @param imageView
     * @param url
     */
    fun loadBigHeader(imageView: ImageView, url: String?) {
        Glide.with(imageView.context)
                .load(checkUrl(url))
                .apply(RequestOptions.placeholderOf(R.mipmap.ic_launcher)
                        .error(R.mipmap.ic_launcher)
                        .dontAnimate()
                        .diskCacheStrategy(DiskCacheStrategy.ALL))
                .into(imageView)
    }

    /**
     * 没有占位图
     *
     * @param imageView
     * @param url
     */
    fun loadNoHolder(imageView: ImageView, url: String?) {
        Glide.with(imageView.context)
                .load(checkUrl(url))
                .apply(RequestOptions.noAnimation()
                        .diskCacheStrategy(DiskCacheStrategy.ALL)) //  .placeholder(R.mipmap.ic_launcher)
                //  .error(R.mipmap.ic_launcher)
                .into(imageView)
    }

    /**
     * 高斯模糊
     *
     * @param imageView
     * @param url
     */
    fun loadBlur(imageView: ImageView, url: String?) {
        Glide.with(imageView.context)
                .load(checkUrl(url))
                .apply(RequestOptions.placeholderOf(R.mipmap.ic_launcher)
                        .error(R.mipmap.ic_launcher)
                        .dontAnimate() // .bitmapTransform(new BlurTransformation(imageView.getContext(), 1, 3))
                        .diskCacheStrategy(DiskCacheStrategy.ALL))
                .into(imageView)
    }

    fun loadBlur(imageView: ImageView, url: String?, radius: Int, sampling: Int) {
        // radius "23":模糊度；sampling "4":图片缩放4倍后再进行模糊a
        Glide.with(imageView.context)
                .load(checkUrl(url))
                .apply(RequestOptions.placeholderOf(R.mipmap.ic_launcher)
                        .error(R.mipmap.ic_launcher)
                        .dontAnimate() // .bitmapTransform(new BlurTransformation(imageView.getContext(), radius, sampling))
                        .diskCacheStrategy(DiskCacheStrategy.ALL))
                .into(imageView)
    }

    fun loadAsGif(imageView: ImageView, url: String?) {
        Glide.with(imageView.context)
                .asGif()
                .thumbnail(0.5f)
                .load(checkUrl(url))
                .apply(RequestOptions.placeholderOf(R.mipmap.ic_launcher)
                        .error(R.mipmap.ic_launcher)
                        .dontAnimate()
                        .diskCacheStrategy(DiskCacheStrategy.ALL))
                .into(imageView)
    }

    /**
     * 清除缓存
     * [FileUtil.clearImageAllCache]
     *
     * @param context
     * @return
     */
    fun clearCache(context: BaseActivity<*>): Observable<Boolean> {
        return Observable.create(ObservableOnSubscribe { e: ObservableEmitter<Boolean?> ->
            // GlideApp.get(context).clearDiskCache();
            FileUtil.clearImageAllCache()
            e.onNext(true)
            e.onComplete()
        }).subscribeOn(Schedulers.newThread())
                .observeOn(AndroidSchedulers.mainThread())
                .doOnSubscribe { disposable: Disposable? -> context.showProgressDialog() }
                .doOnTerminate { context.hideProgressDialog() }
                .compose(context.bindUntilEvent(ActivityEvent.DESTROY))
    }

    /**
     * 保存图片集合到sd卡
     *
     * @param context
     * @param imgList
     * @param isSilentLoad 是否静默下载
     */
    fun loadImagesToSDCard(context: AppCompatActivity, imgList: List<String>?, isSilentLoad: Boolean) {
        if (imgList == null || imgList.isEmpty()) return
        //申请权限
        RxPermissionsUtil.request(context, *RxPermissionsUtil.STORAGE).subscribe { aBoolean: Boolean ->
            if (aBoolean) {
                val dialog = LoadingDialog(context)
                if (!isSilentLoad) {
                    dialog.show()
                }
                Observable.create(ObservableOnSubscribe { e: ObservableEmitter<List<String>> ->
                    val count: MutableList<String> = ArrayList()
                    for (path in imgList) {
                        save2SDCard(context, checkUrl(path))
                                .subscribe({ bitmap: Bitmap ->
                                    saveBitmap2File(context, bitmap, FileUtil.appRootDir, path.hashCode().toString() + ".jpeg")
                                    count.add(path)
                                    if (!e.isDisposed) e.onNext(count)
                                }) { throwable: Throwable? -> if (!isSilentLoad) ToastUtil.s(path + "下载失败") }
                    }
                } as ObservableOnSubscribe<List<String>>).subscribeOn(Schedulers.io())
                        .observeOn(AndroidSchedulers.mainThread())
                        .subscribe({ count: List<String> ->
                            if (count.size == imgList.size) {
                                if (!isSilentLoad) {
                                    dialog.dismiss()
                                    ToastUtil.l(String.format("图片已保存至%s文件夹内", FileUtil.appRootDir))
                                }
                            }
                        }) { throwable: Throwable? ->
                            if (!isSilentLoad) {
                                dialog.dismiss()
                                ToastUtil.s("下载失败")
                            }
                        }
            } else { //被拒绝 要去再次申请
                RxPermissionsUtil.shouldShowRequestPermissionRationale(context, Manifest.permission.WRITE_EXTERNAL_STORAGE)
                        .subscribe { aBoolean1: Boolean ->
                            if (aBoolean1) { // 查询是否 应该提醒
                                RxPermissionsUtil.showPermissionDialog(context,
                                        "权限申请", "此功能需要读写存储权限，请授权"
                                , DialogInterface.OnClickListener { dialog, which -> loadImagesToSDCard(context, imgList, isSilentLoad) })
                            } else { //选中了禁止以后提醒
                                RxPermissionsUtil.showLackPermissionDialog(context)
                            }
                        }
            }
        }
    }

    fun loadImagesToSDCard(activity: AppCompatActivity, imageList: List<String>?) {
        loadImagesToSDCard(activity, imageList, false)
    }

    /**
     * 下载图片到本地SDCard
     *
     * @param context
     * @param imageUrl
     * @param saveName
     * @param isSilentLoad
     */
    fun loadImageToSDCard(context: AppCompatActivity, imageUrl: String?, saveName: String, isSilentLoad: Boolean) {
        if (EmptyUtil.isEmpty(imageUrl)) return

        //申请权限
        RxPermissionsUtil.request(context, *RxPermissionsUtil.STORAGE).subscribe { aBoolean: Boolean ->
            if (aBoolean) {
                val dialog = LoadingDialog(context)
                if (!isSilentLoad) {
                    dialog.show()
                }
                Observable.create(ObservableOnSubscribe { e: ObservableEmitter<String?> ->
                    save2SDCard(context, checkUrl(imageUrl))
                            .subscribe { bitmap: Bitmap ->
                                saveBitmap2File(context, bitmap, FileUtil.appRootDir, saveName)
                                e.onNext(" ")
                            }
                }).subscribeOn(Schedulers.io())
                        .observeOn(AndroidSchedulers.mainThread())
                        .subscribe({ count: String? ->
                            if (!isSilentLoad) {
                                dialog.dismiss()
                                ToastUtil.l(String.format("图片已保存至%s文件夹内", FileUtil.appRootDir))
                            }
                        }) { throwable: Throwable? ->
                            if (!isSilentLoad) {
                                dialog.dismiss()
                                ToastUtil.s("保存失败")
                            }
                        }
            } else { //被拒绝 要去再次申请
                RxPermissionsUtil.shouldShowRequestPermissionRationale(context, Manifest.permission.WRITE_EXTERNAL_STORAGE)
                        .subscribe { aBoolean1: Boolean ->
                            if (aBoolean1) { // 查询是否 应该提醒
                                RxPermissionsUtil.showPermissionDialog(context,
                                        "权限申请", "此功能需要读写存储权限，请授权"
                                        , DialogInterface.OnClickListener { dialog, which -> loadImageToSDCard(context, imageUrl, saveName, isSilentLoad) })
                            } else { //选中了禁止以后提醒
                                RxPermissionsUtil.showLackPermissionDialog(context)
                            }
                        }
            }
        }
    }

    fun loadImageToSDCard(activity: AppCompatActivity, imageUrl: String) {
        loadImageToSDCard(activity, imageUrl, imageUrl.hashCode().toString() + ".jpeg", false)
    }

    /**
     * @param context
     * @param url     图片的下载地址
     */
    fun save2SDCard(context: Context?, url: String?): Observable<Bitmap> {
        L.d(TAG, "save2SDCard url=$url")
        return Observable.create<Bitmap> { e ->
            val bitmap = Glide.with(context!!)
                    .asBitmap()
                    .load(url)
                    .submit(Target.SIZE_ORIGINAL, Target.SIZE_ORIGINAL)
                    .get()
            if (!e.isDisposed) e.onNext(bitmap)
        }.subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
    }

    /**
     * 将Bitmap保存为图片文件
     *
     * @param context
     * @param bitmap
     * @param dirFile
     * @param saveName
     */
    fun saveBitmap2File(context: Context, bitmap: Bitmap, dirFile: File?, saveName: String?) {
        if (dirFile != null && dirFile.exists() && dirFile.isDirectory) {
            val file = File(dirFile, saveName)
            L.d(TAG, "saveBitmap2File  file.getAbsolutePath()=" + file.absolutePath)
            var fos: FileOutputStream? = null
            try {
                fos = FileOutputStream(file)
                bitmap.compress(Bitmap.CompressFormat.JPEG, 100, fos)
                fos.flush()
            } catch (e: FileNotFoundException) {
                e.printStackTrace()
            } catch (e: IOException) {
                e.printStackTrace()
            } finally {
                try {
                    fos?.close()
                } catch (e: IOException) {
                    e.printStackTrace()
                }
            }
            context.sendBroadcast(Intent(Intent.ACTION_MEDIA_SCANNER_SCAN_FILE, Uri.fromFile(file))) //刷新相册
        } else {
            throw RuntimeException("the file is not exists or it is not directory !")
        }
    }
}