package com.hupu.tv.player.app.widget.danmuku

import android.annotation.SuppressLint
import android.app.Activity
import android.content.Context
import android.graphics.Color
import android.graphics.RectF
import android.os.SystemClock
import android.util.Log
import android.view.GestureDetector
import android.view.MotionEvent
import android.view.View
import android.view.ViewConfiguration
import android.widget.*
import androidx.appcompat.widget.SwitchCompat
import com.hupu.tv.player.app.R
import com.hupu.tv.player.app.utils.CommonUtils
import com.kuaishou.akdanmaku.DanmakuConfig
import com.kuaishou.akdanmaku.data.DanmakuItemData
import com.kuaishou.akdanmaku.ecs.DanmakuEngine
import com.kuaishou.akdanmaku.ecs.component.action.Actions
import com.kuaishou.akdanmaku.ecs.component.filter.*
import com.kuaishou.akdanmaku.ui.DanmakuPlayer
import com.kuaishou.akdanmaku.ui.DanmakuView
import kotlin.random.Random


/**
 * 快手开源的 Danmaku
 *
 * 使用Native libGdx 开源的 引擎
 *
 * Demo和更多Api 见 github 地址或者  AkDanmaku 项目
 *
 * https://github.com/KwaiAppTeam/AkDanmaku
 */
@SuppressLint("ClickableViewAccessibility")
class DanmakuPlayController(
    activity: Context,
    private val danmakuPlayer: DanmakuPlayer,
    val danmakuView: DanmakuView
) {

    private val hitRect = RectF()

    /** 触摸手势 这里处理 */
    private val gestureDetector = GestureDetector(danmakuView.context, object : GestureDetector.SimpleOnGestureListener() {

        override fun onDown(e: MotionEvent?): Boolean {
            return false
        }

        override fun onSingleTapConfirmed(e: MotionEvent): Boolean {
            val range = ViewConfiguration.get(activity).scaledTouchSlop
            hitRect.set(e.x - range, e.y - range, e.x + range, e.y + range)
            val danmakus = danmakuPlayer.getDanmakusInRect(hitRect)
            val item = danmakus?.firstOrNull()
            danmakuPlayer.hold(item)
            if (item == null && config.alpha != 1f) {
                config = config.copy(alpha = 1f)
                danmakuPlayer.updateConfig(config)
            } else if (item != null && config.alpha == 1f) {
                config = config.copy(alpha = 0.3f)
                danmakuPlayer.updateConfig(config)
            }
            return false
        }
    })

    private var isPlaying = false
    private val colorFilter = TextColorFilter()
    private var dataFilters = emptyMap<Int, DanmakuFilter>()
    private var config = DanmakuConfig().apply {
        dataFilter = createDataFilters()
        dataFilters = dataFilter.associateBy { it.filterParams }
        layoutFilter = createLayoutFilters()
        textSizeScale = 0.8f
    }

    private val textScaleList = listOf(
        0.8f, 1.0f, 1.25f, 2f
    )
    private var textScaleIndex = 0

    init {
        danmakuView.setOnTouchListener { _, event ->
            return@setOnTouchListener gestureDetector.onTouchEvent(event)
        }
    }

    fun start() {
        isPlaying = true
        danmakuPlayer.start(config)
    }

    /** 显示和隐藏 弹幕*/
    fun setVisibility(visible: Boolean) {
        config = config.copy(visibility = visible)
        danmakuPlayer.updateConfig(config)
    }

    /** 发送danmu弹幕*/
    fun sendDanmaku(text: String, isMine: Boolean) {
        val danmaku = DanmakuItemData(
            Random.nextLong(),
            danmakuPlayer.getCurrentTimeMs() + 500,
            text,
            DanmakuItemData.DANMAKU_MODE_ROLLING,
            25,
            Color.parseColor(CommonUtils.getRandColor()),
            9,
            if (isMine) DanmakuItemData.DANMAKU_STYLE_ICON_UP else DanmakuItemData.DANMAKU_STYLE_NONE,
            9
        )
        val item = danmakuPlayer.obtainItem(danmaku)
        /** 弹幕添加动画*/
//        val sequenceAction = Actions.sequence(
//            Actions.rotateBy(360f, 1000L),
//            Actions.scaleTo(1.5f, 1.5f, 500L),
//            Actions.scaleTo(0.8f, 0.8f, 300L)
//        )
//        item.addAction(
//            Actions.moveBy(0f, 300f, 1735L),
//            sequenceAction,
//            Actions.sequence(Actions.fadeOut(500L), Actions.fadeIn(300L))
//        )
        danmakuPlayer.send(item)
    }

    fun pause() {
        isPlaying = false
        danmakuPlayer.pause()
    }


    /** 缩放文字大小 */
    private fun switchTextScale() {
        textScaleIndex = (textScaleIndex + 1) % textScaleList.size
        config = config.copy(textSizeScale = textScaleList[textScaleIndex])
        danmakuPlayer.updateConfig(config)
    }

    private fun createDataFilters(): List<DanmakuDataFilter> =
        listOf(
            TypeFilter(),
            colorFilter,
            UserIdFilter(),
            GuestFilter(),
            BlockedTextFilter { it == 0L },
            DuplicateMergedFilter()
        )

    private fun createLayoutFilters(): List<DanmakuLayoutFilter> = emptyList()
}
