package com.hupu.tv.player.app.widget.danmuku

import android.content.Context
import android.util.AttributeSet
import android.util.Log
import android.view.View
import android.view.ViewGroup
import android.widget.Toast
import androidx.core.content.res.ResourcesCompat
import com.google.gson.Gson
import com.google.gson.reflect.TypeToken
import com.shuyu.gsyvideoplayer.utils.Debuger
import com.shuyu.gsyvideoplayer.video.StandardGSYVideoPlayer
import com.shuyu.gsyvideoplayer.video.base.GSYBaseVideoPlayer
import com.shuyu.gsyvideoplayer.video.base.GSYVideoPlayer
import com.shuyu.gsyvideoplayer.video.base.GSYVideoView
import com.hupu.tv.player.app.R
import com.hupu.tv.player.app.utils.GlideUtils
import com.kuaishou.akdanmaku.data.DanmakuItemData
import com.kuaishou.akdanmaku.ecs.DanmakuEngine
import com.kuaishou.akdanmaku.render.SimpleRenderer
import com.kuaishou.akdanmaku.render.TypedDanmakuRenderer
import com.kuaishou.akdanmaku.ui.DanmakuPlayer
import com.kuaishou.akdanmaku.ui.DanmakuView
import kotlinx.android.synthetic.main.danmaku_layout.view.*

/**
 * Created by guoshuyu on 2017/2/16.
 *
 *
 * 配置弹幕使用的播放器，目前使用的是本地模拟数据。
 *
 *
 * 模拟数据的弹幕时常比较短，后面的时长点是没有数据的。
 *
 *
 * 注意：这只是一个例子，演示如何集合弹幕，需要完善如弹出输入弹幕等的，可以自行完善。
 * 注意：b站的弹幕so只有v5 v7 x86、没有64，所以记得配置上ndk过滤。
 */
class DanmakuVideoPlayer : StandardGSYVideoPlayer {
    var danmakuPlayer: DanmakuPlayer? = null
    var danmakuPlayController: DanmakuPlayController? = null
    var danmaKuShow = true
    var ksDanmuku: DanmakuView? = null
    private var onBusinessListener: OnBusinessListener? = null


    constructor(context: Context?, fullFlag: Boolean?) : super(context, fullFlag) {}
    constructor(context: Context?) : super(context) {}
    constructor(context: Context?, attrs: AttributeSet?) : super(context, attrs) {}

    override fun getLayoutId(): Int {
        return R.layout.danmaku_layout
    }

    override fun init(context: Context) {
        super.init(context)

        //初始化弹幕显示

        initKsDamaku(context)
        initListener()


    }

    private fun initKsDamaku(context: Context) {
        val ksDanmuku = findViewById<DanmakuView>(R.id.ks_danmaku_view)
        danmakuPlayer = DanmakuPlayer(
            renderer = TypedDanmakuRenderer(
                SimpleRenderer(),
                DanmakuItemData.DANMAKU_STYLE_ICON_UP to UpLogoRenderer(
                    ResourcesCompat.getDrawable(resources, R.mipmap.icon_danmaku_input_text_up_icon, context.theme)!!
                )
            )
        ).apply {
            bindView(ksDanmuku)
        }

        danmakuPlayController = DanmakuPlayController(context, danmakuPlayer!!, ksDanmuku)
//        updateDanmakuData(context)
        danmakuPlayController?.start()
    }

    private fun updateDanmakuData(context: Context) {
        Thread {
            Log.d(DanmakuEngine.TAG, "[Sample] 开始加载数据")
            val jsonString = context.assets.open("test_danmaku_data.json").bufferedReader().use { it.readText() }
            val type = object : TypeToken<List<DanmakuItemData>>() {}.type
            Log.d(DanmakuEngine.TAG, "[Sample] 开始解析数据")
            val dataList = Gson().fromJson<List<DanmakuItemData>>(jsonString, type)
            danmakuPlayer?.updateData(dataList)
            Log.d(DanmakuEngine.TAG, "[Sample] 数据已加载(count = ${dataList.size})")
            ksDanmuku?.post {
                Toast.makeText(context, "数据已加载", Toast.LENGTH_SHORT).show()
            }
        }.start()
        ksDanmuku?.post {
            Toast.makeText(context, "开始加载数据", Toast.LENGTH_SHORT).show()
        }
    }

    private fun initListener() {
        toogle_danmaku.setOnClickListener {
            danmaKuShow = !danmaKuShow
            resolveDanmakuShow()
        }
        iv_reload.setOnClickListener {
            onBusinessListener?.onReload()
        }
        iv_start.setOnClickListener {
            if (onBusinessListener != null) {
                onBusinessListener?.onBusinessStart()
            }
            clickStartIcon()
        }
        iv_live_more.setOnClickListener {
            onBusinessListener?.onLiveMore()
        }
        tv_copy_link.setOnClickListener {
            onBusinessListener?.onCopy()
        }
        back.setOnClickListener {
            onBusinessListener?.back()
        }
    }

    fun setCover(cover: String) {
        GlideUtils.loadImage(activityContext, cover, iv_cover, R.mipmap.bg_live_place_holder)
    }

    override fun touchSurfaceMoveFullLogic(absDeltaX: Float, absDeltaY: Float) {
        super.touchSurfaceMoveFullLogic(absDeltaX, absDeltaY)
        //不给触摸快进，如果需要，屏蔽下方代码即可
        mChangePosition = false

        //不给触摸音量，如果需要，屏蔽下方代码即可
        mChangeVolume = false

        //不给触摸亮度，如果需要，屏蔽下方代码即可
        mBrightness = false
    }

    fun showCopy(showCopy: Boolean) {
        if (showCopy) {
            tv_copy_link.visibility = View.VISIBLE
        } else {
            tv_copy_link.visibility = View.GONE
        }
    }

    fun setOnBusinessListener(onBusinessListener: OnBusinessListener?) {
        this.onBusinessListener = onBusinessListener
    }

    override fun changeUiToPreparingShow() {
        super.changeUiToPreparingShow()
        rl_progress.visibility = View.VISIBLE
    }

    override fun changeUiToPlayingShow() {
        super.changeUiToPlayingShow()
        rl_progress.visibility = View.GONE
    }

    override fun changeUiToCompleteClear() {
        super.changeUiToCompleteClear()
        rl_progress.visibility = View.GONE
    }

    override fun updateStartImage() {
        super.updateStartImage()
        mStartButton.visibility = View.GONE
        when (mCurrentState) {
            GSYVideoView.CURRENT_STATE_PLAYING -> {
                iv_start.setImageResource(R.mipmap.icon_live_pause)
            }
            GSYVideoView.CURRENT_STATE_ERROR -> {
                iv_start.setImageResource(R.mipmap.icon_live_play)
            }
            else -> {
                iv_start.setImageResource(R.mipmap.icon_live_play)
            }
        }
    }

    override fun onPrepared() {
        super.onPrepared()

        iv_cover.visibility = View.GONE
    }

    override fun onVideoPause() {
        super.onVideoPause()
        danmakuOnPause()
    }

    override fun onVideoResume(isResume: Boolean) {
        super.onVideoResume(isResume)
        danmakuOnResume()
    }

    override fun clickStartIcon() {
        super.clickStartIcon()
        if (mCurrentState == GSYVideoView.CURRENT_STATE_PLAYING) {
            danmakuOnResume()
        } else if (mCurrentState == GSYVideoView.CURRENT_STATE_PAUSE) {
            danmakuOnPause()
        }
    }

    override fun onCompletion() {
        releaseDanmaku(this)
    }

    fun setFireText(fire: String?) {
        tv_fire.text = fire
    }

    override fun cloneParams(from: GSYBaseVideoPlayer, to: GSYBaseVideoPlayer) {
        super.cloneParams(from, to)
    }

    /**
     * 处理播放器在全屏切换时，弹幕显示的逻辑
     * 需要格外注意的是，因为全屏和小屏，是切换了播放器，所以需要同步之间的弹幕状态
     */
    override fun startWindowFullscreen(context: Context, actionBar: Boolean, statusBar: Boolean): GSYBaseVideoPlayer {
        val gsyBaseVideoPlayer = super.startWindowFullscreen(context, actionBar, statusBar)
        if (gsyBaseVideoPlayer != null) {
            val gsyVideoPlayer = gsyBaseVideoPlayer as DanmakuVideoPlayer
            gsyVideoPlayer.danmaKuShow = danmaKuShow
            gsyVideoPlayer.onBusinessListener = onBusinessListener
//            onPrepareDanmaku(gsyVideoPlayer)
        }
        return gsyBaseVideoPlayer
    }

    /**
     * 处理播放器在退出全屏时，弹幕显示的逻辑
     * 需要格外注意的是，因为全屏和小屏，是切换了播放器，所以需要同步之间的弹幕状态
     */
    override fun resolveNormalVideoShow(oldF: View, vp: ViewGroup, gsyVideoPlayer: GSYVideoPlayer) {
        super.resolveNormalVideoShow(oldF, vp, gsyVideoPlayer)
        val gsyDanmaVideoPlayer = gsyVideoPlayer as DanmakuVideoPlayer
        danmaKuShow = gsyDanmaVideoPlayer.danmaKuShow
        if (gsyDanmaVideoPlayer.ksDanmuku != null) {
//                resolveDanmakuShow()
            releaseDanmaku(gsyDanmaVideoPlayer)
        }
    }

    override fun release() {
        super.release()
        danmakuPlayer?.release()
    }

    protected fun danmakuOnPause() {
        danmakuPlayController?.pause()
    }

    protected fun danmakuOnResume() {
        danmakuPlayController?.start()
    }

    /**
     * 弹幕的显示与关闭
     */
    private fun resolveDanmakuShow() {
        post {
            toogle_danmaku?.setImageResource(if (danmaKuShow) R.mipmap.icon_live_danmu_on else R.mipmap.icon_live_danmu_off)
            danmakuPlayController?.setVisibility(danmaKuShow)
        }
    }

    /**
     * 开始播放弹幕
     */
    private fun onPrepareDanmaku(gsyVideoPlayer: DanmakuVideoPlayer) {
        if (gsyVideoPlayer.ksDanmuku != null) {
            gsyVideoPlayer.danmakuPlayController?.start()
        }
    }


    /**
     * 释放弹幕控件
     */
    private fun releaseDanmaku(danmakuVideoPlayer: DanmakuVideoPlayer?) {
        if (danmakuVideoPlayer?.ksDanmuku != null) {
            Debuger.printfError("release Danmaku!")
            danmakuPlayer?.release()
        }
    }


    /**
     * 模拟添加弹幕数据
     */
    fun addDanmaku(text: String?, isMine: Boolean) {
        text?.let {
            this.danmakuPlayController?.sendDanmaku(text, isMine)
        }
    }

    interface OnBusinessListener {
        fun onLiveMore()
        fun onReload()
        fun onBusinessStart()
        fun onCopy()
        fun back()
    }
}