package com.softgarden.baselibrary.base

import android.app.Activity
import android.app.AlertDialog
import android.content.Context
import android.content.Intent
import android.content.res.Configuration
import android.net.Uri
import android.os.Bundle
import android.os.Handler
import android.os.Message
import android.provider.Settings

import android.view.View
import android.view.WindowManager
import android.widget.FrameLayout
import androidx.annotation.LayoutRes
import androidx.core.content.ContextCompat
import com.google.gson.JsonParseException
import com.mirkowu.statusbarutil.StatusBarUtil
import com.softgarden.baselibrary.BuildConfig
import com.softgarden.baselibrary.R
import com.softgarden.baselibrary.network.ApiException
import com.softgarden.baselibrary.utils.DisplayUtil
import com.softgarden.baselibrary.utils.InstanceUtil
import com.softgarden.baselibrary.utils.L
import com.softgarden.baselibrary.utils.ToastUtil
import com.trello.rxlifecycle4.components.support.RxAppCompatActivity

import java.lang.ref.WeakReference
import java.lang.reflect.ParameterizedType
import java.net.ConnectException
import java.net.SocketTimeoutException
import java.net.UnknownHostException
import java.util.*

/**
 * Activity 基类  已实现以下功能
 *
 *
 * 1.切换语言
 * 2.切换日夜模式
 * 3.检测横竖屏
 * 4.显示/隐藏Loading弹框
 * 5.ButterKnife 绑定数据
 * 6.控制RxJava生命周期，防止内存泄漏
 * 7.MVP模式 参考 https://github.com/north2016/T-MVP
 * 需要时 可重写createPresenter() [BaseActivity.createPresenter]  并且使用泛型 <P extends BasePresenter> 为当前Presenter实例
</P> */
abstract class BaseActivity<P : IBasePresenter?> : RxAppCompatActivity(), IBaseDisplay {
    val TAG = javaClass.simpleName
    protected var mBaseDelegate: BaseDelegate? = null
    val baseDelegate: BaseDelegate
        get() {
            if (mBaseDelegate == null) {
                mBaseDelegate = BaseDelegate(this)
            }
            return mBaseDelegate!!
        }

    class BaseHandler() : Handler() {
        private var mActivity: WeakReference<BaseActivity<*>>? = null

        constructor (activity: BaseActivity<*>) : this() {
            mActivity = WeakReference(activity)
        }


        override fun handleMessage(msg: Message?) {
            super.handleMessage(msg)
            val activity: BaseActivity<*>? = mActivity?.get()
            if (activity != null) {

            }
        }

    }


    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        onBaseCreate(savedInstanceState)
        //键盘不顶起布局，以后若有需要顶起评论框的，需要改动父类
//        window.setSoftInputMode(WindowManager.LayoutParams.SOFT_INPUT_ADJUST_PAN)
        val delegate = baseDelegate
        delegate.onCreate(savedInstanceState)
        bindView()
        initPresenter()
//        setTransBar()
        setStatusTextBlank()
        initialize()

        //显示当前的Activity路径
        L.e("当前打开的Activity:  " + javaClass.name)
    }

    open fun setStatusTextWhite() {
//        StatusBarUtil.setTransparent(this);
        StatusBarUtil.setStatusBarColor(activity, ContextCompat.getColor(this, R.color.white))
        StatusBarUtil.setStatusBarLightMode(activity)
    }


    protected fun setStatus() {
//        window.addFlags(WindowManager.LayoutParams.FLAG_LAYOUT_IN_SCREEN)
//        window.addFlags(WindowManager.LayoutParams.FLAG_LAYOUT_NO_LIMITS)
//        val attrs = window.attributes
//        attrs.flags = attrs.flags and WindowManager.LayoutParams.FLAG_FULLSCREEN.inv()
//        window.attributes = attrs
//        val decorView = window.decorView
//        decorView.systemUiVisibility = (View.SYSTEM_UI_FLAG_HIDE_NAVIGATION
//                or View.SYSTEM_UI_FLAG_FULLSCREEN
//                or View.SYSTEM_UI_FLAG_IMMERSIVE_STICKY)
//        window.addFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON)
    }

    protected fun setMainFull() {
        window.addFlags(WindowManager.LayoutParams.FLAG_LAYOUT_IN_SCREEN)
        window.addFlags(WindowManager.LayoutParams.FLAG_LAYOUT_NO_LIMITS)
        val attrs = window.attributes
        attrs.flags = attrs.flags and WindowManager.LayoutParams.FLAG_FULLSCREEN.inv()
        window.attributes = attrs
        val decorView = window.decorView
        decorView.systemUiVisibility = (View.SYSTEM_UI_FLAG_HIDE_NAVIGATION
                or View.SYSTEM_UI_FLAG_FULLSCREEN
                or View.SYSTEM_UI_FLAG_IMMERSIVE_STICKY)
        window.addFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON)
    }

    fun statusBarTranForView(view: View, isWhite: Boolean = true) {
        StatusBarUtil.setStatusBarColor(this, getColor(R.color.transparent))
        if (isWhite) {
            StatusBarUtil.setStatusBarLightMode(this, 3)
        } else {
            StatusBarUtil.setStatusBarDarkMode(this, 3)
        }

        val top =
            -DisplayUtil.px2dip(this, StatusBarUtil.getStatusBarHeight().toFloat()).toFloat() - 1f
        val layoutParams = view.layoutParams as FrameLayout.LayoutParams
        layoutParams.setMargins(0, DisplayUtil.dip2px(this, top), 0, 0)
        view.layoutParams = layoutParams
    }

    protected fun setStatusColor(color: Int) {
        val decorView = window.decorView
        decorView.systemUiVisibility =
            View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN or View.SYSTEM_UI_FLAG_LAYOUT_STABLE
        window.statusBarColor = resources.getColor(color)
    }

    protected fun setFUllWindow() {
        val decorView = window.decorView
        decorView.systemUiVisibility = (View.SYSTEM_UI_FLAG_HIDE_NAVIGATION
                or View.SYSTEM_UI_FLAG_FULLSCREEN
                or View.SYSTEM_UI_FLAG_IMMERSIVE_STICKY)
        window.addFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON)
    }

    protected fun clearFullWindow() {
        val decorView = window.decorView
        decorView.systemUiVisibility = View.SYSTEM_UI_FLAG_VISIBLE
    }


    fun startActivity(cls: Class<out Activity?>?) {
        this.startActivity(Intent(activity, cls))
    }

    /**
     *
     * @param savedInstanceState
     */
    private fun onBaseCreate(savedInstanceState: Bundle?) {}

    //    /**
    //     * 取消适配
    //     */
    //    public void cancelAdapterScreen() {
    //        ScreenUtil.cancelAdaptScreen(this);
    //    }
    protected open fun bindView() {
        setContentView(layoutId)
    }

    override fun onResume() {
        baseDelegate.onResume()
        super.onResume()
    }

    fun setStatusTextBlank() {
        StatusBarUtil.setStatusBarColor(this, getColor(R.color.white))
        StatusBarUtil.setStatusBarLightMode(this, 3)
    }

    fun setStatusTextBlankF5() {
        StatusBarUtil.setStatusBarColor(this, getColor(R.color.color_F5F6F7))
        StatusBarUtil.setStatusBarLightMode(this, 3)
    }


    override fun attachBaseContext(newBase: Context) {
        super.attachBaseContext(baseDelegate.attachBaseContext(newBase))
    }

    /**
     * 这个可以视情况 重写 (当横竖屏等配置发生改变时)
     *
     * @param newConfig
     */
    override fun onConfigurationChanged(newConfig: Configuration) {
        super.onConfigurationChanged(newConfig)
        baseDelegate.onConfigurationChanged(newConfig)
    }

    /**
     * 设置横屏竖屏
     *
     * @param mOrientationPortrait true 竖屏 false 横屏
     */
    var isOrientationPortrait: Boolean
        get() = baseDelegate.mOrientationPortrait
        set(mOrientationPortrait) {
            baseDelegate.setOrientationPortrait(mOrientationPortrait)
        }

    /**
     * 切换语言 (设置完后要重启Activity才生效 [.reload])
     *
     * @param language
     */
    fun changeLanguage(language: Locale) {
        baseDelegate.changeLanguage(language, true)
    }

    /**
     * 是否相同 二种语言 （语言和 国家都相同才算是相同）
     *
     * @param mLanguage
     * @param locale
     * @return
     */
    fun isEqualsLanguage(mLanguage: Locale?, locale: Locale): Boolean {
        return baseDelegate.isEqualsLanguage(mLanguage, locale)
    }
    /**
     * 切换日夜模式
     *
     *
     * 需要注意的两个地方，
     * 一是app或者activity引用的style需要是Theme.AppCompat.DayNight或者它的子style，
     * 二是调用getDelegate().setLocalNightMode()你的Activity必须是继承AppCompatActivity的。
     *
     * @param isNightMode
     */
    //    @Override
    //    public void changeDayNightMode(boolean isNightMode) {
    ////        getBaseDelegate().changeDayNightMode(isNightMode);
    //    }
    /**
     * 重启Activity
     * 此方法会比 recreate() 效果更好
     */
    fun reload() {
        baseDelegate.reload()
    }

    /**
     * 权限提示对话框
     */
    fun showPermissionDialog() {
        AlertDialog.Builder(this)
            .setTitle(R.string.base_prompt_message)
            .setMessage(R.string.base_permission_lack)
            .setNegativeButton(R.string.base_cancel) { dialog, which -> dialog.dismiss() }
            .setPositiveButton(R.string.base_ok) { dialog, which -> startAppSettings() }.show()
    }

    /**
     * 启动当前应用设置页面
     */
    fun startAppSettings() {
        val intent = Intent(Settings.ACTION_APPLICATION_DETAILS_SETTINGS)
        intent.data = Uri.parse("package:$packageName")
        startActivity(intent)
    }

    /**
     * 设置为亮色模式 状态栏 颜色变黑
     */
    fun setStatusBarLightMode() {
//        if (!BaseSPManager.isNightMode()) {
//            StatusBarUtil.setStatusBarLightModeWithNoSupport(this, true);
//        }
    }

    /**
     * 回复状态栏颜色状态
     */
    fun setStatusBarDarkMode() {
//        if (!BaseSPManager.isNightMode()) {
//            if (StatusBarUtil.setStatusBarDarkMode(this) == 0) {//不支持 亮色 模式
////                //恢复过来时的 处理
////                StatusBarUtil.setTransparent(this);
//            }
//        }
    }

    val activity: BaseActivity<*>
        get() = this

    override val baseFragment: BaseFragment<*>?
        get() = null

    /***********************************  LoadingDialog start    */
    /**
     * 显示加载框
     */
    override fun showProgressDialog() {
        baseDelegate.showProgressDialog()
    }

    /**
     * 显示加载框（带文字）
     */
    override fun showProgressDialog(message: CharSequence?) {
        baseDelegate.showProgressDialog(message)
    }

    /**
     * 隐藏加载框
     */
    override fun hideProgressDialog() {
        baseDelegate.hideProgressDialog()
    }
    /*******************************  LoadingDialog end   */
    /**
     * 全局简单异常处理
     *
     * @param t
     */
    override fun showError(t: Throwable, isConfigStatus: Boolean) {
        when (t) {
            is ConnectException -> {
                ToastUtil.s(getString(R.string.base_connect_failed))

            }
            is UnknownHostException -> {
                ToastUtil.s(getString(R.string.base_request_serve_failed))

            }
            is SocketTimeoutException -> {
                ToastUtil.s(getString(R.string.base_socket_timeout))

            }
            is JsonParseException -> {
                ToastUtil.s(getString(R.string.base_parse_failed))
                t.printStackTrace()
            }
            is ApiException -> {
                //通用的Api异常处理
                if (isConfigStatus){
                    onApiException(t)
                }else{
                    ToastUtil.s(t.message)
                }

            }
            else -> {
                if (isConfigStatus) {
                    showCustomException(t)
                }else{
                    ToastUtil.s(t.message)
                }
            }
        }
    }


    /**
     * 全局的详细异常处理 根据项目情况重写
     *
     * @param e
     */
    override fun onApiException(e: ApiException) {
        showCustomException(e)
//        baseDelegate.showError(e)
    }

    open fun showErrorLayout() {

    }


    open fun showCustomException(t: Throwable) {
        ToastUtil.s(t.message)
        if (BuildConfig.DEBUG) t.printStackTrace()
        showErrorLayout()
    }

    override fun onActivityResult(requestCode: Int, resultCode: Int, data: Intent?) {
        super.onActivityResult(requestCode, resultCode, data)
        if (resultCode == Activity.RESULT_OK && requestCode == REQUEST_LOGIN) {
            var eventId = 0
            if (data != null) eventId = data.getIntExtra(KEY_LOGIN_EVENT, 0)
            backFromLogin(eventId) //从登录界面返回  登录成功
        }
    }

    /**
     * 登录成功 返回回调
     *
     * @param eventId 一般为点击View的id，可根据id判断接点击事件，从而继续操作流程
     */
    protected fun backFromLogin(eventId: Int) {}

    /**
     * 网络请求结束（无论成功还是失败）
     */
    override fun onRequestFinish() {}

    /*********************** MVP 参考 https://github.com/north2016/T-MVP  */
    override fun onDestroy() {
        super.onDestroy()
        if (presenter != null) {
            presenter?.detachView()
            presenter = null
        }

    }

    var presenter: P? = null
        private set

    protected fun initPresenter() {
        presenter = createPresenter()
        if (presenter != null) presenter?.attachView(this)
    }

    /**
     * 创建Presenter 此处已重写 需要时重写即可
     *
     * @return
     */
    fun createPresenter(): P? {
        if (this is IBaseDisplay
            && this.javaClass.genericSuperclass is ParameterizedType
            && (this.javaClass.genericSuperclass as ParameterizedType).actualTypeArguments.size > 0
        ) {
            val mPresenterClass = (this.javaClass.genericSuperclass as ParameterizedType)
                .actualTypeArguments[0] as Class<*> //获取Presenter的class
            return InstanceUtil.getInstance(mPresenterClass)
        }
        return null
    }

    /*********************** MVP 参考 https://github.com/north2016/T-MVP  */
    @get:LayoutRes
    protected abstract val layoutId: Int
    protected abstract fun initialize()

    companion object {
        /*** 通用的 用于传递数据的Key   */
        const val KEY_DATA = "data"
        const val KEY_TITLE = "title"
        const val KEY_TYPE = "type"
        const val KEY_LOGIN_EVENT = "login_event"
        const val REQUEST_LOGIN = 0x00001234
        const val REQUEST_CODE = 0x00005678
    }
}