package com.softgarden.baselibrary.utils

import android.app.Activity
import android.app.Application
import android.os.Process
import java.lang.ref.WeakReference
import java.util.*

/**
 * @author by DELL
 * @date on 2020/06/11
 * @describe Activity管理类
 *
 *
 * 添加/删除 建议在[Application.registerActivityLifecycleCallbacks]中统一处理
 * (此方法比在BaseActivity中处理要好)
 */
class ActivityManager private constructor() {
    private object Singleton {
        val INSTANCE = ActivityManager()
    }

    /***  添加 建议在Application中统一处理  */
    fun add(activity: Activity?) {
        mActStack.add(WeakReference(activity))
    }

    /***  移除 建议在Application中统一处理  */
    fun remove(activity: Activity) {
        for (temp in mActStack) {
            if (isEqualsActivity(temp, activity)) {
                mActStack.remove(temp)
                break
            }
        }
    }

    /**
     * 获取当前Activity数量
     *
     * @return
     */
    val count: Int
        get() = mActStack.size

    /**
     * 栈内是否包含此activity
     *
     * @param cls
     * @return
     */
    fun isContains(cls: Class<*>): Boolean {
        for (temp in mActStack) {
            if (isEqualsActivity(temp, cls)) {
                return true
            }
        }
        return false
    }

    /**
     * 查找指定Activity 默认第一个
     *
     * @param cls
     * @return 未找到则返回 null
     */
//    fun <T : Activity?> find(cls: Class<Any?>): T? {
//        return findFirst(cls)
//    }

    /**
     * 查找指定Activity 第一个
     *
     * @param cls
     * @return
     */
    fun <T : Activity?> findFirst(cls:  Class<Any?>): T? {
        for (temp in mActStack) {
            if (isEqualsActivity(temp, cls)) {
                return temp?.get() as T
            }
        }
        return null
    }

    /**
     * 查找指定Activity 最后一个
     *
     * @param cls
     * @return
     */
    fun <T : Activity?> findLast(cls: Class<*>): T? {
        for (i in mActStack.indices.reversed()) {
            val temp = mActStack[i]
            if (isEqualsActivity(temp, cls)) {
                return temp!!.get() as T?
            }
        }
        return null
    }

    /**
     * 获取当前（即最后一个） Activity
     *
     * @return
     */
    fun <T : Activity?> getCurrent(): T? {
        return if (mActStack.lastElement() != null) {
            mActStack.lastElement()!!.get() as T?
        } else null
    }

    private fun isEqualsActivity(temp: WeakReference<Activity?>?, cls: Class<*>): Boolean {
        return temp != null && temp.get() != null && temp.get()!!.javaClass == cls
    }

    private fun isEqualsActivity(temp: WeakReference<Activity?>?, activity: Activity): Boolean {
        return temp != null && temp.get() != null && temp.get() === activity
    }

    /**
     * 结束指定Activity
     *
     * @param activity
     */
    fun finish(activity: Activity?) {
        if (activity != null && !activity.isFinishing) {
            activity.finish()
        }
        // remove(activity);//这个可以不用的 记得在Application中调用
    }

    /**
     * 结束指定Activity
     * 注：当栈中可能包含多个该Activity时，该方法会将所有的该Activity都finish
     *
     * @param cls
     */
    fun finish(cls: Class<*>) {
        for (temp in mActStack) {
            if (isEqualsActivity(temp, cls)) {
                finish(temp!!.get())
            }
        }
    }

    /**
     * 结束此Activity之前的所有Activity（不包括当前的 ，结束当前需手动） 最终显示此Activity
     * 1-2-3  * 4-5
     * 1-2  * 4-5
     * 1  * 4-5
     * * 4-5
     *
     * @param cls
     */
    fun finishBefore(cls: Class<*>) {
        var isFound = false
        for (i in mActStack.indices.reversed()) {
            val temp = mActStack[i]
            if (isFound) {
                if (temp != null && temp.get() != null) {
                    finish(temp.get())
                }
            } else if (isEqualsActivity(temp, cls)) {
                isFound = true
            }
        }
    }

    /**
     * 结束此Activity之后的所有Activity（不包括当前的 ，结束当前需手动） 最终显示此Activity
     * 1-2-3 * 4-5
     * 1-2-3 * 5
     * 1-2-3 *
     *
     * @param cls
     */
    fun finishAfter(cls: Class<*>) {
        var isFound = false
        for (temp in mActStack) {
            if (isFound) {
                if (temp != null && temp.get() != null) {
                    finish(temp.get())
                }
            } else if (isEqualsActivity(temp, cls)) {
                isFound = true
            }
        }
    }

    fun finishAll() {
        for (temp in mActStack) {
            if (temp != null && temp.get() != null) {
                finish(temp.get())
            }
        }
    }

    fun exitApp() {
        finishAll()
        System.exit(0)
        Process.killProcess(Process.myPid())
    }

    companion object {
        private val mActStack = Stack<WeakReference<Activity?>?>()
        val instance: ActivityManager
            get() = Singleton.INSTANCE
    }
}