package com.softgarden.baselibrary.utils

import android.content.Context
import android.content.Intent
import android.content.pm.PackageInfo
import android.graphics.Rect
import android.net.Uri
import android.os.Build
import android.provider.Settings
import android.view.View
import android.view.inputmethod.InputMethodManager

/**
 * @author by DELL
 * @date on 2020/06/11
 * @describe
 */
class AppUtil private constructor() {
    companion object {
        /**
         * 获取版本号
         *
         * @return 当前应用的版本号
         */
        fun getVersionName(context: Context?): String? {
            val info = getPackageInfo(context)
            return info?.versionName
        }

        /**
         * 获取当前应用信息类
         *
         * @param context
         * @return PackageInfo
         */
        fun getPackageInfo(context: Context?): PackageInfo? {
            return try {
                val manager = context!!.packageManager
                manager.getPackageInfo(context.packageName, 0)
            } catch (e: Exception) {
                e.printStackTrace()
                null
            }
        }

        /**
         * 根据包名获取意图
         *
         * @param context     上下文
         * @param packageName 包名
         * @return 意图
         */
        private fun getIntentByPackageName(context: Context, packageName: String): Intent? {
            return context.packageManager.getLaunchIntentForPackage(packageName)
        }

        /**
         * 根据包名判断App是否安装
         *
         * @param context     上下文
         * @param packageName 包名
         * @return `true`: 已安装<br></br>`false`: 未安装
         */
        fun isInstallApp(context: Context, packageName: String): Boolean {
            return getIntentByPackageName(context, packageName) != null
        }

        /**
         * 打开指定包名的App
         *
         * @param context     上下文
         * @param packageName 包名
         * @return `true`: 打开成功<br></br>`false`: 打开失败
         */
        fun openAppByPackageName(context: Context, packageName: String): Boolean {
            val intent = getIntentByPackageName(context, packageName)
            if (intent != null) {
                context.startActivity(intent)
                return true
            }
            return false
        }

        /**
         * 打开指定包名的App应用信息界面
         *
         * @param context 上下文
         */
        fun openAppSetting(context: Context) {
            val intent = Intent(Settings.ACTION_APPLICATION_DETAILS_SETTINGS)
            intent.data = Uri.parse("package:" + context.packageName)
            context.startActivity(intent)
        }

        /**
         * 打开定位信息、GPS
         *
         * @param context
         */
        fun openLocationSetting(context: Context) {
            val intent = Intent()
            intent.action = Settings.ACTION_LOCATION_SOURCE_SETTINGS
            context.startActivity(intent)
        }

        /**
         * 前往网络设置界面
         *
         * @param context
         */
        fun openNetworkSetting(context: Context?) {
            // 跳转到系统的网络设置界面
            var intent: Intent? = null
            // 先判断当前系统版本
            if (Build.VERSION.SDK_INT > 10) {  // 3.0以上
                intent = Intent(Settings.ACTION_WIRELESS_SETTINGS)
            } else {
                intent = Intent()
                intent.setClassName("com.android.settings", "com.android.settings.WirelessSettings")
            }
            context!!.startActivity(intent)
        }

        /**
         * 软键盘是否显示
         *
         * @param rootView
         * @return
         */
        fun isKeyboardShown(rootView: View): Boolean {
            val softKeyboardHeight = 100
            val r = Rect()
            rootView.getWindowVisibleDisplayFrame(r)
            val dm = rootView.resources.displayMetrics
            val heightDiff = rootView.bottom - r.bottom
            return heightDiff > softKeyboardHeight * dm.density
        }

        /**
         * 测量软键盘的底部距离屏幕底部大小
         *
         * @param rootView
         * @return
         */
        fun getKeyboardToBottomhighet(rootView: View): Int {
            val softKeyboardHeight = 100
            val r = Rect()
            rootView.getWindowVisibleDisplayFrame(r)
            val dm = rootView.resources.displayMetrics
            return rootView.bottom - r.bottom
        }

        /**
         * 显示和隐藏软键盘 View ： EditText、TextView isShow : true = show , false = hide
         *
         * @param context
         * @param view
         * @param isShow
         */
        fun popSoftKeyboard(context: Context, view: View,
                            isShow: Boolean) {
            val imm = context
                    .getSystemService(Context.INPUT_METHOD_SERVICE) as InputMethodManager
            if (isShow) {
                view.requestFocus()
                imm.showSoftInput(view, InputMethodManager.SHOW_IMPLICIT)
            } else {
                imm.hideSoftInputFromWindow(view.windowToken, 0)
            }
        }

        /**
         * 显示软键盘
         *
         * @param view
         */
        fun showSoftKeyboard(view: View) {
            val context = view.context
            val imm = context
                    .getSystemService(Context.INPUT_METHOD_SERVICE) as InputMethodManager
            view.requestFocus()
            imm.showSoftInput(view, InputMethodManager.SHOW_IMPLICIT)
        }

        /**
         * 隐藏软键盘
         *
         * @param view
         */
        fun hideSoftKeyboard(view: View) {
            val context = view.context
            val imm = context
                    .getSystemService(Context.INPUT_METHOD_SERVICE) as InputMethodManager
            imm.hideSoftInputFromWindow(view.windowToken, 0)
        }
    }

    init {
        throw UnsupportedOperationException("u can't fuck me...")
    }
}