package com.softgarden.baselibrary.utils

import android.content.Context
import android.graphics.Bitmap
import android.graphics.BitmapFactory
import android.graphics.Matrix
import android.media.ExifInterface
import com.trello.rxlifecycle4.internal.Preconditions
import io.reactivex.rxjava3.core.Observable

import java.io.ByteArrayOutputStream
import java.io.File
import java.io.FileOutputStream
import java.io.IOException

/**
 * @author by DELL
 * @date on 2020/06/11
 * @describe 图片压缩工具 抽自"LuBan压缩" 比较接近微信压缩方案
 */
class CompressUtil internal constructor() {
    private val compressUtil: CompressUtil? = null
    private var mByteArrayOutputStream: ByteArrayOutputStream? = null
    fun singleAction(context: Context, file: File): Observable<File> {
        return Observable.fromCallable { compressImage(context, file) } /*.subscribeOn(Schedulers.computation()).observeOn(AndroidSchedulers.mainThread())*/
    }

    private fun getCacheFilePath(context: Context, path: String): String {
        //命名规则
        val name = StringBuilder("mobile_android_" + System.currentTimeMillis()
                + path.hashCode() + Math.random() * 1000 + ".jpg")
        return context.cacheDir.absolutePath + File.separator + name
    }

    @Throws(IOException::class)
    fun compressImage(context: Context, file: File): File? {
        val thumb = getCacheFilePath(context, file.path)
        var size: Double
        val filePath = file.absolutePath

        // int angle = getImageSpinAngle(filePath);
        val angle = 0
        var width = getImageSize(filePath)[0]
        var height = getImageSize(filePath)[1]
        val flip = width > height
        var thumbW = if (width % 2 == 1) width + 1 else width
        var thumbH = if (height % 2 == 1) height + 1 else height
        width = if (thumbW > thumbH) thumbH else thumbW
        height = if (thumbW > thumbH) thumbW else thumbH
        val scale = width.toDouble() / height
        if (scale <= 1 && scale > 0.5625) {
            if (height < 1664) {
                if (file.length() / 1024 < 150) {
                    return file
                }
                size = width * height / Math.pow(1664.0, 2.0) * 150
                size = if (size < 60) 60.00 else size
            } else if (height >= 1664 && height < 4990) {
                thumbW = width / 2
                thumbH = height / 2
                size = thumbW * thumbH / Math.pow(2495.0, 2.0) * 300
                size = if (size < 60) 60.00 else size
            } else if (height >= 4990 && height < 10240) {
                thumbW = width / 4
                thumbH = height / 4
                size = thumbW * thumbH / Math.pow(2560.0, 2.0) * 300
                size = if (size < 100) 100.00 else size
            } else {
                val multiple = if (height / 1280 == 0) 1 else height / 1280
                thumbW = width / multiple
                thumbH = height / multiple
                size = thumbW * thumbH / Math.pow(2560.0, 2.0) * 300
                size = if (size < 100) 100.00 else size
            }
        } else if (scale <= 0.5625 && scale > 0.5) {
            if (height < 1280 && file.length() / 1024 < 200) {
                return file
            }
            val multiple = if (height / 1280 == 0) 1 else height / 1280
            thumbW = width / multiple
            thumbH = height / multiple
            size = thumbW * thumbH / (1440.0 * 2560.0) * 400
            size = if (size < 100) 100.00 else size
        } else {
            val multiple = Math.ceil(height / (1280.0 / scale)).toInt()
            thumbW = width / multiple
            thumbH = height / multiple
            size = thumbW * thumbH / (1280.0 * (1280 / scale)) * 500
            size = if (size < 100) 100.00 else size
        }
        return compress(filePath, thumb, if (flip) thumbH else thumbW, if (flip) thumbW else thumbH, angle,
                size.toLong())
    }

    /**
     * obtain the image rotation angle
     *
     * @param path path of target image
     */
    private fun getImageSpinAngle(path: String): Int {
        var degree = 0
        var exifInterface: ExifInterface? = null
        exifInterface = try {
            ExifInterface(path)
        } catch (e: IOException) {
            // 图片不支持获取角度
            return 0
        }
        val orientation = exifInterface?.getAttributeInt(ExifInterface.TAG_ORIENTATION,
                ExifInterface.ORIENTATION_NORMAL)
        when (orientation) {
            ExifInterface.ORIENTATION_ROTATE_90 -> degree = 90
            ExifInterface.ORIENTATION_ROTATE_180 -> degree = 180
            ExifInterface.ORIENTATION_ROTATE_270 -> degree = 270
        }
        return degree
    }

    /**
     * obtain the thumbnail that specify the size
     *
     * @param imagePath the target image path
     * @param width     the width of thumbnail
     * @param height    the height of thumbnail
     * @return [Bitmap]
     */
    private fun compress(imagePath: String, width: Int, height: Int): Bitmap {
        val options = BitmapFactory.Options()
        options.inJustDecodeBounds = true
        BitmapFactory.decodeFile(imagePath, options)
        val outH = options.outHeight
        val outW = options.outWidth
        var inSampleSize = 1
        while (outH / inSampleSize > height || outW / inSampleSize > width) {
            inSampleSize *= 2
        }
        options.inSampleSize = inSampleSize
        options.inJustDecodeBounds = false
        return BitmapFactory.decodeFile(imagePath, options)
    }

    /**
     * 指定参数压缩图片
     * create the thumbnail with the true rotate angle
     *
     * @param largeImagePath the big image path
     * @param thumbFilePath  the thumbnail path
     * @param width          width of thumbnail
     * @param height         height of thumbnail
     * @param angle          rotation angle of thumbnail
     * @param size           the file size of image
     */
    @Throws(IOException::class)
    private fun compress(largeImagePath: String, thumbFilePath: String, width: Int, height: Int,
                         angle: Int, size: Long): File? {
        var thbBitmap = compress(largeImagePath, width, height)
        thbBitmap = rotatingImage(angle, thbBitmap)
        return saveImage(thumbFilePath, thbBitmap, size)
    }

    /**
     * 旋转图片
     * rotate the image with specified angle
     *
     * @param angle  the angle will be rotating 旋转的角度
     * @param bitmap target image               目标图片
     */
    private fun rotatingImage(angle: Int, bitmap: Bitmap): Bitmap {
        //rotate image
        val matrix = Matrix()
        matrix.postRotate(angle.toFloat())

        //create a new image
        return Bitmap.createBitmap(bitmap, 0, 0, bitmap.width, bitmap.height, matrix,
                true)
    }

    /**
     * 保存图片到指定路径
     * Save image with specified size
     *
     * @param filePath the image file save path 储存路径
     * @param bitmap   the image what be save   目标图片
     * @param size     the file size of image   期望大小
     */
    @Throws(IOException::class)
    private fun saveImage(filePath: String, bitmap: Bitmap, size: Long): File? {
        Preconditions.checkNotNull(bitmap, TAG + "bitmap cannot be null")
        val result = File(filePath.substring(0, filePath.lastIndexOf("/")))
        if (!result.exists() && !result.mkdirs()) {
            return null
        }
        if (mByteArrayOutputStream == null) {
            mByteArrayOutputStream = ByteArrayOutputStream(
                    bitmap.width * bitmap.height)
        } else {
            mByteArrayOutputStream!!.reset()
        }
        var options = 100
        bitmap.compress(Bitmap.CompressFormat.JPEG, options, mByteArrayOutputStream)
        while (mByteArrayOutputStream!!.size() / 1024 > size && options > 6) {
            mByteArrayOutputStream!!.reset()
            options -= 6
            bitmap.compress(Bitmap.CompressFormat.JPEG, options, mByteArrayOutputStream)
        }
        bitmap.recycle()
        val fos = FileOutputStream(filePath)
        mByteArrayOutputStream!!.writeTo(fos)
        fos.close()
        return File(filePath)
    }

    companion object {
        const val TAG = "CompressUtil"
        private fun newInstance(): CompressUtil {
            return CompressUtil()
        }

        /**
         * 返回File Observable
         */
        fun asObservable(context: Context, file: File): Observable<File> {
            return newInstance().singleAction(context, file)
        }

        /**
         * obtain the image's width and height
         *
         * @param imagePath the path of image
         */
        fun getImageSize(imagePath: String?): IntArray {
            val res = IntArray(2)
            val options = BitmapFactory.Options()
            options.inJustDecodeBounds = true
            options.inSampleSize = 1
            BitmapFactory.decodeFile(imagePath, options)
            res[0] = options.outWidth
            res[1] = options.outHeight
            return res
        }
    }
}