package com.softgarden.baselibrary.utils

import android.content.Context
import android.os.Environment
import android.os.Looper
import android.text.TextUtils
import android.widget.ProgressBar
import com.bumptech.glide.Glide
import com.softgarden.baselibrary.BaseApplication
import com.softgarden.baselibrary.R
import java.io.File
import java.math.BigDecimal

/**
 * @author by DELL
 * @date on 2020/06/11
 * @describe Glide缓存工具类
 */
object FileUtil {
    /**
     * 可根据项目需求自行更改
     */
    val APP_PATH: String = BaseApplication.Companion.instance!!.resources.getString(R.string.app_name)
    const val IMAGE_CACHE_PATH = "imageCache"

    /**
     * 是否存在SD卡
     *
     * @return
     */
    fun hasSDCard(): Boolean {
        val status = Environment.getExternalStorageState()
        return TextUtils.equals(status, Environment.MEDIA_MOUNTED)
    }//判断是否有SD卡

    /**
     * 获取应用SD卡缓存的根路径
     *
     * @return
     */
    val externalStorageDir: File?
        get() {
            val RootDir: File
            //判断是否有SD卡
            if (hasSDCard()) {
                RootDir = Environment.getExternalStorageDirectory()
                return RootDir
            } else {
                ToastUtil.Companion.s("没有发现SD卡")
            }
            return null
        }

    //判断是否有SD卡
    val externalStoragePath: String
        get() {
            var path = ""
            //判断是否有SD卡
            if (hasSDCard()) {
                path = Environment.getExternalStorageDirectory().absolutePath
                return path
            }
            return path
        }//判断是否有SD卡

    /**
     * 获取应用SD卡缓存的根路径
     *
     * @return
     */
    val appRootDir: File?
        get() {
            val RootDir: File
            //判断是否有SD卡
            if (hasSDCard()) {
                RootDir = File(Environment.getExternalStorageDirectory(), APP_PATH)
                if (!RootDir.exists()) RootDir.mkdirs()
                return RootDir
            } else {
                ToastUtil.Companion.s("没有发现SD卡")
            }
            return null
        }

    /**
     * Glide 缓存路径
     *
     * @return
     */
    val glideCacheDir: File
        get() {
            val file = File(diskCacheDir, IMAGE_CACHE_PATH)
            if (!file.exists()) {
                file.mkdirs()
            }
            return file
        }

    /**
     * 缓存路径
     * 卸载应用时会一并删除
     * /data/data/包名/cache
     * /mnt/sdcard/Android/data/包名/cache
     *
     * @return
     */
    val diskCacheDir: File?
        get() {
            var dir: File? = null
            dir = if (hasSDCard()) {
                BaseApplication.Companion.instance!!.externalCacheDir
            } else {
                BaseApplication.Companion.instance!!.cacheDir
            }
            return dir
        }

    /**
     * Files路径
     * 卸载应用时会一并删除
     * /data/data/包名/files
     * /mnt/sdcard/Android/data/包名/files
     *
     * @return
     */
    val diskFilesDir: File?
        get() {
            var dir: File? = null
            dir = if (hasSDCard()) {
                BaseApplication.Companion.instance!!.getExternalFilesDir(null)
            } else {
                BaseApplication.Companion.instance!!.filesDir
            }
            return dir
        }

    fun getAppDir(dirName: String?): File? {
        if (appRootDir != null) {
            val file = File(appRootDir, dirName)
            if (!file.exists()) file.mkdirs()
            return file
        }
        return null
    }//判断是否有SD卡

    /**
     * 获取应用SD卡缓存的根路径
     *
     * @return
     */
    val rootDir: File?
        get() {
            val RootDir: File
            val status = Environment.getExternalStorageState()
            //判断是否有SD卡
            if (status == Environment.MEDIA_MOUNTED) {
                RootDir = File(Environment.getExternalStorageDirectory(), APP_PATH)
                if (!RootDir.exists()) RootDir.mkdirs()
                return RootDir
            } else {
                ToastUtil.Companion.s("没有发现SD卡")
            }
            return null
        }
    /**
     * 保存bitmap为图片
     *
     * @param bitmap   要保存的bitmap
     * @param dirFile  要保存的文件夹
     * @param saveName 要保存的文件名称
     */
    //    public static void saveBitmap2File(Context context, Bitmap bitmap, File dirFile, String saveName) {
    //        ImageUtil.saveBitmap2File(context,bitmap, dirFile, saveName);
    //        // 通知图库更新
    //      //  BaseApplication.getInstance().sendBroadcast(new Intent(Intent.ACTION_MEDIA_SCANNER_SCAN_FILE, Uri.fromFile(new File(dirFile, saveName))));
    //    }
    /**
     * 清除图片磁盘缓存
     */
    fun clearImageDiskCache() {
        try {
            if (Looper.myLooper() == Looper.getMainLooper()) {
                Thread(Runnable { Glide.get(BaseApplication.Companion.instance!!).clearDiskCache() }).start()
            } else {
                Glide.get(BaseApplication.Companion.instance!!).clearDiskCache()
            }
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }

    /**
     * 清除图片内存缓存
     */
    fun clearImageMemoryCache() {
        try {
            if (Looper.myLooper() == Looper.getMainLooper()) { //只能在主线程执行
                Glide.get(BaseApplication.Companion.instance!!.applicationContext).clearMemory()
            }
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }

    /**
     * 清除图片所有缓存
     */
    fun clearImageAllCache() {
        clearImageDiskCache()
        clearImageMemoryCache()
        deleteFolderFile(glideCacheDir.path, true)
    }

    /**
     * 获取Glide造成的缓存大小
     *
     * @return CacheSize
     */
    val cacheSize: String
        get() {
            try {
                return getFormatSize(getFolderSize(glideCacheDir).toDouble())
            } catch (e: Exception) {
                e.printStackTrace()
            }
            return ""
        }

    /**
     * 获取指定文件夹内所有文件大小的和
     *
     * @param file file
     * @return size
     * @throws Exception
     */
    @Throws(Exception::class)
    fun getFolderSize(file: File): Long {
        var size: Long = 0
        try {
            val fileList = file.listFiles()
            for (aFileList in fileList) {
                size = if (aFileList.isDirectory) {
                    size + getFolderSize(aFileList)
                } else {
                    size + aFileList.length()
                }
            }
        } catch (e: Exception) {
            e.printStackTrace()
        }
        return size
    }

    /**
     * 删除指定目录下的文件，这里用于缓存的删除
     *
     * @param filePath       filePath
     * @param deleteThisPath deleteThisPath
     */
    fun deleteFolderFile(filePath: String?, deleteThisPath: Boolean) {
        if (!TextUtils.isEmpty(filePath)) {
            try {
                val file = File(filePath)
                if (file.isDirectory) {
                    val files = file.listFiles()
                    for (file1 in files) {
                        deleteFolderFile(file1.absolutePath, true)
                    }
                }
                if (deleteThisPath) {
                    if (!file.isDirectory) {
                        file.delete()
                    } else {
                        if (file.listFiles().size == 0) {
                            file.delete()
                        }
                    }
                }
            } catch (e: Exception) {
                e.printStackTrace()
            }
        }
    }

    /**
     * 格式化单位
     *
     * @param size size
     * @return size
     */
    fun getFormatSize(size: Double): String {
        val kiloByte = size / 1024
        //        if (kiloByte < 1) {
        //            return size + "Byte";
        //        }
        val megaByte = kiloByte / 1024
        //        if (megaByte < 1) {
        //            BigDecimal result1 = new BigDecimal(Double.toString(kiloByte));
        //            return result1.setScale(2, BigDecimal.ROUND_HALF_UP).toPlainString() + "KB";
        //        }
        val gigaByte = megaByte / 1024
        if (gigaByte < 1) {
            val result2 = BigDecimal(java.lang.Double.toString(megaByte))
            return result2.setScale(2, BigDecimal.ROUND_HALF_UP).toPlainString() + "MB"
        }
        val teraBytes = gigaByte / 1024
        if (teraBytes < 1) {
            val result3 = BigDecimal(java.lang.Double.toString(gigaByte))
            return result3.setScale(2, BigDecimal.ROUND_HALF_UP).toPlainString() + "GB"
        }
        val result4 = BigDecimal(teraBytes)
        return result4.setScale(2, BigDecimal.ROUND_HALF_UP).toPlainString() + "TB"
    }
    fun calcProgressToView(progressBar: ProgressBar, offset: Long, total: Long) {
        val percent = offset.toFloat() / total
        progressBar.progress = (percent * progressBar.max).toInt()
    }


    fun getParentFile(context: Context): File {
        val externalSaveDir = context.externalCacheDir
        return externalSaveDir ?: context.cacheDir
    }
}