package com.softgarden.baselibrary.utils

import java.text.ParseException
import java.text.SimpleDateFormat
import java.util.*
import java.util.regex.Pattern

class IDCardUtil private constructor() {
    companion object {
        /*********************************** 身份证验证开始  */
        /**
         * 身份证号码验证 1、号码的结构 公民身份号码是特征组合码，由十七位数字本体码和一位校验码组成。排列顺序从左至右依次为：六位数字地址码，
         * 八位数字出生日期码，三位数字顺序码和一位数字校验码。 2、地址码(前六位数）
         * 表示编码对象常住户口所在县(市、旗、区)的行政区划代码，按GB/T2260的规定执行。 3、出生日期码（第七位至十四位）
         * 表示编码对象出生的年、月、日，按GB/T7408的规定执行，年、月、日代码之间不用分隔符。 4、顺序码（第十五位至十七位）
         * 表示在同一地址码所标识的区域范围内，对同年、同月、同日出生的人编定的顺序号， 顺序码的奇数分配给男性，偶数分配给女性。 5、校验码（第十八位数）
         * （1）十七位数字本体码加权求和公式 S = Sum(Ai * Wi), i = 0, ... , 16 ，先对前17位数字的权求和
         * Ai:表示第i位置上的身份证号码数字值 Wi:表示第i位置上的加权因子 Wi: 7 9 10 5 8 4 2 1 6 3 7 9 10 5 8 4 2
         * （2）计算模 Y = mod(S, 11) （3）通过模得到对应的校验码 Y: 0 1 2 3 4 5 6 7 8 9 10 校验码: 1 0 X 9 8 7 6 5 4 3 2
         */
        /**
         * 功能：身份证的有效验证
         *
         * @param IDStr 身份证号
         * @return 有效：返回"" 无效：返回String信息
         * @throws ParseException
         */
        @Throws(ParseException::class)
        fun validate(IDStr: String): String {
            var IDStr = IDStr
            IDStr = IDStr.toLowerCase()
            var errorInfo = "" // 记录错误信息
            val ValCodeArr = arrayOf("1", "0", "x", "9", "8", "7", "6", "5", "4",
                    "3", "2")
            val Wi = arrayOf("7", "9", "10", "5", "8", "4", "2", "1", "6", "3", "7",
                    "9", "10", "5", "8", "4", "2")
            var Ai = ""
            // ================ 号码的长度 15位或18位 ================
            if (IDStr.length != 18) {
                errorInfo = "身份证号码长度应该为18位。"
                return errorInfo
            }
            // =======================(end)========================

            // ================ 数字 除最后以为都为数字 ================
            if (IDStr.length == 18) {
                Ai = IDStr.substring(0, 17)
            } else if (IDStr.length == 15) {
                Ai = IDStr.substring(0, 6) + "19" + IDStr.substring(6, 15)
            }
            if (!isNumeric(Ai)) {
                errorInfo = "身份证15位号码都应为数字 ; 18位号码除最后一位外，都应为数字。"
                return errorInfo
            }
            // =======================(end)========================

            // ================ 出生年月是否有效 ================
            val strYear = Ai.substring(6, 10) // 年份
            val strMonth = Ai.substring(10, 12) // 月份
            val strDay = Ai.substring(12, 14) // 月份
            if (!isDate("$strYear-$strMonth-$strDay")) {
                errorInfo = "身份证生日无效。"
                return errorInfo
            }
            val gc = GregorianCalendar()
            val s = SimpleDateFormat("yyyy-MM-dd")
            try {
                if (gc[Calendar.YEAR] - strYear.toInt() > 150
                        || gc.time.time - s.parse(
                                "$strYear-$strMonth-$strDay").time < 0) {
                    errorInfo = "身份证生日不在有效范围。"
                    return errorInfo
                }
            } catch (e: NumberFormatException) {
                e.printStackTrace()
            } catch (e: ParseException) {
                e.printStackTrace()
            }
            if (strMonth.toInt() > 12 || strMonth.toInt() == 0) {
                errorInfo = "身份证月份无效"
                return errorInfo
            }
            if (strDay.toInt() > 31 || strDay.toInt() == 0) {
                errorInfo = "身份证日期无效"
                return errorInfo
            }
            // =====================(end)=====================

            // ================ 地区码时候有效 ================
            val h = GetAreaCode()
            if (h[Ai.substring(0, 2)] == null) {
                errorInfo = "身份证地区编码错误。"
                return errorInfo
            }
            // ==============================================

            // ================ 判断最后一位的值 ================
            var TotalmulAiWi = 0
            for (i in 0..16) {
                TotalmulAiWi = (TotalmulAiWi
                        + Ai[i].toString().toInt() * Wi[i].toInt())
            }
            val modValue = TotalmulAiWi % 11
            val strVerifyCode = ValCodeArr[modValue]
            Ai = Ai + strVerifyCode
            if (IDStr.length == 18) {
                if (Ai != IDStr) {
                    errorInfo = "身份证无效，不是合法的身份证号码"
                    return errorInfo
                }
            } else {
                return ""
            }
            // =====================(end)=====================
            return ""
        }

        /**
         * 功能：设置地区编码
         *
         * @return Hashtable 对象
         */
        private fun GetAreaCode(): Hashtable<*, *> {
            val hashtable: Hashtable<String, String> = Hashtable()
            hashtable["11"] = "北京"
            hashtable["12"] = "天津"
            hashtable["13"] = "河北"
            hashtable["14"] = "山西"
            hashtable["15"] = "内蒙古"
            hashtable["21"] = "辽宁"
            hashtable["22"] = "吉林"
            hashtable["23"] = "黑龙江"
            hashtable["31"] = "上海"
            hashtable["32"] = "江苏"
            hashtable["33"] = "浙江"
            hashtable["34"] = "安徽"
            hashtable["35"] = "福建"
            hashtable["36"] = "江西"
            hashtable["37"] = "山东"
            hashtable["41"] = "河南"
            hashtable["42"] = "湖北"
            hashtable["43"] = "湖南"
            hashtable["44"] = "广东"
            hashtable["45"] = "广西"
            hashtable["46"] = "海南"
            hashtable["50"] = "重庆"
            hashtable["51"] = "四川"
            hashtable["52"] = "贵州"
            hashtable["53"] = "云南"
            hashtable["54"] = "西藏"
            hashtable["61"] = "陕西"
            hashtable["62"] = "甘肃"
            hashtable["63"] = "青海"
            hashtable["64"] = "宁夏"
            hashtable["65"] = "新疆"
            hashtable["71"] = "台湾"
            hashtable["81"] = "香港"
            hashtable["82"] = "澳门"
            hashtable["91"] = "国外"
            return hashtable
        }

        /**
         * 功能：判断字符串是否为数字
         *
         * @param str
         * @return
         */
        private fun isNumeric(str: String): Boolean {
            val pattern = Pattern.compile("[0-9]*")
            val isNum = pattern.matcher(str)
            return isNum.matches()
        }

        /**
         * 功能：判断字符串是否为日期格式
         *
         * @param strDate
         * @return
         */
        fun isDate(strDate: String?): Boolean {
            val pattern = Pattern
                    .compile("^((\\d{2}(([02468][048])|([13579][26]))[\\-/\\s]?((((0?[13578])|(1[02]))[\\-/\\s]?((0?[1-9])|([1-2][0-9])|(3[01])))|(((0?[469])|(11))[\\-/\\s]?((0?[1-9])|([1-2][0-9])|(30)))|(0?2[\\-/\\s]?((0?[1-9])|([1-2][0-9])))))|(\\d{2}(([02468][1235679])|([13579][01345789]))[\\-/\\s]?((((0?[13578])|(1[02]))[\\-/\\s]?((0?[1-9])|([1-2][0-9])|(3[01])))|(((0?[469])|(11))[\\-/\\s]?((0?[1-9])|([1-2][0-9])|(30)))|(0?2[\\-/\\s]?((0?[1-9])|(1[0-9])|(2[0-8]))))))(\\s(((0?[0-9])|([1-2][0-3])):([0-5]?[0-9])((\\s)|(:([0-5]?[0-9])))))?$")
            val m = pattern.matcher(strDate)
            return m.matches()
        }

        /**
         * 是否是合法的身份证号码
         */
        fun isIdCard(id: String): Boolean {
            try {
                val result = validate(id).trim { it <= ' ' }
                if ("" == result) {
                    return true
                }
                ToastUtil.Companion.s(result)
            } catch (e: ParseException) {
                e.printStackTrace()
            }
            return false
        }
        /*********************************** 身份证验证结束  */
    }

    init {
        throw UnsupportedOperationException("u can't fuck me...")
    }
}