package com.softgarden.baselibrary.utils

import android.R
import android.app.Activity
import android.app.KeyguardManager
import android.content.Context
import android.content.pm.ActivityInfo
import android.content.res.Configuration
import android.content.res.Resources
import android.graphics.Bitmap
import android.os.Build
import android.util.DisplayMetrics
import android.util.TypedValue
import android.view.Window
import android.view.WindowManager
import com.softgarden.baselibrary.BaseApplication

/**
 * @author by DELL
 * @date on 2020/06/11
 * @describe 屏幕相关工具类
 */
class ScreenUtil private constructor() {
    companion object {
        /**
         * Return whether screen is landscape.
         *
         * @return `true`: yes<br></br>`false`: no
         */
        fun isLandscape(activity: Activity): Boolean {
            return (activity.resources.configuration.orientation
                    == Configuration.ORIENTATION_LANDSCAPE)
        }

        /**
         * Return whether screen is portrait.
         *
         * @return `true`: yes<br></br>`false`: no
         */
        fun isPortrait(activity: Activity): Boolean {
            return (activity.resources.configuration.orientation
                    == Configuration.ORIENTATION_PORTRAIT)
        }

        /**
         * Adapt the screen for vertical slide.
         *
         * @param designWidthInDp The size of design diagram's width, in dp,
         *
         *
         * e.g. the design diagram width is 720px, in XHDPI device,
         *
         *
         * the designWidthInDp = 720 / 2.
         */
        fun adaptScreenPortrait(activity: Activity,
                                designWidthInDp: Int) {
            adaptScreen(activity, designWidthInDp.toFloat(), true)
        }

        /**
         * Adapt the screen for horizontal slide.
         *
         * @param designHeightInDp The size of design diagram's height, in dp,
         *
         *
         * e.g. the design diagram height is 1080px, in XXHDPI device,
         *
         *
         * the designHeightInDp = 1080 / 3.
         */
        fun adaptScreenLandscape(activity: Activity,
                                 designHeightInDp: Int) {
            adaptScreen(activity, designHeightInDp.toFloat(), false)
        }

        /**
         * Cancel adapt the screen.
         *
         * @param activity The activity.
         */
        fun cancelAdaptScreen(activity: Activity) {
            val activityDm = activity.resources.displayMetrics
            activityDm.density = density
            activityDm.scaledDensity = scaledDensity
            activityDm.densityDpi = densityDpi.toInt()
        }

        private var scaledDensity = 0f
        private var density = 0f
        private var densityDpi = 0f

        /**
         * Reference from: https://mp.weixin.qq.com/s/d9QCoBP6kV9VSWvVldVVwA
         * 我知道你的意思 依mdpi为例，设计图标的多少xml就写多少 ，实际就是1：1的意思，
         */
        private fun adaptScreen(activity: Activity,
                                sizeInDp: Float,
                                isVerticalSlide: Boolean) {
            val appDm: DisplayMetrics = BaseApplication.Companion.instance?.resources!!.displayMetrics
            //每次创建Activity时，这里的值都会是初始值
            density = appDm.density
            scaledDensity = appDm.scaledDensity
            densityDpi = appDm.densityDpi.toFloat()
            val sysDm = Resources.getSystem().displayMetrics
            val activityDm = activity.resources.displayMetrics
            if (isVerticalSlide) {
                activityDm.density = activityDm.widthPixels / sizeInDp
            } else {
                activityDm.density = activityDm.heightPixels / sizeInDp
            }
            L.d("ScreenUtil", "density=" + density + " appDm.density=" + appDm.density + " appDm.density==" + sysDm.density + " activityDm.density= " + activityDm.density)

            //当activity的density变化时，application的density也会跟着变化，所以这里要使用前面保存的值
            // activityDm.scaledDensity = activityDm.density * (scaledDensity / density);
            activityDm.scaledDensity = activityDm.density * (sysDm.scaledDensity / sysDm.density)
            activityDm.densityDpi = (160 * activityDm.density).toInt()
            appDm.density = activityDm.density
            appDm.scaledDensity = activityDm.scaledDensity
            appDm.densityDpi = activityDm.densityDpi
        }

        /**
         * 获取屏幕的宽度px
         *
         * @param context 上下文
         * @return 屏幕宽px
         */
        fun getScreenWidth(context: Context): Int {
            val windowManager = context.getSystemService(Context.WINDOW_SERVICE) as WindowManager
            val outMetrics = DisplayMetrics() // 创建了一张白纸
            windowManager.defaultDisplay.getMetrics(outMetrics) // 给白纸设置宽高
            return outMetrics.widthPixels
        }

        /**
         * 获取屏幕的高度px
         *
         * @param context 上下文
         * @return 屏幕高px
         */
        fun getScreenHeight(context: Context): Int {
            val windowManager = context.getSystemService(Context.WINDOW_SERVICE) as WindowManager
            val outMetrics = DisplayMetrics() // 创建了一张白纸
            windowManager.defaultDisplay.getMetrics(outMetrics) // 给白纸设置宽高
            return outMetrics.heightPixels
        }

        /**
         * 获取屏幕的真实高低 即包含底部虚拟导航栏
         *
         * @param context
         * @return
         */
        fun getScreenRealHeight(context: Context): Int {
            val windowManager = context.getSystemService(Context.WINDOW_SERVICE) as WindowManager
            val display = windowManager.defaultDisplay
            val dm = DisplayMetrics()
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
                display.getRealMetrics(dm)
            } else {
                display.getMetrics(dm)
            }
            return dm.heightPixels
        }

        /**
         * 横屏时
         *
         * @param context
         * @return
         */
        fun getScreenRealWidth(context: Context): Int {
            val windowManager = context.getSystemService(Context.WINDOW_SERVICE) as WindowManager
            val display = windowManager.defaultDisplay
            val dm = DisplayMetrics()
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
                display.getRealMetrics(dm)
            } else {
                display.getMetrics(dm)
            }
            return dm.widthPixels
        }

        /**
         * 设置透明状态栏(api大于19方可使用)
         *
         * 可在Activity的onCreat()中调用
         *
         * 需在顶部控件布局中加入以下属性让内容出现在状态栏之下
         *
         * android:clipToPadding="true"
         *
         * android:fitsSystemWindows="true"
         *
         * @param activity activity
         */
        fun setTransparentStatusBar(activity: Activity) {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
                //透明状态栏
                activity.window.addFlags(WindowManager.LayoutParams.FLAG_TRANSLUCENT_STATUS)
                //透明导航栏
                activity.window.addFlags(WindowManager.LayoutParams.FLAG_TRANSLUCENT_NAVIGATION)
            }
        }

        /**
         * 隐藏状态栏
         *
         * 也就是设置全屏，一定要在setContentView之前调用，否则报错
         *
         * 此方法Activity可以继承AppCompatActivity
         *
         * 启动的时候状态栏会显示一下再隐藏，比如QQ的欢迎界面
         *
         * 在配置文件中Activity加属性android:theme="@android:style/Theme.NoTitleBar.Fullscreen"
         *
         * 如加了以上配置Activity不能继承AppCompatActivity，会报错
         *
         * @param activity activity
         */
        fun hideStatusBar(activity: Activity) {
            activity.requestWindowFeature(Window.FEATURE_NO_TITLE)
            activity.window.setFlags(WindowManager.LayoutParams.FLAG_FULLSCREEN,
                    WindowManager.LayoutParams.FLAG_FULLSCREEN)
        }

        /**
         * 获取状态栏高度
         *
         * @param context 上下文
         * @return 状态栏高度
         */
        fun getStatusBarHeight(context: Context): Int {
            var result = 0
            val resourceId = context.resources
                    .getIdentifier("status_bar_height", "dimen", "android")
            if (resourceId > 0) {
                result = context.resources.getDimensionPixelSize(resourceId)
            }
            return result
        }

        /**
         * 获取导航栏高度
         * @param context
         * @return
         */
        fun getNegativeHeight(context: Context): Int {
            val result = 0
            var resourceId = 0
            val rid = context.resources.getIdentifier("config_showNavigationBar", "bool", "android")
            return if (rid != 0) {
                resourceId = context.resources.getIdentifier("navigation_bar_height", "dimen", "android")
                context.resources.getDimensionPixelSize(resourceId)
            } else 0
        }

        /**
         * 判断状态栏是否存在
         *
         * @param activity activity
         * @return `true`: 存在<br></br>`false`: 不存在
         */
        fun isStatusBarExists(activity: Activity): Boolean {
            val params = activity.window.attributes
            return params.flags and WindowManager.LayoutParams.FLAG_FULLSCREEN != WindowManager.LayoutParams.FLAG_FULLSCREEN
        }

        /**
         * 获取ActionBar高度
         *
         * @param activity activity
         * @return ActionBar高度
         */
        fun getActionBarHeight(activity: Activity): Int {
            val tv = TypedValue()
            return if (activity.theme.resolveAttribute(R.attr.actionBarSize, tv, true)) {
                TypedValue.complexToDimensionPixelSize(tv.data, activity.resources.displayMetrics)
            } else 0
        }

        /**
         * 显示通知栏
         *
         * 需添加权限 `<uses-permission android:name="android.permission.EXPAND_STATUS_BAR"/>`
         *
         * @param context        上下文
         * @param isSettingPanel `true`: 打开设置<br></br>`false`: 打开通知
         */
        fun showNotificationBar(context: Context, isSettingPanel: Boolean) {
            val methodName = if (Build.VERSION.SDK_INT <= 16) "expand" else if (isSettingPanel) "expandSettingsPanel" else "expandNotificationsPanel"
            invokePanels(context, methodName)
        }

        /**
         * 隐藏通知栏
         *
         * 需添加权限 `<uses-permission android:name="android.permission.EXPAND_STATUS_BAR"/>`
         *
         * @param context 上下文
         */
        fun hideNotificationBar(context: Context) {
            val methodName = if (Build.VERSION.SDK_INT <= 16) "collapse" else "collapsePanels"
            invokePanels(context, methodName)
        }

        /**
         * 反射唤醒通知栏
         *
         * @param context    上下文
         * @param methodName 方法名
         */
        private fun invokePanels(context: Context, methodName: String) {
            try {
                val service = context.getSystemService("statusbar")
                val statusBarManager = Class.forName("android.app.StatusBarManager")
                val expand = statusBarManager.getMethod(methodName)
                expand.invoke(service)
            } catch (e: Exception) {
                e.printStackTrace()
            }
        }

        /**
         * 设置屏幕为横屏
         *
         * 还有一种就是在Activity中加属性android:screenOrientation="landscape"
         *
         * 不设置Activity的android:configChanges时，切屏会重新调用各个生命周期，切横屏时会执行一次，切竖屏时会执行两次
         *
         * 设置Activity的android:configChanges="orientation"时，切屏还是会重新调用各个生命周期，切横、竖屏时只会执行一次
         *
         * 设置Activity的android:configChanges="orientation|keyboardHidden|screenSize"（4.0以上必须带最后一个参数）时
         * 切屏不会重新调用各个生命周期，只会执行onConfigurationChanged方法
         *
         * @param activity activity
         */
        fun setLandscape(activity: Activity) {
            activity.requestedOrientation = ActivityInfo.SCREEN_ORIENTATION_LANDSCAPE
        }

        /**
         * 获取当前屏幕截图，包含状态栏
         *
         * @param activity activity
         * @return Bitmap
         */
        fun captureWithStatusBar(activity: Activity): Bitmap {
            val view = activity.window.decorView
            view.isDrawingCacheEnabled = true
            view.buildDrawingCache()
            val bmp = view.drawingCache
            val width = getScreenWidth(activity)
            val height = getScreenHeight(activity)
            val bp = Bitmap.createBitmap(bmp, 0, 0, width, height)
            view.destroyDrawingCache()
            return bp
        }

        /**
         * 获取当前屏幕截图，不包含状态栏
         *
         * 需要用到上面获取状态栏高度getStatusBarHeight的方法
         *
         * @param activity activity
         * @return Bitmap
         */
        fun captureWithoutStatusBar(activity: Activity): Bitmap {
            val view = activity.window.decorView
            view.isDrawingCacheEnabled = true
            view.buildDrawingCache()
            val bmp = view.drawingCache
            val statusBarHeight = getStatusBarHeight(activity)
            val width = getScreenWidth(activity)
            val height = getScreenHeight(activity)
            val bp = Bitmap.createBitmap(bmp, 0, statusBarHeight, width, height - statusBarHeight)
            view.destroyDrawingCache()
            return bp
        }

        /**
         * 判断是否锁屏
         *
         * @param context 上下文
         * @return `true`: 是<br></br>`false`: 否
         */
        fun isScreenLock(context: Context): Boolean {
            val km = context
                    .getSystemService(Context.KEYGUARD_SERVICE) as KeyguardManager
            return km.inKeyguardRestrictedInputMode()
        }
    }

    init {
        throw UnsupportedOperationException("u can't fuck me...")
    }
}