package com.live.common.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.live.common.constant.ConstantValue;
import com.live.common.constant.RedisKeySplicing;
import com.live.common.domain.ResponseData;
import com.live.common.domain.ResponseStatusEnum;
import com.live.common.domain.ResultPage;
import com.live.common.domain.dto.api.*;
import com.live.common.domain.dto.back.UserApplyAnchorListResponse;
import com.live.common.domain.dto.back.UserListResponse;
import com.live.common.domain.entity.*;
import com.live.common.domain.request.*;
import com.live.common.domain.request.back.BackGetUserRequest;
import com.live.common.domain.request.back.CreateAccountRequest;
import com.live.common.domain.request.back.EditApplyAnchorRequest;
import com.live.common.domain.request.back.EditUserInfoBackRequest;
import com.live.common.enums.StatusEnum;
import com.live.common.enums.SystemConfigEnum;
import com.live.common.mapper.*;
import com.live.common.service.HuanXinIMService;
import com.live.common.service.RedisUtilsService;
import com.live.common.service.UserService;
import com.live.common.utils.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.RandomStringUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.locks.ReentrantLock;
import java.util.stream.Collectors;

@Slf4j
@Service
public class UserServiceImpl implements UserService {

    @Resource
    private UserMapper userMapper;
    @Resource
    private FeedbackMapper feedbackMapper;
    @Resource
    private UserGoldMapper userGoldMapper;
    @Resource
    private UserReserveMapper userReserveMapper;
    @Resource
    private RedisUtilsService redisUtilsService;
    @Resource
    private ApplyAnchorMapper applyAnchorMapper;
    @Resource
    private UserMessageMapper userMessageMapper;
    @Resource
    private SystemConfigMapper systemConfigMapper;

    @Value("${surroundings}")
    private String surroundings;

    @Override
    public ResponseData<?> sendSms(SendSmsRequest sendSmsRequest, HttpServletRequest request) {
        String phone = sendSmsRequest.getPhone();
        String requestLock = RedisKeySplicing.requestLock("sendSms", phone);
        if(redisUtilsService.existsKey(requestLock)) {
            return ResponseData.fail400Response("请求太快~");
        }
        redisUtilsService.cacheStringInfo(requestLock, "1", 3);

        try {
            String key = RedisKeySplicing.getSmsCodeKey(phone);
            if (redisUtilsService.existsKey(key)) {
                return ResponseData.fail400Response("请 3 分钟后再尝试");
            }
            String verifyCode = RandomStringUtils.randomNumeric(6);
            redisUtilsService.cacheStringInfo(key, verifyCode, 180);
            JuheDemo.mobileQuery(phone, verifyCode);
            log.info("{}:发送短信", phone);
            return ResponseData.successResponse();
        } catch (Exception e) {
            e.printStackTrace();
            log.error("发送验证码失败:{}", e.getMessage());
        }
        return ResponseData.failResponse();
    }

    @Override
    @Transactional
    public ResponseData<?> register(RegisterRequest registerRequest, HttpServletRequest request) {
        User user = userMapper.selectOne(Wrappers.<User>lambdaQuery()
                .eq(User::getPhone, registerRequest.getPhone())
        );
        if (user != null) {
            return login(registerRequest.getPhone(), registerRequest.getSmsCode());
        }

        String cacheCode = redisUtilsService.getCacheStringInfo(RedisKeySplicing.getSmsCodeKey(registerRequest.getPhone()));
        if (cacheCode == null) {
            return ResponseData.fail400Response("验证码已过期,请重新获取");
        }
        if (!cacheCode.equals(registerRequest.getSmsCode())) {
            return ResponseData.fail400Response("验证码错误");
        }
        String ip = IPUtils.getRealIP(request);

        String deviceId = null;
        try {
            deviceId = CommonMethod.getDeviceId(request);
        } catch (RuntimeException e) {
            //nothing
        }

        User accountDb = User.builder()
                .id(IdGen.uuid())
                .phone(registerRequest.getPhone())
                .ip(ip)
                .userName(registerRequest.getPhone())
                .role(0)
                .userIcon("photo_2021-04-16_10-39-07.jpg")
                .type(0)
                .build();
        createUser(accountDb, deviceId);
        try {
            String token = JWTUtils.generTokenByRS256(accountDb);
            redisUtilsService.cacheStringInfoByDay(RedisKeySplicing.getUserToken(registerRequest.getPhone()), token, 365);
            return ResponseData.successResponse(token);
        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException();
        }
    }

    @Override
    public ResponseData<?> userInfoById(UserInfoByIdRequest ids, HttpServletRequest request) {
        if (ids.getUserIds().size() == 0) {
            return ResponseData.fail400Response("至少传入一个id");
        }
        List<User> account = userMapper.selectBatchIds(ids.getUserIds());
        if (account == null) {
            return ResponseData.fail400Response("id错误用户不存在");
        }

        List<AccountInfoResponse> result = account.stream().map(b -> AccountInfoResponse.builder()
                .avatarUrl(StringUtils.isNotBlank(b.getUserIcon()) ? ConstantValue.AVATAR_URL + b.getUserIcon() : null)
                .id(b.getId())
                .nickName(b.getUserName())
                .remark(b.getUserDesc())
                .role(b.getType())
                .sex(b.getSex())
                .level(b.getLevel())
                .build()).collect(Collectors.toList());
        return ResponseData.successResponse(result);
    }

    @Override
    public ResponseData<?> registerByPc(RegisterPcRequest registerRequest, HttpServletRequest request) {
        User user = userMapper.selectOne(Wrappers.<User>lambdaQuery()
                .eq(User::getPhone, registerRequest.getPhone())
        );
        if (user != null) {
            return ResponseData.fail400Response("用户已存在可直接登录");
        }

        String cacheCode = redisUtilsService.getCacheStringInfo(RedisKeySplicing.getSmsCodeKey(registerRequest.getPhone()));
        if (cacheCode == null) {
            return ResponseData.fail400Response("验证码已过期,请重新获取");
        }
        if (!cacheCode.equals(registerRequest.getSmsCode())) {
            return ResponseData.fail400Response("验证码错误");
        }
        String ip = IPUtils.getRealIP(request);

        String deviceId = null;
        try {
            deviceId = CommonMethod.getDeviceId(request);
        } catch (RuntimeException e) {
            //nothing
        }

        User accountDb = User.builder()
                .id(IdGen.uuid())
                .phone(registerRequest.getPhone())
                .ip(ip)
                .userName(registerRequest.getPhone())
                .role(0)
                .type(0)
                .channel(0)
                .password(MD5Util.string2MD5(registerRequest.getPassword()))
                .build();

        createUser(accountDb, deviceId);
        try {
            String token = JWTUtils.generTokenByRS256(accountDb);
            redisUtilsService.cacheStringInfoByDay(RedisKeySplicing.getUserToken(registerRequest.getPhone()), token, 365);
            return ResponseData.successResponse(token);
        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException();
        }
    }

    @Override
    @Transactional
    public void createUser(User accountDb, String deviceId) {
        User dbUser = null;
        if (StringUtils.isNotBlank(deviceId)) {
            dbUser = userMapper.selectOne(Wrappers.<User>lambdaQuery()
                    .eq(User::getDeviceId, deviceId)
                    .eq(User::getDeleted, StatusEnum.ENABLE.getCode())
            );
        }

        if (dbUser != null) {
            dbUser.setPhone(accountDb.getPhone());
            dbUser.setUserName(accountDb.getUserName());
            dbUser.setUserIcon(accountDb.getUserIcon());
            dbUser.setType(0);
            userMapper.updateById(dbUser);
        } else {
            if (StringUtils.isBlank(accountDb.getUserIcon())) {
                accountDb.setUserIcon("photo_2021-04-16_10-39-07.jpg");
            }
            accountDb.setUserName(accountDb.getUserName());
            userMapper.insert(accountDb);
            userGoldMapper.insert(UserGold.builder()
                    .gold(0)
                    .userId(accountDb.getId())
                    .build());
        }
    }

    @Override
    public User getRandomAuthor() {
        String userId;
        User responseUser;

        List<User> users = userMapper.selectList(Wrappers.<User>lambdaQuery()
                .eq(User::getRole, 1)
                .eq(User::getDeleted, StatusEnum.ENABLE.getCode())
        );
        if (users.size() == 0) {
            userId = IdGen.uuid();
            String fakePhone;
            do {
                fakePhone = IdGen.getPhone();
            } while (userMapper.selectCount(Wrappers.<User>lambdaQuery().eq(User::getPhone, fakePhone)) != 0);

            User accountDb = User.builder()
                    .id(userId)
                    .phone(fakePhone)
                    .ip("127.0.0.1")
                    .userName("官方小编" + IdGen.randomLong(4))
                    .role(1)
                    .type(0)
                    .channel(0)
                    .sex(0)
                    .password(MD5Util.string2MD5("abc123456"))
                    .build();
            createUser(accountDb, null);
            responseUser = userMapper.selectById(userId);
        } else {
            if (users.size() > 1)
                Collections.shuffle(users);
            responseUser = users.get(0);
        }
        return responseUser;
    }

    @Override
    public ResponseData<?> getUserMsgList(CommonPage commonPage, HttpServletRequest request) {
        String phone = CommonMethod.getUserPhone(request);
        User account = userMapper.selectOne(Wrappers.<User>lambdaQuery()
                .eq(User::getPhone, phone)
        );

        long count = userMessageMapper.selectCount(Wrappers.<UserMessage>lambdaQuery()
                .eq(UserMessage::getDeleted, StatusEnum.ENABLE.getCode())
                .eq(UserMessage::getUserId, account.getId())
        );

        int jumpNum = (commonPage.getPageNum() - 1) * commonPage.getPageSize();

        List<UserMessage> messages = userMessageMapper.selectList(Wrappers.<UserMessage>lambdaQuery()
                .eq(UserMessage::getDeleted, StatusEnum.ENABLE.getCode())
                .eq(UserMessage::getUserId, account.getId())
                .last(String.format(" limit %s,%s", jumpNum, commonPage.getPageSize()))
        );

        List<UserMsgResponse> userMsgResponses = messages.stream().map(b -> UserMsgResponse.builder()
                .id(b.getId())
                .msgContent(b.getMsgContent())
                .msgType(b.getMsgType() == 0 ? "系统消息" : "其它消息")
                .time(DateUtil.format(b.getUpdateTime(), DateUtil.YMDHMS_))
//                .readStatus(b.getReadStatus())
                .build()).collect(Collectors.toList());

        userMsgResponses.add(UserMsgResponse.builder()
                .msgType("系统消息")
                .msgContent("尊敬的用户:\n感谢您选择和使用本 APP!")
                .time("2021-07-28 15:00:00")
                .build());
        return ResponseData.successResponse(new ResultPage<>(commonPage.getPageNum(), commonPage.getPageSize(), (int) count + 1, userMsgResponses));
    }

    @Override
    public ResponseData<?> updateMsgStatus(UserMsgRequest userMsgRequest, HttpServletRequest request) {
        if (userMsgRequest.getIds().size() == 0) {
            return ResponseData.fail400Response("至少一条消息ID");
        }

        List<UserMessage> userMessages = userMessageMapper.selectBatchIds(userMsgRequest.getIds());
        for (UserMessage msg : userMessages) {
            msg.setReadStatus(1);
            userMessageMapper.updateById(msg);
        }
        return ResponseData.successResponse();
    }

    @Override
    public ResponseData<?> getUserList(BackGetUserRequest getUserRequest, HttpServletRequest request) {
        QueryWrapper<User> wrapper = new QueryWrapper<>();
        wrapper.eq("deleted", 0);

        if (getUserRequest.getType() != null) {
            wrapper.eq("type", getUserRequest.getType());
        }

        if (StringUtils.isNotBlank(getUserRequest.getChannel())) {
            wrapper.likeLeft("device_id", getUserRequest.getChannel());
        }

        if (StringUtils.isNotBlank(getUserRequest.getPhone())) {
            wrapper.eq("phone", getUserRequest.getPhone());
        }

        if (StringUtils.isNotBlank(getUserRequest.getRegisterEndTime())) {
            wrapper.le("create_time", getUserRequest.getRegisterEndTime());
        }

        if (StringUtils.isNotBlank(getUserRequest.getRegisterStartTime())) {
            wrapper.ge("create_time", getUserRequest.getRegisterStartTime());
        }

        long count = userMapper.selectCount(wrapper);

        int jumpNum = (getUserRequest.getPageNum() - 1) * getUserRequest.getPageSize();
        wrapper.last(String.format(" limit %s,%s", jumpNum, getUserRequest.getPageSize()));

        List<User> users = userMapper.selectList(wrapper);
        List<UserListResponse> responses = users.stream().map(b -> {
            UserGold userGold = userGoldMapper.selectById(b.getId());
            return UserListResponse.builder()
                    .avatarUrl(ConstantValue.completeAddress(b.getUserIcon()))
                    .desc(b.getUserDesc())
                    .id(b.getId())
                    .phone(encryptedPhone(b.getPhone()))
                    .sex(b.getSex())
                    .type(b.getType())
                    .userName(b.getUserName())
                    .gold(userGold.getGold())
                    .createTime(DateUtil.format(b.getCreateTime(), DateUtil.YMDHMS_))
                    .hotAnchor(b.getHotAnchor())
                    .hotNum(b.getHotNum())
                    .lastLoginTime(DateUtil.format(b.getUpdateTime(), DateUtil.YMDHMS_))
                    .loginPwd(b.getPassword())
                    .build();
        }).collect(Collectors.toList());
        return ResponseData.successResponse(new ResultPage<>(getUserRequest.getPageNum(), getUserRequest.getPageSize(), (int) count, responses));
    }

    @Override
    public ResponseData<?> editUserInfoBack(EditUserInfoBackRequest editUserInfoRequest, HttpServletRequest request) {
        User user = userMapper.selectById(editUserInfoRequest.getId());
        if (user != null) {
            if (StringUtils.isNotBlank(editUserInfoRequest.getAvatarUrl())) {
                user.setUserIcon(editUserInfoRequest.getAvatarUrl());
            }
            if (StringUtils.isNotBlank(editUserInfoRequest.getNickName())) {
                user.setUserName(editUserInfoRequest.getNickName());
            }
            if (StringUtils.isNotBlank(editUserInfoRequest.getLoginPwd())) {
                String password = MD5Util.string2MD5(editUserInfoRequest.getLoginPwd());
                user.setPassword(password);
            }
            user.setHotAnchor(editUserInfoRequest.getHotAnchor());
            user.setHotNum(editUserInfoRequest.getHotNum());
            userMapper.updateById(user);
            return ResponseData.successResponse();
        }

        return ResponseData.fail400Response("ID不正确或头像为null");
    }

    private String encryptedPhone(String phone) {
        if (StringUtils.isBlank(phone) || phone.length() < 10) {
            return phone;
        }
        return phone.substring(0, 3) + "****" + phone.substring(phone.length() - 4);
    }

    @Override
    public ResponseData<?> delUserById(CommonStringId stringId, HttpServletRequest request) {
        User user = userMapper.selectById(stringId.getId());
        user.setDeleted(1);
        userMapper.updateById(user);
        return ResponseData.successResponse();
    }

    @Override
    public ResponseData<?> userApplyAnchorList(CommonPage commonPage, HttpServletRequest request) {
        int jumpNum = (commonPage.getPageNum() - 1) * commonPage.getPageSize();

        long count = applyAnchorMapper.selectCount(Wrappers.<ApplyAnchor>lambdaQuery()
                .eq(ApplyAnchor::getDeleted, StatusEnum.ENABLE.getCode())
                .eq(ApplyAnchor::getStatus, 0)
        );

        List<ApplyAnchor> applyAnchors = applyAnchorMapper.selectList(Wrappers.<ApplyAnchor>lambdaQuery()
                .eq(ApplyAnchor::getDeleted, StatusEnum.ENABLE.getCode())
                .eq(ApplyAnchor::getStatus, 0)
                .last(String.format(" limit %s,%s", jumpNum, commonPage.getPageSize()))
        );

        List<UserApplyAnchorListResponse> responses = applyAnchors.stream().map(b -> {
            User user = userMapper.selectById(b.getUserId());

            return UserApplyAnchorListResponse.builder()
                    .applyTime(DateUtil.format(b.getCreateTime(), DateUtil.YMDHMS_))
                    .cardPhotoPositive(b.getCardPhotoPositive())
                    .cardPhotoReverse(b.getCardPhotoReverse())
                    .id(b.getId())
                    .idCard(b.getIdCard())
                    .realName(b.getRealName())
                    .phone(user.getPhone())
                    .userName(user.getUserName())
                    .build();
        }).collect(Collectors.toList());
        return ResponseData.successResponse(new ResultPage<>(commonPage.getPageNum(), commonPage.getPageSize(), (int) count, responses));
    }

    @Override
    public ResponseData<?> editUserApply(EditApplyAnchorRequest applyAnchorRequest, HttpServletRequest request) {
        ApplyAnchor applyAnchor = applyAnchorMapper.selectById(applyAnchorRequest.getId());

        if (applyAnchor == null) {
            return ResponseData.fail400Response("ID 错误用户未找到");
        }
        if (applyAnchor.getStatus() != 0) {
            return ResponseData.fail400Response("只可以处理申请中的");
        }

        String msg = "恭喜，您的主播申请，现已审核通过。可以开始直播啦 ~~~";
        if (applyAnchorRequest.getStatus() == 0) {
            User user = userMapper.selectById(applyAnchor.getUserId());
            user.setType(1);
            user.setCardPhotoPositive(applyAnchor.getCardPhotoPositive());
            user.setCardPhotoReverse(applyAnchor.getCardPhotoReverse());
            user.setIdCard(applyAnchor.getIdCard());
            user.setRealName(applyAnchor.getRealName());
            userMapper.updateById(user);

            applyAnchor.setStatus(1);
            applyAnchor.setDeleted(1);
            applyAnchorMapper.updateById(applyAnchor);
        } else {
            applyAnchor.setStatus(2);
            applyAnchor.setDeleted(1);
            applyAnchor.setReason(applyAnchorRequest.getReason());
            applyAnchorMapper.updateById(applyAnchor);

            msg = String.format("很遗憾，您的主播申请，未通过，以下是拒绝理由 【%s】", applyAnchorRequest.getReason());
        }

        userMessageMapper.insert(UserMessage.builder()
                .userId(applyAnchor.getUserId())
                .msgType(0)
                .readStatus(0)
                .msgContent(msg)
                .build());
        return ResponseData.successResponse();
    }

    @Override
    public ResponseData<?> createAccount(CreateAccountRequest accountRequest) {
        String fakePhone;
        do {
            fakePhone = IdGen.getPhone();
        } while (userMapper.selectCount(Wrappers.<User>lambdaQuery().eq(User::getPhone, fakePhone)) != 0);

        User accountDb = User.builder()
                .id(StringUtils.isBlank(accountRequest.getUserId()) ? IdGen.uuid() : accountRequest.getUserId())
                .phone(fakePhone)
                .ip("127.0.0.1")
                .userName(accountRequest.getNickName())
                .type(accountRequest.getType())
                .channel(0)
                .sex(accountRequest.getSex())
                .role(accountRequest.getRole())
                .password(MD5Util.string2MD5("123456"))
                .build();
        createUser(accountDb, null);
        return ResponseData.successResponse("PC 登录手机:" + fakePhone + " PC 登录密码: 123456");
    }

    @Override
    public ResponseData<?> loginByPc(LoginPcRequest loginPcRequest, HttpServletRequest request) {
        User user = userMapper.selectOne(Wrappers.<User>lambdaQuery()
                .eq(User::getPhone, loginPcRequest.getPhone())
        );
        if (user == null) {
            return ResponseData.fail400Response("用户不存在,请先注册");
        }

        if (!Objects.equals(MD5Util.string2MD5(loginPcRequest.getPassword()), user.getPassword())) {
            return ResponseData.fail400Response("密码错误");
        }

        try {
            String token = JWTUtils.generTokenByRS256(user);
            redisUtilsService.cacheStringInfoByDay(RedisKeySplicing.getUserToken(user.getPhone()), token, 365);
            updateLoginTime(user);
            return ResponseData.successResponse(token);
        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException();
        }
    }

    private void updateLoginTime(User user) {
        user.setUpdateTime(new Date());
        userMapper.updateById(user);
    }

    @Override
    public ResponseData<?> forgotPassword(ForgotPasswordRequest forgotPasswordRequest, HttpServletRequest request) {
        User user = userMapper.selectOne(Wrappers.<User>lambdaQuery()
                .eq(User::getPhone, forgotPasswordRequest.getPhone())
        );
        if (user == null) {
            return ResponseData.fail400Response("用户不存在,请先注册");
        }

        String cacheCode = redisUtilsService.getCacheStringInfo(RedisKeySplicing.getSmsCodeKey(forgotPasswordRequest.getPhone()));
        if (cacheCode == null) {
            return ResponseData.fail400Response("验证码已过期,请重新获取");
        }
        if (!cacheCode.equals(forgotPasswordRequest.getSmsCode())) {
            return ResponseData.fail400Response("验证码错误");
        }

        user.setPassword(MD5Util.string2MD5(forgotPasswordRequest.getPassword()));
        userMapper.updateById(user);
        return ResponseData.successResponse();
    }

    @Override
    public ResponseData<?> loginOut(HttpServletRequest request) {
        String userPhone = CommonMethod.getUserPhone(request);
        redisUtilsService.removeByKey(RedisKeySplicing.getUserToken(userPhone));
        return ResponseData.successResponse();
    }

    @Override
    public ResponseData<?> userInfo(HttpServletRequest request) {
        String userPhone = CommonMethod.getUserPhone(request);
        User account = userMapper.selectOne(Wrappers.<User>lambdaQuery()
                .eq(User::getPhone, userPhone)
        );

        long count = userMessageMapper.selectCount(Wrappers.<UserMessage>lambdaQuery()
                .eq(UserMessage::getDeleted, StatusEnum.ENABLE.getCode())
                .eq(UserMessage::getUserId, account.getId())
        );

        long collectTeam = userReserveMapper.selectCount(Wrappers.<UserReserve>lambdaQuery()
                .eq(UserReserve::getDeleted, StatusEnum.ENABLE.getCode())
                .eq(UserReserve::getUserId, account.getId())
                .eq(UserReserve::getType, 1)
        );

        long collectArticle = userReserveMapper.selectCount(Wrappers.<UserReserve>lambdaQuery()
                .eq(UserReserve::getDeleted, StatusEnum.ENABLE.getCode())
                .eq(UserReserve::getUserId, account.getId())
                .eq(UserReserve::getType, 2)
        );

        return ResponseData.successResponse(UserInfoResponse.builder()
                .avatarUrl(ConstantValue.completeAddress(account.getUserIcon()))
                .id(account.getId())
                .userName(account.getUserName())
                .userDesc(account.getUserDesc())
                .message((int) count + 1)
                .collectArticle((int) collectArticle)
                .collectTeam((int) collectTeam)
                .build());
    }

    @Override
    public ResponseData<?> editUserInfo(EditUserInfoRequest userInfoRequest, HttpServletRequest request) {
        String userPhone = CommonMethod.getUserPhone(request);
        User account = userMapper.selectOne(Wrappers.<User>lambdaQuery()
                .eq(User::getPhone, userPhone)
        );

        if (StringUtils.isNotBlank(userInfoRequest.getNickName())) {
            account.setUserName(userInfoRequest.getNickName());
        }
        if (StringUtils.isNotBlank(userInfoRequest.getDescription())) {
            account.setUserDesc(userInfoRequest.getDescription());
        }
        if (userInfoRequest.getSex() != null) {
            account.setSex(userInfoRequest.getSex());
        }
        if (StringUtils.isNotBlank(userInfoRequest.getBirthday())) {
            account.setBirthday(DateUtil.parse(userInfoRequest.getBirthday(), DateUtil.YMD_));
        }
        userMapper.updateById(account);
        return ResponseData.successResponse();
    }

    @Override
    public ResponseData<?> editUserAvatar(EditUserAvatarRequest avatarRequest, HttpServletRequest request) {
        String userPhone = CommonMethod.getUserPhone(request);
        User account = userMapper.selectOne(Wrappers.<User>lambdaQuery()
                .eq(User::getPhone, userPhone)
        );

        if (StringUtils.isNotBlank(avatarRequest.getAvatarUrl())) {
            account.setUserIcon(avatarRequest.getAvatarUrl());
        }
        userMapper.updateById(account);
        return ResponseData.successResponse();
    }

    @Override
    public ResponseData<?> getFeedbackList() {
        List<SystemConfig> systemConfigList = systemConfigMapper.selectList(Wrappers.<SystemConfig>lambdaQuery()
                .eq(SystemConfig::getDeleted, 0)
                .eq(SystemConfig::getConfigName, SystemConfigEnum.FEEDBACK_KEY_LIST.getTitle())
        );

        List<FeedbackTypeResponse> feedbackTypeResponses = systemConfigList.stream().map(feedback -> FeedbackTypeResponse.builder()
                .feedbackType(feedback.getConfigValue())
                .id(feedback.getId())
                .build())
                .collect(Collectors.toList());
        return ResponseData.successResponse(feedbackTypeResponses);
    }

    @Override
    public ResponseData<?> userFeedback(FeedbackRequest feedbackRequest, HttpServletRequest request) {
        try {
            String userPhone = CommonMethod.getUserPhone(request);
            User account = userMapper.selectOne(Wrappers.<User>lambdaQuery()
                    .eq(User::getPhone, userPhone)
            );

            feedbackMapper.insert(Feedback.builder()
                    .contactDetails(feedbackRequest.getContactDetails())
                    .feedbackText(feedbackRequest.getFeedbackText())
                    .feedbackType(feedbackRequest.getId())
                    .feedbackImg(feedbackRequest.getFeedbackImg())
                    .userId(account.getId())
                    .requestType(1)
                    .build());
        } catch (RuntimeException e) {
            return ResponseData.fail400Response("未携带 token ");
        }
        return ResponseData.successResponse();
    }

    @Override
    public ResponseData<?> getUserFeedbackList(HttpServletRequest request) {
        String userPhone = CommonMethod.getUserPhone(request);
        User account = userMapper.selectOne(Wrappers.<User>lambdaQuery()
                .eq(User::getPhone, userPhone)
        );

        List<Feedback> feedbacks = feedbackMapper.selectList(Wrappers.<Feedback>lambdaQuery()
                .eq(Feedback::getUserId, account.getId())
        );

        List<FeedbackListResponse> feedbackList = feedbacks.stream().map(b -> FeedbackListResponse.builder()
                .contactDetails(b.getContactDetails())
                .feedbackImg(ConstantValue.completeAddress(b.getFeedbackImg()))
                .feedbackText(b.getFeedbackText())
                .feedbackType(b.getFeedbackType())
                .build()).collect(Collectors.toList());
        return ResponseData.successResponse(feedbackList);
    }

    @Override
    public ResponseData<?> userApplyAnchor(UserApplyAnchorRequest anchorRequest, HttpServletRequest request) {
        String userPhone = CommonMethod.getUserPhone(request);
        User account = userMapper.selectOne(Wrappers.<User>lambdaQuery()
                .eq(User::getPhone, userPhone)
        );

        if (account.getType() == 1) {
            return ResponseData.fail400Response("您已是主播");
        }

        long apply = applyAnchorMapper.selectCount(Wrappers.<ApplyAnchor>lambdaQuery()
                .eq(ApplyAnchor::getDeleted, StatusEnum.ENABLE.getCode())
                .eq(ApplyAnchor::getUserId, account.getId())
                .eq(ApplyAnchor::getStatus, 0)
        );
        if (apply > 0) {
            return ResponseData.fail400Response("请等待申请结果");
        }

        applyAnchorMapper.insert(ApplyAnchor.builder()
                .userId(account.getId())
                .idCard(anchorRequest.getIdCard())
                .cardPhotoPositive(anchorRequest.getCardPhotoPositive())
                .cardPhotoReverse(anchorRequest.getCardPhotoReverse())
                .realName(anchorRequest.getName())
                .build());
        return ResponseData.successResponse();
    }

    @Override
    public ResponseData<?> updatePhoneSt(UpdatePhoneStRequest loginPcRequest, HttpServletRequest request) {
        String phone = CommonMethod.getUserPhone(request);
        User account = userMapper.selectOne(Wrappers.<User>lambdaQuery()
                .eq(User::getPhone, phone)
        );

        String cacheCode = redisUtilsService.getCacheStringInfo(RedisKeySplicing.getSmsCodeKey(account.getPhone()));
        if (cacheCode == null) {
            return ResponseData.fail400Response("验证码已过期,请重新获取");
        }
        if (!cacheCode.equals(loginPcRequest.getSmsCode())) {
            return ResponseData.fail400Response("验证码错误");
        }
        return ResponseData.successResponse();
    }

    @Override
    public ResponseData<?> updatePhoneNd(UpdatePhoneNdRequest loginPcRequest, HttpServletRequest request) {
        long exist = userMapper.selectCount(Wrappers.<User>lambdaQuery()
                .eq(User::getPhone, loginPcRequest.getPhone())
        );
        if (exist > 0) {
            return ResponseData.fail400Response("该手机号已被绑定");
        }

        String phone = CommonMethod.getUserPhone(request);
        User account = userMapper.selectOne(Wrappers.<User>lambdaQuery()
                .eq(User::getPhone, phone)
        );

        account.setPhone(loginPcRequest.getPhone());
        userMapper.updateById(account);

        redisUtilsService.removeByKey(RedisKeySplicing.getUserToken(phone));
        return ResponseData.successResponse();
    }

    @Override
    public ResponseData<?> updatePhoneByApp(UpdatePhoneByAppRequest loginPcRequest, HttpServletRequest request) {
        long exist = userMapper.selectCount(Wrappers.<User>lambdaQuery()
                .eq(User::getPhone, loginPcRequest.getPhone())
        );
        if (exist > 0) {
            return ResponseData.fail400Response("该手机号已被绑定");
        }

        String phone = CommonMethod.getUserPhone(request);
        User account = userMapper.selectOne(Wrappers.<User>lambdaQuery()
                .eq(User::getPhone, phone)
        );

        String cacheCode = redisUtilsService.getCacheStringInfo(RedisKeySplicing.getSmsCodeKey(loginPcRequest.getPhone()));
        if (cacheCode == null) {
            return ResponseData.fail400Response("验证码已过期,请重新获取");
        }
        if (!cacheCode.equals(loginPcRequest.getSmsCode())) {
            return ResponseData.fail400Response("验证码错误");
        }

        account.setPhone(loginPcRequest.getPhone());
        userMapper.updateById(account);

        redisUtilsService.removeByKey(RedisKeySplicing.getUserToken(phone));
        return ResponseData.successResponse();
    }

    @Override
    public User registerByDeviceId(String deviceId, String ip) {
        User accountDb = User.builder()
                .id(IdGen.uuid())
                .ip(ip)
                .userName("游客" + IdGen.randomLong(6))
                .role(0)
                .type(2)
                .channel(0)
                .deviceId(deviceId)
                .level(1)
                .build();

        createUser(accountDb, deviceId);
        return accountDb;
    }

    private ResponseData<?> login(String phone, String smsCode) {
        String cacheCode = redisUtilsService.getCacheStringInfo(RedisKeySplicing.getSmsCodeKey(phone));
        if (cacheCode == null) {
            return ResponseData.fail400Response("请重新发送验证码");
        }
        if (!cacheCode.equals(smsCode)) {
            return ResponseData.fail400Response("验证码错误");
        }

        User user = userMapper.selectOne(Wrappers.<User>lambdaQuery()
                .eq(User::getPhone, phone)
        );
        if (user.getDeleted() != StatusEnum.ENABLE.getCode()) {
            return ResponseData.failResponse(ResponseStatusEnum.DISABLED_ACCOUNT.getCode(),
                    StringUtils.isBlank(user.getReason()) ? ResponseStatusEnum.DISABLED_ACCOUNT.getName() : user.getReason());
        }

        try {
            String token = JWTUtils.generTokenByRS256(user);
            redisUtilsService.cacheStringInfoByDay(RedisKeySplicing.getUserToken(phone), token, 365);
            updateLoginTime(user);
            return ResponseData.successResponse(token);
        } catch (Exception e) {
            e.printStackTrace();
            log.error("获取token错误:{}", e.getMessage());
        }
        return ResponseData.failResponse();
    }

}
