package com.live.job.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.toolkit.StringUtils;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.live.common.domain.entity.*;
import com.live.common.enums.SystemConfigEnum;
import com.live.common.mapper.*;
import com.live.common.service.UserService;
import com.live.common.utils.DateUtil;
import com.live.common.utils.IdGen;
import com.live.job.constant.CrawlPlatformEnum;
import com.live.job.entity.dto.*;
import com.live.job.service.CrawlMatchService;
import lombok.extern.slf4j.Slf4j;
import org.jsoup.Jsoup;
import org.jsoup.nodes.Document;
import org.jsoup.nodes.Element;
import org.jsoup.select.Elements;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.dao.DuplicateKeyException;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;
import java.io.IOException;
import java.util.*;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.locks.ReentrantLock;
import java.util.stream.Collectors;

import static org.springframework.http.MediaType.APPLICATION_JSON;

@Slf4j
@Service
public class CrawlAlStatServiceImpl implements CrawlMatchService {

    @Resource
    private TeamMapper teamMapper;
    @Resource
    private MatchMapper matchMapper;
    @Resource
    private SportsMapper sportsMapper;
    @Resource
    private RestTemplate restTemplate;
    @Resource
    private MatchOddsMapper matchOddsMapper;
    @Resource
    private ScheduledTaskMapper scheduledTaskMapper;
    @Resource
    private SystemConfigMapper systemConfigMapper;
    @Resource
    private IndexCompanyMapper indexCompanyMapper;
    @Resource
    private FootballStatMapper footballStatMapper;
    @Resource
    private CrawlHistoryMapper crawlHistoryMapper;
    @Resource
    private ThreadPoolExecutor threadPoolExecutor;
    @Resource
    private FootballEventMapper footballEventMapper;
    @Resource
    private TeamCompetitionMapper teamCompetitionMapper;
    @Resource
    private PlayerCompetitionMapper playerCompetitionMapper;
    @Resource
    private FootballLeagueRankMapper footballLeagueRankMapper;
    @Resource
    private BasketballLeagueRankMapper basketballLeagueRankMapper;
    @Resource
    private PlayerCompetitionFootballMapper playerCompetitionFootballMapper;

    private final ReentrantLock lock = new ReentrantLock();

    @Value("${alStat.basketball}")
    private String basketballKey;
    @Value("${alStat.football}")
    private String footballKey;

    private final String urlBasketballKey = "data71.aistat.cn";
    private final String urlFootballKey = "data43.aistat.cn";

    private final HttpHeaders headers;

    //比分统计字段
    private final Map<String, String> recordField = new HashMap<>();

    public CrawlAlStatServiceImpl() {
        headers = new HttpHeaders();
        headers.add("User-Agent", "Mozilla/5.0 (Linux; Android 6.0; Nexus 5 Build/MRA58N) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/89.0.4389.90 Mobile Safari/537.36");
        headers.add("Connection", "keep-alive");
        headers.setAccept(Collections.singletonList(APPLICATION_JSON));

        recordField.put("yelCards", "黄牌");
        recordField.put("redCards", "红牌");
        recordField.put("goals", "进球");
        recordField.put("assists", "助攻");
        recordField.put("shots", "射门");
        recordField.put("shotsOffTag", "射偏");
        recordField.put("unsTouches", "失误");
        recordField.put("fouls", "犯规");
        recordField.put("passes", "传球");
        recordField.put("possessionNum", "控球");
        recordField.put("corners", "角球");
        recordField.put("totalSaves", "扑救");
    }

    /**
     * 爬取赛事信息 (英超 - 等)
     */
//    @Scheduled(fixedRate = 50 * 60 * 1000)
//    @Scheduled(cron = "01 10 02/6 * * ?")
    public void crawlSportsInfoFootball() {
        log.info("爬取赛事基本信息 -> 足球");
        String url = "http://data43.aistat.cn/competitions/infos?key=" + footballKey;
        ResponseEntity<String> response = restTemplate.getForEntity(url, String.class);
        if (response.getStatusCode() == HttpStatus.OK) {
            JSONObject jsonObject = JSONObject.parseObject(response.getBody());
            AlStatSportsFootballDto companyInfoDto = JSONObject.toJavaObject(jsonObject, AlStatSportsFootballDto.class);
            if (companyInfoDto.getState().equals("success")) {
                for (AlStatSportsFootballDto.AlStatCompetition competition : companyInfoDto.getCompetitions()) {
                    Sports sports = sportsMapper.selectOne(Wrappers.<Sports>lambdaQuery()
                            .eq(Sports::getCompetitionShortName, competition.getName())
                            .eq(Sports::getSourceType, 1)
                    );
                    if (sports == null) {
                        sports = new Sports();
                    }
                    sports.setCompetitionName(competition.getNameFull());
                    sports.setCompetitionNameEn(competition.getNameEnFull());
                    sports.setCompetitionShortName(competition.getName());
                    sports.setCompetitionShortNameEn(competition.getNameEn());
                    sports.setCompetitionType(competition.getCompetitionType());
                    sports.setCountry(competition.getCountry());
                    sports.setCurrSeason(competition.getCurrSeason());
                    sports.setSeasons(String.join(",", competition.getSeasons()));
                    sports.setSourceType(1);
                    sports.setTeamType(competition.getTeamType());
                    sports.setSportsId(0);
                    sports.setSourceId(competition.getId());
                    sports.setSportsName("足球");

                    if (StringUtils.isBlank(sports.getId())) {
                        sportsMapper.insert(sports);
                    } else {
                        sportsMapper.updateById(sports);
                    }
                }
            }
        }
    }

    /**
     * 爬取赛事信息 (美国职业篮球赛 - 等)
     */
//    @Scheduled(fixedRate = 50 * 60 * 1000)
//    @Scheduled(cron = "01 01 02/6 * * ?")
    public void crawlSportsInfoBasketball() {
        log.info("爬取赛事基本信息 -> 篮球");
        String url = "http://data71.aistat.cn/basketball/competitions/infos?key=" + basketballKey;
        ResponseEntity<String> response = restTemplate.getForEntity(url, String.class);
        if (response.getStatusCode() == HttpStatus.OK) {
            JSONObject jsonObject = JSONObject.parseObject(response.getBody());
            AlStatSportsBasketballDto companyInfoDto = JSONObject.toJavaObject(jsonObject, AlStatSportsBasketballDto.class);
            if (companyInfoDto.getState().equals("success")) {
                for (AlStatSportsBasketballDto.AlStatBasketballCompetition competition : companyInfoDto.getCompetitions()) {
                    Sports sports = sportsMapper.selectOne(Wrappers.<Sports>lambdaQuery()
                            .eq(Sports::getCompetitionShortName, competition.getName())
                            .eq(Sports::getSourceType, 1)
                    );
                    if (sports == null) {
                        sports = new Sports();
                    }
                    sports.setCompetitionName(competition.getNameFull());
                    sports.setCompetitionNameEn(competition.getNameEnFull());
                    sports.setCompetitionShortName(competition.getName());
                    sports.setCompetitionShortNameEn(competition.getNameEn());
                    sports.setCompetitionType(competition.getCompetitionType());
                    sports.setCurrSeason(competition.getCurrSeason());
                    sports.setSourceType(1);
                    sports.setTeamType(competition.getTeamType());
                    sports.setSportsId(1);
                    sports.setSourceId(competition.getId());
                    sports.setSportsName("篮球");
                    sports.setGender(competition.getGender());
                    sports.setQuarterNum(competition.getQuarterNum());
                    sports.setQuarterTime(competition.getQuarterTime());
                    sports.setCompetitionIcon("http://dt.aistat.cn/competitions/" + competition.getId() + ".png");

                    if (StringUtils.isBlank(sports.getId())) {
                        sportsMapper.insert(sports);
                    } else {
                        sportsMapper.updateById(sports);
                    }
                }
            }
        }
        log.info("爬取赛事基本信息 -> 篮球 完成");
    }

    /**
     * 爬取指数公司信息 -> 足球
     */
//    @Scheduled(fixedRate = 50 * 60 * 1000)
//    @Scheduled(cron = "01 01 01/5 * * ?")
    public void crawlIndexCompanyInfoFootball() {
        log.info("爬取指数公司信息 -> 足球");
        String url = "http://data61.aistat.cn/companies/infos?key=" + footballKey;
        ResponseEntity<String> response = restTemplate.getForEntity(url, String.class);
        if (response.getStatusCode() == HttpStatus.OK) {
            JSONObject jsonObject = JSONObject.parseObject(response.getBody());
            AlStatIndexCompanyInfoDto companyInfoDto = JSONObject.toJavaObject(jsonObject, AlStatIndexCompanyInfoDto.class);
            if (companyInfoDto.getState().equals("success")) {
                for (AlStatIndexCompanyInfoDto.IndexCompanyInfoDto companyInfo : companyInfoDto.getCompanies()) {
                    try {
                        IndexCompany indexCompany = indexCompanyMapper.selectById(companyInfo.getId());
                        if (indexCompany == null) {
                            indexCompanyMapper.insert(IndexCompany.builder()
                                    .companyName(companyInfo.getName())
                                    .country(companyInfo.getCountry())
                                    .id(companyInfo.getId())
                                    .build());
                        } else {
                            indexCompany.setCompanyName(companyInfo.getName());
                            indexCompany.setCountry(companyInfo.getCountry());
                            indexCompanyMapper.updateById(indexCompany);
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            }
        }
        log.info("爬取指数公司信息 -> 足球 完成");
    }

    /**
     * 爬取指数公司信息 -> 篮球
     */
//    @Scheduled(fixedRate = 50 * 60 * 1000)
//    @Scheduled(cron = "01 01 10/5 * * ?")
    public void crawlIndexCompanyInfoBasketball() {
        log.info("爬取指数公司信息 -> 篮球");
        String url = "http://data91.aistat.cn/basketball/companies/infos?key=" + basketballKey;
        ResponseEntity<String> response = restTemplate.getForEntity(url, String.class);
        if (response.getStatusCode() == HttpStatus.OK) {
            JSONObject jsonObject = JSONObject.parseObject(response.getBody());
            AlStatIndexCompanyInfoDto companyInfoDto = JSONObject.toJavaObject(jsonObject, AlStatIndexCompanyInfoDto.class);
            if (companyInfoDto.getState().equals("success")) {
                for (AlStatIndexCompanyInfoDto.IndexCompanyInfoDto companyInfo : companyInfoDto.getCompanies()) {
                    try {
                        IndexCompany indexCompany = indexCompanyMapper.selectById(companyInfo.getId());
                        if (indexCompany == null) {
                            indexCompanyMapper.insert(IndexCompany.builder()
                                    .companyName(companyInfo.getName())
                                    .country(companyInfo.getCountry())
                                    .id(companyInfo.getId())
                                    .build());
                        } else {
                            indexCompany.setCompanyName(companyInfo.getName());
                            indexCompany.setCountry(companyInfo.getCountry());
                            indexCompanyMapper.updateById(indexCompany);
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            }
        }
    }

    /**
     * 爬取赛事积分榜 -> 足球
     */
//    @Scheduled(fixedRate = 50 * 60 * 1000)
//    @Scheduled(cron = "01 01 0 * * ?")
    public void crawlFootballLeagueRank() {
        log.info("爬取赛事积分榜 -> 足球");
        List<SystemConfig> systemConfigs = systemConfigMapper.selectList(Wrappers.<SystemConfig>lambdaQuery()
                .eq(SystemConfig::getConfigName, SystemConfigEnum.FOOTBALL_LEAGUE_RANK_COMPETITION.getTitle())
                .eq(SystemConfig::getDeleted, 0)
        );
        for (SystemConfig systemConfig : systemConfigs) {
            Map<Integer, String> groupNames = competitionGroup(systemConfig.getConfigValue());
            if (groupNames.size() == 0)
                continue;

            String url = String.format("http://data43.aistat.cn/competitions/leagueTables?key=%s&competitionId=%s", footballKey, systemConfig.getConfigValue());
            ResponseEntity<String> response = restTemplate.getForEntity(url, String.class);
            if (response.getStatusCode() == HttpStatus.OK) {
                JSONObject jsonObject = JSONObject.parseObject(response.getBody());
                AlStatFootballLeagueRankDto companyInfoDto = JSONObject.toJavaObject(jsonObject, AlStatFootballLeagueRankDto.class);
                if (companyInfoDto.getState().equals("success")) {
                    for (AlStatFootballLeagueRankDto.LeagueTable companyInfo : companyInfoDto.getLeagueTables()) {
                        try {
                            lock.lock();
                            if (groupNames.get(companyInfo.getGroupId()) == null)
                                continue;
                            Sports sports = sportsMapper.selectOne(Wrappers.<Sports>lambdaQuery()
                                    .eq(Sports::getSourceType, 1)
                                    .eq(Sports::getSourceId, companyInfo.getCompetitionId())
                                    .eq(Sports::getSportsId, 0)
                                    .eq(Sports::getDeleted, 0)
                            );
                            if (sports == null)
                                continue;
                            Team team = saveTeam(companyInfo.getTeamName(), companyInfo.getTeamId());

                            FootballLeagueRank footballLeagueRank = footballLeagueRankMapper.selectOne(Wrappers.<FootballLeagueRank>lambdaQuery()
                                    .eq(FootballLeagueRank::getGroupName, groupNames.get(companyInfo.getGroupId()))
                                    .eq(FootballLeagueRank::getTeamId, team.getId())
                                    .eq(FootballLeagueRank::getSportsId, sports.getId())
                                    .eq(FootballLeagueRank::getDeleted, 0)
                            );
                            if (footballLeagueRank == null)
                                footballLeagueRank = new FootballLeagueRank();

                            footballLeagueRank.setSportsId(sports.getId());
                            footballLeagueRank.setTeamId(team.getId());
                            footballLeagueRank.setDraws(companyInfo.getDraws());
                            footballLeagueRank.setGoalAga(companyInfo.getGoalAga());
                            footballLeagueRank.setGoalDiff(companyInfo.getGoalDiff());
                            footballLeagueRank.setGoalFor(companyInfo.getGoalFor());
                            footballLeagueRank.setGroupName(groupNames.get(companyInfo.getGroupId()));
                            footballLeagueRank.setLosts(companyInfo.getLosts());
                            footballLeagueRank.setPoints(companyInfo.getPoints());
                            footballLeagueRank.setSeason(companyInfo.getSeason());
                            footballLeagueRank.setQualificate(companyInfo.getQualificate());
                            footballLeagueRank.setMatches(companyInfo.getMatches());
                            footballLeagueRank.setWins(companyInfo.getWins());

                            if (footballLeagueRank.getId() == null) {
                                footballLeagueRankMapper.insert(footballLeagueRank);
                            } else {
                                footballLeagueRankMapper.updateById(footballLeagueRank);
                            }
                        } catch (Exception e) {
                            e.printStackTrace();
                            log.error("爬取赛事积分榜 -> 足球 ERROR:{}", e.getMessage());
                        } finally {
                            lock.unlock();
                        }
                    }
                }
            }
            try {
                Thread.sleep(1000);
            } catch (RuntimeException | InterruptedException e) {
                //nothing
            }
        }
    }

    /**
     * 爬取赛事积分榜 -> 篮球
     */
//    @Scheduled(fixedRate = 50 * 60 * 1000)
//    @Scheduled(cron = "02 02 0 * * ?")
    public void crawlBasketballLeagueRank() {
        log.info("爬取赛事积分榜 -> 篮球");
        List<SystemConfig> systemConfigs = systemConfigMapper.selectList(Wrappers.<SystemConfig>lambdaQuery()
                .eq(SystemConfig::getConfigName, SystemConfigEnum.BASKETBALL_LEAGUE_RANK_COMPETITION.getTitle())
                .eq(SystemConfig::getDeleted, 0)
        );
        for (SystemConfig systemConfig : systemConfigs) {
            String url = String.format("http://data71.aistat.cn/basketball/competitions/standings?key=%s&seasonId=%s", basketballKey, systemConfig.getConfigValue());
            ResponseEntity<String> response = restTemplate.getForEntity(url, String.class);
            if (response.getStatusCode() == HttpStatus.OK) {
                JSONObject jsonObject = JSONObject.parseObject(response.getBody());
                AlStatBasketballLeagueRankDto companyInfoDto = JSONObject.toJavaObject(jsonObject, AlStatBasketballLeagueRankDto.class);
                if (companyInfoDto.getState().equals("success")) {
                    for (AlStatBasketballLeagueRankDto.Standing companyInfo : companyInfoDto.getStanding()) {
                        for (AlStatBasketballLeagueRankDto.Stages stages : companyInfo.getStages()) {
                            if (stages.getGroups() == null)
                                continue;
                            for (AlStatBasketballLeagueRankDto.Group group : stages.getGroups()) {
                                if (group.getStandings() == null)
                                    continue;
                                for (AlStatBasketballLeagueRankDto.StandingInfo standingInfo : group.getStandings()) {
                                    try {
                                        lock.lock();
                                        Sports sports = sportsMapper.selectOne(Wrappers.<Sports>lambdaQuery()
                                                .eq(Sports::getSourceType, 1)
                                                .eq(Sports::getSourceId, systemConfig.getConfigValue())
                                                .eq(Sports::getSportsId, 1)
                                                .eq(Sports::getDeleted, 0)
                                        );
                                        if (sports == null)
                                            continue;

                                        Team team = saveTeam(standingInfo.getTeamName(), standingInfo.getTeamId());

                                        BasketballLeagueRank basketballLeagueRank = basketballLeagueRankMapper.selectOne(Wrappers.<BasketballLeagueRank>lambdaQuery()
                                                .eq(BasketballLeagueRank::getGroupName, group.getGroupName())
                                                .eq(BasketballLeagueRank::getTeamId, team.getId())
                                                .eq(BasketballLeagueRank::getSportsId, sports.getId())
                                                .eq(BasketballLeagueRank::getStageName, stages.getStageName())
                                                .eq(BasketballLeagueRank::getDeleted, 0)
                                        );

                                        if (basketballLeagueRank == null)
                                            basketballLeagueRank = new BasketballLeagueRank();

                                        basketballLeagueRank.setTeamId(team.getId());
                                        basketballLeagueRank.setSportsId(sports.getId());
                                        basketballLeagueRank.setWins(standingInfo.getWins());
                                        basketballLeagueRank.setForm(standingInfo.getForm());
                                        basketballLeagueRank.setLosts(standingInfo.getLosts());
                                        basketballLeagueRank.setGroupName(group.getGroupName());
                                        basketballLeagueRank.setSeason(companyInfo.getSeason());
                                        basketballLeagueRank.setStageName(stages.getStageName());
                                        basketballLeagueRank.setMatches(standingInfo.getMatches());
                                        basketballLeagueRank.setPosition(standingInfo.getPosition());
                                        basketballLeagueRank.setPointsFor(standingInfo.getPointsFor());
                                        basketballLeagueRank.setPointsAga(standingInfo.getPointsAga());
                                        basketballLeagueRank.setPointsDiff(standingInfo.getPointsDiff());
                                        basketballLeagueRank.setQualificate(standingInfo.getQualificate());

                                        if (basketballLeagueRank.getId() == null) {
                                            basketballLeagueRankMapper.insert(basketballLeagueRank);
                                        } else {
                                            basketballLeagueRankMapper.updateById(basketballLeagueRank);
                                        }
                                    } catch (Exception e) {
                                        e.printStackTrace();
                                        log.error("爬取赛事积分榜 -> 篮球 ERROR:{}", e.getMessage());
                                    } finally {
                                        lock.unlock();
                                    }
                                }
                            }
                        }
                    }
                }
            }
            try {
                Thread.sleep(1000);
            } catch (RuntimeException | InterruptedException e) {
                //nothing
            }
        }
    }

    /**
     * 10 分钟爬取一次 AlStat 足球实时比分
     */
//    @Scheduled(fixedRate = 2 * 60 * 60 * 1000)
    @Scheduled(cron = "11 11/10 * * * ?")
    public void crawlAlStatFootballScore() {
        String url = String.format("http://data43.aistat.cn/matchs/liveScores?key=%s", footballKey);
        ResponseEntity<String> response = restTemplate.getForEntity(url, String.class);
        if (response.getStatusCode() == HttpStatus.OK) {
            JSONObject jsonObject = JSONObject.parseObject(response.getBody());
            AlStatFootballScoreDto scoreDto = JSONObject.toJavaObject(jsonObject, AlStatFootballScoreDto.class);
            if (scoreDto.getState().equals("success")) {
                for (AlStatFootballScoreDto.LiveScores liveScores : scoreDto.getLiveScores()) {
                    if (liveScores.getStatus() != 1)
                        continue;

                    String uniqueUrl = footballUniqueKey(liveScores.getMatchId());
                    Match match = matchMapper.selectOne(Wrappers.<Match>lambdaQuery()
                            .eq(Match::getDeleted, 0)
                            .eq(Match::getSourceType, 1)
                            .eq(Match::getCrawlUrl, uniqueUrl)
                    );
                    if (match == null)
                        continue;

                    boolean hasScore = liveScores.getScore().contains(":");
                    String[] score = liveScores.getScore().split(":");
                    match.setHomeScore(hasScore ? Integer.parseInt(score[0].replaceAll("\\*", "").trim()) : 0);
                    match.setAwayScore(hasScore ? Integer.parseInt(score[1].replaceAll("\\*", "").trim()) : 0);
                    matchMapper.updateById(match);
                }
            }
        }
    }

    private Map<Integer, String> competitionGroup(String comId) {
        String url = String.format("http://data43.aistat.cn/competitions/groups?key=%s&competitionId=%s", footballKey, comId);
        ResponseEntity<String> response = restTemplate.getForEntity(url, String.class);
        if (response.getStatusCode() == HttpStatus.OK) {
            JSONObject jsonObject = JSONObject.parseObject(response.getBody());
            AlStatCompetitionGroupDto groupDto = JSONObject.toJavaObject(jsonObject, AlStatCompetitionGroupDto.class);
            if (groupDto.getState().equals("success")) {
                return groupDto.getGroups().stream().collect(Collectors.toMap(AlStatCompetitionGroupDto.Group::getId, AlStatCompetitionGroupDto.Group::getGroupName));
            }
        }
        return new HashMap<>();
    }

    @Override
    public void crawlScheduleMatch(String url) {
        crawlScheduleMatchByDay(url, 0);
    }

    @Override
    public void crawlScheduleMatchByDay(String url, int day) {
        if (url.contains(urlBasketballKey)) {
            //篮球
            url += basketballKey + String.format("&beginDate=%s&endDate=%s", DateUtil.format(DateUtil.getStartTime(), DateUtil.YMD)
                    , DateUtil.format(DateUtil.getAddDayStartTime(day), DateUtil.YMD));
            ResponseEntity<String> response = restTemplate.getForEntity(url, String.class);
            if (response.getStatusCode() == HttpStatus.OK) {
                JSONObject jsonObject = JSONObject.parseObject(response.getBody());
                AlStatScheduleMatchDto alStatScheduleMatchDto = JSONObject.toJavaObject(jsonObject, AlStatScheduleMatchDto.class);
                if (alStatScheduleMatchDto.getState().equals("success")) {
                    List<Match> matches = matchMapper.selectList(Wrappers.<Match>lambdaQuery()
                            .eq(Match::getSourceType, 1)
                            .eq(Match::getCategoryId, "1,")
                            .eq(Match::getDeleted, 0)
                            .likeRight(Match::getMatchTime, DateUtil.format(new Date(), DateUtil.YMD_))
                    );
                    Set<String> uniqueUrls = matches.stream().map(Match::getCrawlUrl).collect(Collectors.toSet());

                    for (AlStatScheduleMatchDto.AlStatScheduleMatchInfoDto matchInfoDto : alStatScheduleMatchDto.getMatchs()) {
                        try {
                            String uniqueUrl = basketballUniqueKey(matchInfoDto.getId());
                            uniqueUrls.remove(uniqueUrl);

                            ScheduleBasketballMatchHandle(matchInfoDto);
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    }

                    for(String uniqueUrl : uniqueUrls){
                        Match match = matchMapper.selectOne(Wrappers.<Match>lambdaQuery()
                                .eq(Match::getCrawlUrl, uniqueUrl)
                                .eq(Match::getSourceType, 1)
                        );
                        match.setDeleted(1);
                        matchMapper.updateById(match);
                        scheduledTaskMapper.insert(ScheduledTask.builder()
                                .createTime(new Date())
                                .method("修改不存在的赛事状态")
                                .methodDesc(uniqueUrl)
                                .build());
                    }
                }
            }
        } else {
            //足球
            url += footballKey + String.format("&beginDate=%s&endDate=%s", DateUtil.format(DateUtil.getStartTime(), DateUtil.YMD)
                    , DateUtil.format(DateUtil.getAddDayStartTime(day), DateUtil.YMD));
            ResponseEntity<String> response = restTemplate.getForEntity(url, String.class);
            if (response.getStatusCode() == HttpStatus.OK) {
                JSONObject jsonObject = JSONObject.parseObject(response.getBody());
                AlStatFootballScheduleMatchDto alStatScheduleMatchDto = JSONObject.toJavaObject(jsonObject, AlStatFootballScheduleMatchDto.class);
                if (alStatScheduleMatchDto.getState().equals("success")) {
                    List<Match> matches = matchMapper.selectList(Wrappers.<Match>lambdaQuery()
                            .eq(Match::getSourceType, 1)
                            .eq(Match::getCategoryId, "0,")
                            .eq(Match::getDeleted, 0)
                            .likeRight(Match::getMatchTime, DateUtil.format(new Date(), DateUtil.YMD_))
                    );
                    Set<String> uniqueUrls = matches.stream().map(Match::getCrawlUrl).collect(Collectors.toSet());

                    for (AlStatFootballScheduleMatchDto.AlStatScheduleFootballInfoDto matchInfoDto : alStatScheduleMatchDto.getMatchs()) {
                        try {
                            String uniqueUrl = footballUniqueKey(matchInfoDto.getId());
                            uniqueUrls.remove(uniqueUrl);

                            ScheduleFootballMatchHandle(matchInfoDto);
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    }

                    for(String uniqueUrl : uniqueUrls){
                        Match match = matchMapper.selectOne(Wrappers.<Match>lambdaQuery()
                                .eq(Match::getCrawlUrl, uniqueUrl)
                                .eq(Match::getSourceType, 1)
                        );
                        match.setDeleted(1);
                        matchMapper.updateById(match);
                        scheduledTaskMapper.insert(ScheduledTask.builder()
                                .createTime(new Date())
                                .method("修改不存在的赛事状态")
                                .methodDesc(uniqueUrl)
                                .build());
                    }
                }
            }
        }
    }

    @Override
    public void crawlIndexCompanyFootball(String url) {
        url += footballKey;
        ResponseEntity<String> response = restTemplate.getForEntity(url, String.class);
        if (response.getStatusCode() == HttpStatus.OK) {
            JSONObject jsonObject = JSONObject.parseObject(response.getBody());
            CrawlIndexCompanyFootballDto footballDto = JSONObject.toJavaObject(jsonObject, CrawlIndexCompanyFootballDto.class);
            if (footballDto.getState().equals("success")) {
                for (CrawlIndexCompanyFootballDto.CrawlIndexCompanyFootballInfoDto footballInfoDto : footballDto.getMainOdds()) {
                    threadPoolExecutor.execute(() -> {
                        try {
                            handlerFootballInfo(footballInfoDto);
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    });
                }
            }
        }
    }

    private void handlerFootballInfo(CrawlIndexCompanyFootballDto.CrawlIndexCompanyFootballInfoDto footballInfoDto) {
        String uniqueUrl = footballUniqueKey(footballInfoDto.getMatchId());
        Match match = matchMapper.selectOne(Wrappers.<Match>lambdaQuery()
                .eq(Match::getCrawlUrl, uniqueUrl)
                .eq(Match::getSourceType, 1)
        );
        if (match == null)
            return;

        //欧赔
        if (footballInfoDto.getOuzhiOdds() != null) {
            for (CrawlIndexCompanyFootballDto.IndexFootballInfoOuZhiOddsDto companyOdds : footballInfoDto.getOuzhiOdds()) {
                MatchOdds matchOdds = getMatchOdds(match.getId(), companyOdds.getCompanyId(), companyOdds.getType(), 0);
                if (matchOdds == null)
                    matchOdds = new MatchOdds();

                matchOdds.setChangeTime(DateUtil.parse(companyOdds.getChangeTime(), DateUtil.YMDHMS_));
                matchOdds.setCompanyId(companyOdds.getCompanyId());
                matchOdds.setFieldOne(companyOdds.getWin());
                matchOdds.setFieldTwo(companyOdds.getDraw());
                matchOdds.setFieldThree(companyOdds.getLoss());
                matchOdds.setMatchType(companyOdds.getType());
                matchOdds.setMatchId(match.getId());
                matchOdds.setOddsType(0);

                if (matchOdds.getId() == null) {
                    matchOddsMapper.insert(matchOdds);
                } else {
                    matchOddsMapper.updateById(matchOdds);
                }
            }
        }

        //亚赔
        if (footballInfoDto.getYazhiOdds() != null) {
            for (CrawlIndexCompanyFootballDto.IndexFootballInfoYaZhiOddsDto companyOdds : footballInfoDto.getYazhiOdds()) {
                MatchOdds matchOdds = getMatchOdds(match.getId(), companyOdds.getCompanyId(), companyOdds.getType(), 1);
                if (matchOdds == null)
                    matchOdds = new MatchOdds();

                matchOdds.setChangeTime(DateUtil.parse(companyOdds.getChangeTime(), DateUtil.YMDHMS_));
                matchOdds.setCompanyId(companyOdds.getCompanyId());
                matchOdds.setFieldOne(companyOdds.getHandicap());
                matchOdds.setFieldTwo(companyOdds.getHome());
                matchOdds.setFieldThree(companyOdds.getAway());
                matchOdds.setMatchType(companyOdds.getType());
                matchOdds.setMatchId(match.getId());
                matchOdds.setOddsType(1);

                if (matchOdds.getId() == null) {
                    matchOddsMapper.insert(matchOdds);
                } else {
                    matchOddsMapper.updateById(matchOdds);
                }
            }
        }

        //大小
        if (footballInfoDto.getDaxiaoOdds() != null) {
            for (CrawlIndexCompanyFootballDto.IndexFootballInfoDaXiaoOddsDto companyOdds : footballInfoDto.getDaxiaoOdds()) {
                MatchOdds matchOdds = getMatchOdds(match.getId(), companyOdds.getCompanyId(), companyOdds.getType(), 2);
                if (matchOdds == null)
                    matchOdds = new MatchOdds();

                matchOdds.setChangeTime(DateUtil.parse(companyOdds.getChangeTime(), DateUtil.YMDHMS_));
                matchOdds.setCompanyId(companyOdds.getCompanyId());
                matchOdds.setFieldOne(companyOdds.getHandicap());
                matchOdds.setFieldTwo(companyOdds.getOver());
                matchOdds.setFieldThree(companyOdds.getUnder());
                matchOdds.setMatchType(companyOdds.getType());
                matchOdds.setMatchId(match.getId());
                matchOdds.setOddsType(2);

                if (matchOdds.getId() == null) {
                    matchOddsMapper.insert(matchOdds);
                } else {
                    matchOddsMapper.updateById(matchOdds);
                }
            }
        }
    }

    @Override
    public void crawlIndexCompanyBasketball(String url) {
        url += basketballKey;
        ResponseEntity<String> response = restTemplate.getForEntity(url, String.class);
        if (response.getStatusCode() == HttpStatus.OK) {
            JSONObject jsonObject = JSONObject.parseObject(response.getBody());
            CrawlIndexCompanyBasketballDto basketballDto = JSONObject.toJavaObject(jsonObject, CrawlIndexCompanyBasketballDto.class);
            if (basketballDto.getState().equals("success")) {
                for (CrawlIndexCompanyBasketballDto.CrawlIndexCompanyBasketballInfoDto basketballInfoDto : basketballDto.getMainOdds()) {
                    threadPoolExecutor.execute(() -> {
                        try {
                            handlerBasketballInfo(basketballInfoDto);
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    });
                }
            }
        }
    }

    private void handlerBasketballInfo(CrawlIndexCompanyBasketballDto.CrawlIndexCompanyBasketballInfoDto basketballInfoDto) {
        String uniqueUrl = basketballUniqueKey(basketballInfoDto.getMatchId());
        Match match = matchMapper.selectOne(Wrappers.<Match>lambdaQuery()
                .eq(Match::getCrawlUrl, uniqueUrl)
                .eq(Match::getSourceType, 1)
        );
        if (match == null)
            return;

        //欧赔
        if (basketballInfoDto.getOuzhiOdds() != null) {
            for (CrawlIndexCompanyBasketballDto.IndexBasketballInfoOuZhiOddsDto companyOdds : basketballInfoDto.getOuzhiOdds()) {
                MatchOdds matchOdds = getMatchOdds(match.getId(), companyOdds.getCompanyId(), companyOdds.getType(), 0);
                if (matchOdds == null)
                    matchOdds = new MatchOdds();

                matchOdds.setChangeTime(DateUtil.parse(companyOdds.getChangeTime(), DateUtil.YMDHMS_));
                matchOdds.setCompanyId(companyOdds.getCompanyId());
                matchOdds.setFieldOne(companyOdds.getWin());
                matchOdds.setFieldTwo(0.00);
                matchOdds.setFieldThree(companyOdds.getLoss());
                matchOdds.setMatchType(companyOdds.getType());
                matchOdds.setMatchId(match.getId());
                matchOdds.setOddsType(0);
                matchOdds.setSportsType(1);

                if (matchOdds.getId() == null) {
                    matchOddsMapper.insert(matchOdds);
                } else {
                    matchOddsMapper.updateById(matchOdds);
                }
            }
        }

        //亚赔
        if (basketballInfoDto.getRfOdds() != null) {
            for (CrawlIndexCompanyBasketballDto.IndexBasketballInfoYaZhiOddsDto companyOdds : basketballInfoDto.getRfOdds()) {
                MatchOdds matchOdds = getMatchOdds(match.getId(), companyOdds.getCompanyId(), companyOdds.getType(), 1);
                if (matchOdds == null)
                    matchOdds = new MatchOdds();

                matchOdds.setChangeTime(DateUtil.parse(companyOdds.getChangeTime(), DateUtil.YMDHMS_));
                matchOdds.setCompanyId(companyOdds.getCompanyId());
                matchOdds.setFieldOne(companyOdds.getHandicap());
                matchOdds.setFieldTwo(companyOdds.getHome());
                matchOdds.setFieldThree(companyOdds.getAway());
                matchOdds.setMatchType(companyOdds.getType());
                matchOdds.setMatchId(match.getId());
                matchOdds.setOddsType(1);
                matchOdds.setSportsType(1);

                if (matchOdds.getId() == null) {
                    matchOddsMapper.insert(matchOdds);
                } else {
                    matchOddsMapper.updateById(matchOdds);
                }
            }
        }

        //大小
        if (basketballInfoDto.getDaxiaoOdds() != null) {
            for (CrawlIndexCompanyBasketballDto.IndexBasketballInfoDaXiaoOddsDto companyOdds : basketballInfoDto.getDaxiaoOdds()) {
                MatchOdds matchOdds = getMatchOdds(match.getId(), companyOdds.getCompanyId(), companyOdds.getType(), 2);
                if (matchOdds == null)
                    matchOdds = new MatchOdds();

                matchOdds.setChangeTime(DateUtil.parse(companyOdds.getChangeTime(), DateUtil.YMDHMS_));
                matchOdds.setCompanyId(companyOdds.getCompanyId());
                matchOdds.setFieldOne(companyOdds.getHandicap());
                matchOdds.setFieldTwo(companyOdds.getOver());
                matchOdds.setFieldThree(companyOdds.getUnder());
                matchOdds.setMatchType(companyOdds.getType());
                matchOdds.setMatchId(match.getId());
                matchOdds.setOddsType(2);
                matchOdds.setSportsType(1);

                if (matchOdds.getId() == null) {
                    matchOddsMapper.insert(matchOdds);
                } else {
                    matchOddsMapper.updateById(matchOdds);
                }
            }
        }
    }

    private MatchOdds getMatchOdds(String matchId, int companyId, int type, int oddsType) {
        return matchOddsMapper.selectOne(Wrappers.<MatchOdds>lambdaQuery()
                .eq(MatchOdds::getMatchId, matchId)
                .eq(MatchOdds::getCompanyId, companyId)
                .eq(MatchOdds::getMatchType, type)
                .eq(MatchOdds::getOddsType, oddsType)
        );
    }

    @Override
    public void crawlMatchStageScore(String url) {
        url += basketballKey;
        ResponseEntity<String> response = restTemplate.getForEntity(url, String.class);
        if (response.getStatusCode() == HttpStatus.OK) {
            JSONObject jsonObject = JSONObject.parseObject(response.getBody());
            AlStatMatchStageScoreDto matchStageScoreDto = JSONObject.toJavaObject(jsonObject, AlStatMatchStageScoreDto.class);
            if (matchStageScoreDto.getState().equals("success")) {
                for (AlStatMatchStageScoreDto.AlStatMatchStageScoreInfoDto matchInfoDto : matchStageScoreDto.getLiveScores()) {
                    try {
                        handlerMatchStageScore(matchInfoDto);
                    } catch (Exception e) {
                        e.printStackTrace();
                        log.error("crawlMatchStageScore Error!:{}", e.getMessage());
                    }
                }
            }
        }
    }

    @Transactional
    public void handlerMatchStageScore(AlStatMatchStageScoreDto.AlStatMatchStageScoreInfoDto matchInfoDto) {
        String uniqueUrl = basketballUniqueKey(matchInfoDto.getMatchId());

        Match match = matchMapper.selectOne(Wrappers.<Match>lambdaQuery()
                .eq(Match::getCrawlUrl, uniqueUrl)
                .eq(Match::getSourceType, 1)
        );
        //比赛不存在，或团队比分不存在
        if (match == null || StringUtils.isBlank(match.getCompetitionAwayId()) || StringUtils.isBlank(match.getCompetitionHomeId()))
            return;

        TeamCompetition awayTeam = teamCompetitionMapper.selectById(match.getCompetitionAwayId());
        TeamCompetition homeTeam = teamCompetitionMapper.selectById(match.getCompetitionHomeId());

        String awaySettle = channelInteger(matchInfoDto.getQ1awayScore()) + "," + channelInteger(matchInfoDto.getQ2awayScore())
                + "," + channelInteger(matchInfoDto.getQ3awayScore()) + "," + channelInteger(matchInfoDto.getQ4awayScore())
                + ",0," + channelInteger(matchInfoDto.getAwayScore());

        String homeSettle = channelInteger(matchInfoDto.getQ1homeScore()) + "," + channelInteger(matchInfoDto.getQ2homeScore())
                + "," + channelInteger(matchInfoDto.getQ3homeScore()) + "," + channelInteger(matchInfoDto.getQ4homeScore())
                + ",0," + channelInteger(matchInfoDto.getHomeScore());

        awayTeam.setScoreSettle(awaySettle);
        homeTeam.setScoreSettle(homeSettle);
        teamCompetitionMapper.updateById(awayTeam);
        teamCompetitionMapper.updateById(homeTeam);

        match.setHomeScore(matchInfoDto.getHomeScore());
        match.setAwayScore(matchInfoDto.getAwayScore());
        matchMapper.updateById(match);
    }

    private int channelInteger(Integer score) {
        return score == null ? 0 : score;
    }

    private String footballUniqueKey(Integer matchId) {
        return urlFootballKey + matchId;
    }

    private String basketballUniqueKey(Integer matchId) {
        return urlBasketballKey + matchId;
    }

    public void ScheduleFootballMatchHandle(AlStatFootballScheduleMatchDto.AlStatScheduleFootballInfoDto matchInfoDto) {
        String uniqueUrl = footballUniqueKey(matchInfoDto.getId());
        Match match = matchMapper.selectOne(Wrappers.<Match>lambdaQuery()
                .eq(Match::getCrawlUrl, uniqueUrl)
                .eq(Match::getSourceType, 1)
        );
        CrawlHistory crawlHistory = crawlHistoryMapper.selectOne(Wrappers.<CrawlHistory>lambdaQuery()
                .eq(CrawlHistory::getCrawlUrl, uniqueUrl)
        );
        if (crawlHistory != null)
            return;

        String sportsId = saveSports(matchInfoDto.getCompetitionName(), 0);
        Team away = saveTeam(matchInfoDto.getAwayTeamName(), matchInfoDto.getAwayTeamId());
        Team home = saveTeam(matchInfoDto.getHomeTeamName(), matchInfoDto.getHomeTeamId());

        String matchId = IdGen.uuid();
        if (match != null)
            matchId = match.getId();

        if (matchInfoDto.getStatus() == 2) {
            try {
                crawlHistoryMapper.insert(CrawlHistory.builder().crawlUrl(uniqueUrl).build());
            } catch (DuplicateKeyException d) {
                //nothing
            }
        }

        try {
            boolean hasScore = matchInfoDto.getFinalScore().contains(":");
            String[] score = matchInfoDto.getFinalScore().split(":");
            if (match != null) {
                match.setHomeScore(hasScore ? Integer.parseInt(score[0].replaceAll("\\*", "").trim()) : 0);
                match.setAwayScore(hasScore ? Integer.parseInt(score[1].replaceAll("\\*", "").trim()) : 0);
                //1正在进行 2未开始 3完场 4 未知
                //比赛状态（0:未开始, 1:进行中, 2:已结束, 3:延期, 4:中断, 5:取消）
                match.setCompetitionStatus(statusChange(matchInfoDto.getStatus()));
                matchMapper.updateById(match);
            } else {
                matchMapper.insert(Match.builder()
                        .id(matchId)
                        .sportsId(sportsId)
                        .awayId(away.getId())
                        .homeId(home.getId())
                        .awayScore(hasScore ? Integer.parseInt(score[1].replaceAll("\\*", "").trim()) : 0)
                        .homeScore(hasScore ? Integer.parseInt(score[0].replaceAll("\\*", "").trim()) : 0)
                        .matchTime(DateUtil.parse(matchInfoDto.getStartTime(), DateUtil.YMDHM_))
                        .competitionName(matchInfoDto.getCompetitionName())
                        .competitionShortName(matchInfoDto.getCompetitionName())
                        .competitionStatus(statusChange(matchInfoDto.getStatus()))
                        .crawlUrl(uniqueUrl)
                        .categoryId("0,")
                        .sourceType(1)
                        .build());
                log.info("抓取足球成功");
            }
        } catch (DuplicateKeyException d) {
            //nothing
        }

        //比赛状态（0:未开始, 1:进行中, 2:已结束, 3:延期, 4:中断, 5:取消）
        if (matchInfoDto.getStatus() == 1 || matchInfoDto.getStatus() == 2) {
            lock.lock();
            try {
                competitionFootballTeam(matchId, matchInfoDto.getId());
                //技术统计 团队得分总计
                footballStatTeam(matchId, matchInfoDto.getId());
                //足球阵容 / 个人得分统计
                if (match != null)
                    competitionPlayerFootball(matchInfoDto.getId(), match);
            } finally {
                lock.unlock();
            }
        }
    }

    private void footballStatTeam(String dbId, Integer matchId) {
        try {
            String requestUrl = String.format("http://data43.aistat.cn/matchs/liveTeamStats?key=%s&matchId=%s", footballKey, matchId);
            ResponseEntity<String> response = restTemplate.getForEntity(requestUrl, String.class);
            if (response.getStatusCode() == HttpStatus.OK) {
                JSONObject jsonObject = JSONObject.parseObject(response.getBody());
                AlStatFootballStatDto alStatFootballStatDto = JSONObject.toJavaObject(jsonObject, AlStatFootballStatDto.class);
                if (alStatFootballStatDto.getState().equals("success")) {
                    for (AlStatFootballStatDto.AlStatFootballStatMatch footballStatMatch : alStatFootballStatDto.getMatchs()) {
                        JSONObject homeJson = null;
                        JSONObject awayJson = null;
                        for (AlStatFootballStatDto.AlStatFootballStatStats statStats : footballStatMatch.getStats()) {
                            if (statStats.getTeamId().equals(footballStatMatch.getHomeTeamId())) {
                                homeJson = JSON.parseObject(JSON.toJSONString(statStats));
                            } else {
                                awayJson = JSON.parseObject(JSON.toJSONString(statStats));
                            }
                        }
                        if (homeJson == null || awayJson == null)
                            return;

                        for (String mapKey : recordField.keySet()) {
                            try {
                                footballStatMapper.insert(FootballStat.builder()
                                        .matchId(dbId)
                                        .statAway(awayJson.getString(mapKey))
                                        .statHome(homeJson.getString(mapKey))
                                        .statName(recordField.get(mapKey))
                                        .build());
                            } catch (DuplicateKeyException e) {
                                //nothing
                            } catch (RuntimeException e) {
                                log.error("新增足球比分 footballStatTeam 统计时报错:{}", e.getMessage());
                            }
                        }
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void competitionFootballTeam(String dbId, Integer matchId) {
        try {
            String requestUrl = String.format("http://data43.aistat.cn/matchs/liveTimelines?key=%s&matchId=%s", footballKey, matchId);
            ResponseEntity<String> response = restTemplate.getForEntity(requestUrl, String.class);
            if (response.getStatusCode() == HttpStatus.OK) {
                JSONObject jsonObject = JSONObject.parseObject(response.getBody());
                AlStatFootballEventDto alStatFootballStatDto = JSONObject.toJavaObject(jsonObject, AlStatFootballEventDto.class);
                if (alStatFootballStatDto.getState().equals("success")) {
                    for (AlStatFootballEventDto.AlStatFootballStatMatchDto statMatchDto : alStatFootballStatDto.getMatchs()) {
                        for (AlStatFootballEventDto.AlStatFootballStatInfoDto statInfoDto : statMatchDto.getLiveTimelines()) {
                            try {
                                String upName = "";
                                String downName = "";
                                if (statInfoDto.getDescription().trim().equals("替补登场")) {
                                    upName = StringUtils.isBlank(statInfoDto.getRelatedPlayerName()) ? "" : statInfoDto.getRelatedPlayerName();
                                    downName = StringUtils.isBlank(statInfoDto.getPlayerName()) ? "" : statInfoDto.getPlayerName();
                                }
                                if (statInfoDto.getDescription().trim().equals("替换下场")) {
                                    downName = StringUtils.isBlank(statInfoDto.getRelatedPlayerName()) ? "" : statInfoDto.getRelatedPlayerName();
                                    upName = StringUtils.isBlank(statInfoDto.getPlayerName()) ? "" : statInfoDto.getPlayerName();
                                }
                                footballEventMapper.insert(FootballEvent.builder()
                                        .matchId(dbId)
                                        .assistedName(StringUtils.isBlank(statInfoDto.getRelatedPlayerName()) ? "" : statInfoDto.getRelatedPlayerName())
                                        .downName(downName)
                                        .eventName(StringUtils.isBlank(statInfoDto.getDescription()) ? "" : statInfoDto.getDescription())
                                        .minute(statInfoDto.getMinute())
                                        .pName(StringUtils.isBlank(statInfoDto.getPlayerName()) ? "" : statInfoDto.getPlayerName())
                                        .score(StringUtils.isBlank(statInfoDto.getScore()) ? "" : statInfoDto.getScore())
                                        .type(statInfoDto.getTeamId().equals(statMatchDto.getHomeTeamId()) ? 1 : 2)
                                        .upName(upName)
                                        .build());
                            } catch (DuplicateKeyException e) {
                                //nothing
                            } catch (RuntimeException e) {
                                log.error("新增足球事件时报错:{}", e.getMessage());
                            }
                        }
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Transactional
    public void ScheduleBasketballMatchHandle(AlStatScheduleMatchDto.AlStatScheduleMatchInfoDto matchInfoDto) {
        String uniqueUrl = basketballUniqueKey(matchInfoDto.getId());

        Match match = matchMapper.selectOne(Wrappers.<Match>lambdaQuery()
                .eq(Match::getCrawlUrl, uniqueUrl)
                .eq(Match::getSourceType, 1)
        );
        CrawlHistory crawlHistory = crawlHistoryMapper.selectOne(Wrappers.<CrawlHistory>lambdaQuery()
                .eq(CrawlHistory::getCrawlUrl, uniqueUrl)
        );
        if (crawlHistory != null) {
            return;
        }
        String sportsId = saveSports(matchInfoDto.getCompetitionName(), 1);
        Team away = saveTeam(matchInfoDto.getAwayTeamName(), matchInfoDto.getAwayTeamId());
        Team home = saveTeam(matchInfoDto.getHomeTeamName(), matchInfoDto.getHomeTeamId());

        String matchId = IdGen.uuid();
        if (match != null) {
            matchId = match.getId();
        }

        //比赛状态（0:未开始, 1:进行中, 2:已结束, 3:延期, 4:中断, 5:取消）
        if (matchInfoDto.getStatus() == 1 || matchInfoDto.getStatus() == 2) {
            lock.lock();
            try {
                match = competitionTeam(match, matchInfoDto.getId(), home.getId(), away.getId());
                competitionPlayer(matchInfoDto.getId(), match, home.getId(), away.getId());
            } finally {
                lock.unlock();
            }
        }

        if (matchInfoDto.getStatus() == 2) {
            try {
                crawlHistoryMapper.insert(CrawlHistory.builder().crawlUrl(uniqueUrl).build());
            } catch (DuplicateKeyException d) {
                //nothing
            }
        }

        try {
            if (match != null && StringUtils.isNotBlank(match.getId())) {
                match.setHomeScore(matchInfoDto.getHomeScore());
                match.setAwayScore(matchInfoDto.getAwayScore());
                //1正在进行 2未开始 3完场 4 未知
                //比赛状态（0:未开始, 1:进行中, 2:已结束, 3:延期, 4:中断, 5:取消）
                match.setCompetitionStatus(statusChange(matchInfoDto.getStatus()));
                matchMapper.updateById(match);
            } else {
                if (match == null) {
                    match = new Match();
                }
                match.setId(matchId);
                match.setSportsId(sportsId);
                match.setAwayId(away.getId());
                match.setHomeId(home.getId());
                match.setAwayScore(matchInfoDto.getAwayScore() == null ? 0 : matchInfoDto.getAwayScore());
                match.setHomeScore(matchInfoDto.getHomeScore() == null ? 0 : matchInfoDto.getHomeScore());
                match.setMatchTime(DateUtil.parse(matchInfoDto.getStartTime(), DateUtil.YMDHM_));
                match.setCompetitionName(matchInfoDto.getCompetitionName());
                match.setCompetitionShortName(matchInfoDto.getCompetitionName());
                match.setCompetitionStatus(statusChange(matchInfoDto.getStatus()));
                match.setCrawlUrl(uniqueUrl);
                match.setCategoryId("1,");
                match.setSourceType(1);

                matchMapper.insert(match);
                log.info("抓取成功");
            }
        } catch (DuplicateKeyException d) {
            //nothing
        }
    }

    private int statusChange(int status) {
        switch (status) {
            case 0:
                return 2;
            case 1:
                return 1;
            case 2:
                return 3;
            default:
                return 4;
        }
    }

    //足球个人比分
    private void competitionPlayerFootball(Integer sourceMatchId, Match match) {
        try {
            String requestUrl = String.format("http://data43.aistat.cn/matchs/livePlayerStats?key=%s&matchIds=%s", footballKey, sourceMatchId);
            ResponseEntity<String> response = restTemplate.getForEntity(requestUrl, String.class);
            if (response.getStatusCode() == HttpStatus.OK) {
                JSONObject jsonObject = JSONObject.parseObject(response.getBody());
                AlStatPlayerFootballStatisticsDto alStatScheduleMatchDto = JSONObject.toJavaObject(jsonObject, AlStatPlayerFootballStatisticsDto.class);
                if (alStatScheduleMatchDto.getState().equals("success")) {
                    for (AlStatPlayerFootballStatisticsDto.AlStatScheduleMatchFootballInfoDto footballInfoDto : alStatScheduleMatchDto.getMatchs()) {
                        for (AlStatPlayerFootballStatisticsDto.AlStatStatsInfoFootballDto footballDto : footballInfoDto.getStats()) {
                            try {
                                String dbTeamId = footballDto.getHomeAway().equals("home") ? match.getHomeId() : match.getAwayId();

                                PlayerCompetitionFootball footballCompetition = playerCompetitionFootballMapper.selectOne(Wrappers.<PlayerCompetitionFootball>lambdaQuery()
                                        .eq(PlayerCompetitionFootball::getMatchId, match.getId())
                                        .eq(PlayerCompetitionFootball::getTeamId, dbTeamId)
                                        .eq(PlayerCompetitionFootball::getPlayerName, footballDto.getPlayerName())
                                );

                                if (footballCompetition == null) {
                                    footballCompetition = new PlayerCompetitionFootball();
                                }
                                footballCompetition.setMatchId(match.getId());
                                footballCompetition.setPlayerName(footballDto.getPlayerName());
                                footballCompetition.setPosition(footballDto.getPosition());
                                footballCompetition.setTeamId(dbTeamId);

                                if (footballCompetition.getId() == null) {
                                    playerCompetitionFootballMapper.insert(footballCompetition);
                                } else {
                                    playerCompetitionFootballMapper.updateById(footballCompetition);
                                }
                            } catch (Exception e) {
                                e.printStackTrace();
                            }
                        }
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    //足球赛况
//    private void crawlGameCommentary(String matchId, int sourceMatchId) {
//        try {
//            String requestUrl = String.format("http://data43.aistat.cn/matchs/livePlayerStats?key=%s&matchIds=%s", footballKey, sourceMatchId);
//            ResponseEntity<String> response = restTemplate.getForEntity(requestUrl, String.class);
//            if (response.getStatusCode() == HttpStatus.OK) {
//
//            }
//        } catch (Exception e) {
//            log.error(e.getMessage());
//        }
//    }

    //篮球个人比分
    private void competitionPlayer(Integer sourceMatchId, Match match, String homeId, String awayId) {
        try {
            String requestUrl = String.format("http://data71.aistat.cn/basketball/matchs/playerStats?key=%s&matchIds=%s", basketballKey, sourceMatchId);
            ResponseEntity<String> response = restTemplate.getForEntity(requestUrl, String.class);
            if (response.getStatusCode() == HttpStatus.OK) {
                JSONObject jsonObject = JSONObject.parseObject(response.getBody());
                AlStatPlayerBasketballStatisticsDto alStatScheduleMatchDto = JSONObject.toJavaObject(jsonObject, AlStatPlayerBasketballStatisticsDto.class);
                if (alStatScheduleMatchDto.getState().equals("success")) {
                    for (AlStatPlayerBasketballStatisticsDto.AlStatScheduleMatchInfoDto matchInfoDto : alStatScheduleMatchDto.getMatchs()) {
                        for (AlStatPlayerBasketballStatisticsDto.AlStatStatsInfoDto playerInfo : matchInfoDto.getStats()) {
                            try {
                                String teamId = playerInfo.getHomeAway().equals("home") ? homeId : awayId;
                                PlayerCompetition playerCompetition = playerCompetitionMapper.selectOne(Wrappers.<PlayerCompetition>lambdaQuery()
                                        .eq(PlayerCompetition::getMatchId, match.getId())
                                        .eq(PlayerCompetition::getTeamId, teamId)
                                        .eq(PlayerCompetition::getPlayerName, playerInfo.getPlayerName())
                                );
                                if (playerCompetition == null) {
                                    playerCompetition = new PlayerCompetition();
                                }

                                playerCompetition.setTeamId(teamId);
                                playerCompetition.setMatchId(match.getId());
                                playerCompetition.setTeamCompetitionId(playerInfo.getHomeAway().equals("home") ? match.getCompetitionHomeId() : match.getCompetitionAwayId());
                                playerCompetition.setPosition(playerInfo.getPosition());
                                playerCompetition.setPlayerName(playerInfo.getPlayerName());
                                playerCompetition.setPlayingTime(playerInfo.getMinsPlayed());
                                playerCompetition.setAbsentReason(playerInfo.getIsFirstFive() == 1 ? "0" : "1");
                                playerCompetition.setAssists(playerInfo.getAssists());
                                playerCompetition.setAttackBoard(playerInfo.getOffRebounds());
                                playerCompetition.setBlocks(playerInfo.getBlocks());
                                playerCompetition.setError(playerInfo.getTurnovers());
                                playerCompetition.setFoul(playerInfo.getFouls());
                                playerCompetition.setFreeThrowNum(playerInfo.getFreeThrowsMade());
                                playerCompetition.setFreeThrows(playerInfo.getFreeThrowsAtt());
                                playerCompetition.setHomeOrAway(playerInfo.getHomeAway().equals("home") ? 0 : 1);
                                playerCompetition.setShootNum(playerInfo.getFieldGoalsMade());
                                playerCompetition.setSteals(playerInfo.getSteals());
                                playerCompetition.setScore(playerInfo.getPoints());
                                playerCompetition.setThreeShotNum(playerInfo.getThreePointsMade());
                                playerCompetition.setThreeShots(playerInfo.getThreePointsAtt());
                                playerCompetition.setTotalBoards(playerInfo.getRebounds());

                                if (StringUtils.isNotBlank(playerCompetition.getId())) {
                                    playerCompetitionMapper.updateById(playerCompetition);
                                } else {
                                    playerCompetition.setId(IdGen.uuid());
                                    playerCompetitionMapper.insert(playerCompetition);
                                }
                            } catch (Exception e) {
                                log.error(e.getMessage());
                            }
                        }
                    }
                }
            }
        } catch (RuntimeException e) {
            log.error("competitionPlayer ERROR:{}", e.getMessage());
        }
    }

    //团队比分
    private Match competitionTeam(Match match, Integer matchId, String homeTeamId, String awayTeamId) {
        String requestUrl = String.format("http://data71.aistat.cn/basketball/matchs/teamStats?key=%s&matchIds=%s", basketballKey, matchId);
        ResponseEntity<String> response = restTemplate.getForEntity(requestUrl, String.class);
        if (response.getStatusCode() == HttpStatus.OK) {
            JSONObject jsonObject = JSONObject.parseObject(response.getBody());
            AlStatTeamBasketballStatisticsDto alStatScheduleMatchDto = JSONObject.toJavaObject(jsonObject, AlStatTeamBasketballStatisticsDto.class);
            if (alStatScheduleMatchDto.getState().equals("success")) {
                for (AlStatTeamBasketballStatisticsDto.AlStatScheduleMatchInfoDto alStatScheduleMatch : alStatScheduleMatchDto.getMatchs()) {
                    for (AlStatTeamBasketballStatisticsDto.AlStatStatsInfoDto alStatStatsInfoDto : alStatScheduleMatch.getStats()) {
                        try {
                            String dbID = null;
                            if (match != null) {
                                if (alStatStatsInfoDto.getHomeAway().equals("away")) {
                                    dbID = match.getCompetitionAwayId();
                                } else {
                                    dbID = match.getCompetitionHomeId();
                                }
                            } else {
                                match = new Match();
                                match.setHomeId(homeTeamId);
                                match.setAwayId(awayTeamId);
                            }

                            TeamCompetition teamCompetition = null;
                            if (StringUtils.isNotBlank(dbID))
                                teamCompetition = teamCompetitionMapper.selectById(dbID);
                            if (teamCompetition == null) {
                                teamCompetition = new TeamCompetition();
                                teamCompetition.setScoreSettle("0,0,0,0,0,0");
                            }

                            teamCompetition.setAssists(alStatStatsInfoDto.getAssists());
                            teamCompetition.setAttackBoard(alStatStatsInfoDto.getOffRebounds());
                            teamCompetition.setBlocks(alStatStatsInfoDto.getBlocks());
                            teamCompetition.setError(alStatStatsInfoDto.getTurnovers());
                            teamCompetition.setDefensiveBoard(alStatStatsInfoDto.getDefRebounds());
                            teamCompetition.setFoul(alStatStatsInfoDto.getFouls());
                            teamCompetition.setFreeThrowNum(alStatStatsInfoDto.getFreeThrowsMade());
                            teamCompetition.setFreeThrows(alStatStatsInfoDto.getFreeThrowsAtt());
                            teamCompetition.setHomeOrAway(alStatStatsInfoDto.getHomeAway().equals("away") ? 1 : 0);
                            teamCompetition.setResult(alStatStatsInfoDto.getResult() != null && alStatStatsInfoDto.getResult() == 1 ? 0 : 1);
                            teamCompetition.setScore(alStatStatsInfoDto.getPoints());
                            teamCompetition.setTeamId(alStatStatsInfoDto.getHomeAway().equals("away") ? awayTeamId : homeTeamId);
                            teamCompetition.setThreeShotNum(alStatStatsInfoDto.getThreePointsMade());
                            teamCompetition.setThreeShots(alStatStatsInfoDto.getThreePointsAtt());
                            teamCompetition.setTotalBoards(alStatStatsInfoDto.getRebounds());
                            teamCompetition.setTwoScore(alStatStatsInfoDto.getTwoPointsAtt());
                            teamCompetition.setSteals(alStatStatsInfoDto.getSteals());
                            teamCompetition.setShots(alStatStatsInfoDto.getFieldGoalsAtt());
                            teamCompetition.setShootNum(alStatStatsInfoDto.getFieldGoalsMade());

                            if (StringUtils.isNotBlank(dbID)) {
                                teamCompetitionMapper.updateById(teamCompetition);
                            } else {
                                dbID = IdGen.uuid();
                                teamCompetition.setId(dbID);
                                teamCompetitionMapper.insert(teamCompetition);

                                if (alStatStatsInfoDto.getHomeAway().equals("away")) {
                                    match.setCompetitionAwayId(dbID);
                                } else {
                                    match.setCompetitionHomeId(dbID);
                                }
                            }
                        } catch (RuntimeException e) {
                            e.printStackTrace();
                            log.error("competitionTeam ERROR:{}", e.getMessage());
                        }
                    }
                }
            }
        }
        return match;
    }

    private String saveSports(String competitionShortName, int sportsType) throws RuntimeException {
        try {
            lock.lock();
            Sports sports = sportsMapper.selectOne(Wrappers.<Sports>lambdaQuery()
                    .eq(Sports::getDeleted, 0)
                    .eq(Sports::getCompetitionShortName, competitionShortName)
                    .eq(Sports::getSourceType, 1)
            );
            if (sports != null) {
                return sports.getId();
            }

            String id = IdGen.uuid();
            sportsMapper.insert(Sports.builder()
                    .id(id)
                    .sportsId(sportsType)
                    .sportsName(sportsType == 1 ? "篮球" : "足球")
                    .competitionShortName(competitionShortName)
                    .build());
            return id;
        } catch (RuntimeException e) {
            log.warn("新增 Sports 时报错:{}", e.getMessage());
        } finally {
            lock.unlock();
        }
        throw new RuntimeException();
    }

    private Team saveTeam(String teamName, Integer originalId) throws RuntimeException {
        lock.lock();
        try {
            Team teamDb = teamMapper.selectOne(Wrappers.<Team>lambdaQuery()
                    .eq(Team::getDeleted, 0)
                    .eq(Team::getTeamName, teamName)
                    .eq(Team::getOriginalType, 1)
            );
            if (teamDb != null) {
                return teamDb;
            }

            String id = IdGen.uuid();
            Team saveTeam = Team.builder()
                    .id(id)
                    .originalId("" + originalId)
                    .teamIcon("http://dt.aistat.cn/teams/" + originalId + ".png")
                    .teamName(teamName)
                    .originalType(1)
                    .build();
            teamMapper.insert(saveTeam);
            return saveTeam;
        } catch (DuplicateKeyException d) {
            //nothing
        } catch (RuntimeException e) {
            log.warn("新增球队时报错:{}", e.getMessage());
        } finally {
            lock.unlock();
        }
        throw new RuntimeException();
    }

    @Override
    public void crawlScheduleArticle(String Url) {
    }

    @Override
    public void updateMatchList() {

    }

    @Override
    public void updateFootballState(String matchId, long time) {

    }

    @Override
    public void updateBasketballState(String matchId, long time) {

    }

    @Override
    public String getPlatform() {
        return CrawlPlatformEnum.AlStat.getCode();
    }
}
