package com.live.common.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.live.common.domain.request.CreateGroupRequest;
import com.live.common.domain.request.HXRoomRequest;
import com.live.common.domain.request.SendMessageByRoomRequest;
import com.live.common.service.HuanXinIMService;
import lombok.Builder;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.*;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Slf4j
@Service
public class HuanXinIMServiceImpl implements HuanXinIMService {

    @Value("${huanXin_IM.Orgname}")
    private String org_name;
    @Value("${huanXin_IM.Appname}")
    private String app_name;
    @Value("${huanXin_IM.Client_ID}")
    private String client_id;
    @Value("${huanXin_IM.ClientSecret}")
    private String client_secret;
    @Value("${huanXin_IM.REST_API}")
    private String requestUrl;

    @Resource
    private RestTemplate restTemplate;

    private static String token = null;

    private String commonRequestUrl() {
        return requestUrl.concat("/").concat(org_name).concat("/").concat(app_name);
    }

    private String friendRequestUrl(String owner, String friend) {
        return commonRequestUrl().concat("/users/").concat(owner).concat("/").concat("contacts/users/").concat(friend);
    }

    private String tokenAuthenticate() {
        if (token == null) {
            token = getToken();
        }
        return "Bearer ".concat(token);
    }

    @Override
    public synchronized String getToken() {
        if (token != null) {
            return token;
        }

        TokenRequest tokenRequest = TokenRequest.builder()
                .client_id(client_id)
                .grant_type("client_credentials")
                .client_secret(client_secret)
                .build();

        HttpHeaders headers = new HttpHeaders();
        headers.add("org_name", org_name);
        headers.add("app_name", app_name);

        ResponseEntity<String> response = restTemplate.postForEntity(commonRequestUrl().concat("/token"), new HttpEntity<TokenRequest>(tokenRequest, headers), String.class);
        if (response.getStatusCode() == HttpStatus.OK) {
            JSONObject jsonObject = JSON.parseObject(response.getBody());
            token = jsonObject.getString("access_token");
        }
        return token;
    }

    @Override
    public void registerUser(UserRegistration user) {
        HttpHeaders headers = new HttpHeaders();
        headers.add("org_name", org_name);
        headers.add("app_name", app_name);
        headers.add("Authorization", tokenAuthenticate());

        Map<String, String> requestBody = new HashMap<>();
        requestBody.put("username", user.getUsername());
        requestBody.put("password", user.getPassword());
        requestBody.put("nickname", user.getNickName());

        ResponseEntity<String> response = restTemplate.postForEntity(commonRequestUrl().concat("/users"), new HttpEntity<>(requestBody, headers), String.class);
        log.info("带header的Post数据请求:" + response.getBody());
        if (response.getStatusCode() == HttpStatus.UNAUTHORIZED) {
            token = null;
            registerUser(user);
        }
    }

    @Override
    public String createGroup(CreateGroupRequest createGroupRequest, String userName) {
        HttpHeaders headers = new HttpHeaders();
        headers.add("org_name", org_name);
        headers.add("app_name", app_name);
        headers.add("Authorization", tokenAuthenticate());

        Map<String, Object> requestBody = new HashMap<>();
        requestBody.put("groupname", createGroupRequest.getGroupName());
        requestBody.put("desc", createGroupRequest.getDesc());
        requestBody.put("public", createGroupRequest.isPublic());
        requestBody.put("maxusers", 200);
        requestBody.put("approval", createGroupRequest.isApproval());
        requestBody.put("owner", userName);

        ResponseEntity<String> response = restTemplate.postForEntity(commonRequestUrl().concat("/chatgroups"),
                new HttpEntity<>(requestBody, headers), String.class);
        log.info("带header的Post数据请求:" + response.getBody());
        if (response.getStatusCode() == HttpStatus.UNAUTHORIZED) {
            token = null;
            createGroup(createGroupRequest, userName);
        } else if (response.getStatusCode() == HttpStatus.OK) {
            JSONObject jsonObject = JSONObject.parseObject(response.getBody());
            jsonObject.get("data");
            jsonObject = JSONObject.parseObject(jsonObject.get("data").toString());
            return jsonObject.get("groupid").toString();
        }
        return null;
    }

    @Override
    public void deleteGroup(String groupId) {
        HttpHeaders headers = new HttpHeaders();
        headers.add("org_name", org_name);
        headers.add("app_name", app_name);
        headers.add("Authorization", tokenAuthenticate());
        headers.add("group_id", groupId);

        ResponseEntity<String> response = restTemplate.exchange(commonRequestUrl().concat("/chatgroups/" + groupId), HttpMethod.DELETE, new HttpEntity<>(null, headers), String.class);
        log.info("带header的Post数据请求:" + response.getBody());
        if (response.getStatusCode() == HttpStatus.UNAUTHORIZED) {
            token = null;
            deleteGroup(groupId);
        }
    }

    @Override
    public List<GroupInfo> getGroupInfo(List<String> groupIds) {
        HttpHeaders headers = new HttpHeaders();
        headers.add("org_name", org_name);
        headers.add("app_name", app_name);
        headers.add("Authorization", tokenAuthenticate());
        headers.add("group_ids", String.join(",", groupIds));

        ResponseEntity<String> response = restTemplate.exchange(commonRequestUrl().concat("/chatgroups/" + String.join(",", groupIds)), HttpMethod.GET, new HttpEntity<>(null, headers), String.class);
        log.info("带header的Post数据请求:" + response.getBody());
        if (response.getStatusCode() == HttpStatus.UNAUTHORIZED) {
            token = null;
            getGroupInfo(groupIds);
        }

        GroupInfosResponse groupInfosResponse = JSONObject.toJavaObject(JSONObject.parseObject(response.getBody()), GroupInfosResponse.class);
        return groupInfosResponse.getData();
    }

    @Override
    public String createRoom(HXRoomRequest hxRoomRequest) {
        HttpHeaders headers = new HttpHeaders();
        headers.add("org_name", org_name);
        headers.add("app_name", app_name);
        headers.add("Authorization", tokenAuthenticate());


        Map<String, Object> requestBody = new HashMap<>();
        requestBody.put("name", hxRoomRequest.getName());
        requestBody.put("description", hxRoomRequest.getDescription());
        requestBody.put("owner", hxRoomRequest.getOwner());
        requestBody.put("maxusers", 5000);

        ResponseEntity<String> response = restTemplate.exchange(commonRequestUrl().concat("/chatrooms"), HttpMethod.POST,
                new HttpEntity<>(requestBody, headers), String.class);

        log.info("带header的Post数据请求:" + response.getBody());
        if (response.getStatusCode() == HttpStatus.UNAUTHORIZED) {
            token = null;
            createRoom(hxRoomRequest);
        } else {
            JSONObject jsonObject = JSONObject.parseObject(response.getBody());
            jsonObject.get("data");
            jsonObject = JSONObject.parseObject(jsonObject.get("data").toString());
            return jsonObject.get("id").toString();
        }
        return response.getBody();
    }

    @Override
    public void sendMessageByRoom(SendMessageByRoomRequest messageByRoomRequest) {
        HttpHeaders headers = new HttpHeaders();
        headers.add("org_name", org_name);
        headers.add("app_name", app_name);
        headers.add("Authorization", tokenAuthenticate());

        ResponseEntity<String> response = restTemplate.exchange(commonRequestUrl().concat("/message"), HttpMethod.POST,
                new HttpEntity<>(messageByRoomRequest, headers), String.class);

        log.info("带header的Post数据请求:" + response.getBody());
        if (response.getStatusCode() == HttpStatus.UNAUTHORIZED) {
            token = null;
            sendMessageByRoom(messageByRoomRequest);
        }
    }

    @Data
    public static class GroupInfosResponse {
        private List<GroupInfo> data;
    }

    @Data
    public static class GroupInfo {
        private String id;
        private String name;
    }

    @Data
    @Builder
    public static class TokenRequest {
        private String grant_type;
        private String client_id;
        private String client_secret;
    }

    @Data
    @Builder
    public static class UserRegistration {
        private String username;
        private String nickName;
        private String password;
    }

}
