package com.live.job.service.impl;

import com.baomidou.mybatisplus.core.toolkit.StringUtils;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.live.common.domain.entity.Match;
import com.live.common.domain.entity.Sports;
import com.live.common.mapper.MatchMapper;
import com.live.common.mapper.SportsMapper;
import com.live.job.cache.MatchManager;
import com.live.job.event.MatchOngoingEvent;
import com.live.job.service.CrawlStrategyFactory;
import com.live.job.service.UpdateMatchStateService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.event.EventListener;
import org.springframework.scheduling.concurrent.ThreadPoolTaskScheduler;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.time.Instant;
import java.util.List;

@Slf4j
@Service
public class UpdateMatchStateServiceImpl implements UpdateMatchStateService {

    private String platform = "LeYu";
    @Resource
    private MatchMapper matchMapper;
    @Resource
    private SportsMapper sportsMapper;
    @Resource
    private CrawlStrategyFactory crawStrategyFactory;
    @Resource
    private ThreadPoolTaskScheduler taskScheduler;
    //30分钟后更新比赛开始数据
    private long limitTime = 60 * 1000L;

    public void updateStateMatch() {
        List<Match> matches = matchMapper.selectList(Wrappers.<Match>lambdaQuery()
                .eq(Match::getDeleted, 0)
                .ne(Match::getCompetitionStatus, 3)
                .eq(Match::getSourceType, 0)
                .orderByDesc(Match::getMatchTime)
        );

        for (Match match : matches) {
            if(StringUtils.isBlank(match.getCrawlUrl()))
                continue;

            Sports sports = sportsMapper.selectById(match.getSportsId());
            matchOngoing(MatchOngoingEvent.builder()
                    .matchId(match.getId())
                    .time(System.currentTimeMillis())
                    .matchTime(match.getMatchTime())
                    .sportsType(sports.getSportsId())
                    .status(match.getCompetitionStatus())
                    .build());
        }
    }

    @EventListener(MatchOngoingEvent.class)
    public void matchOngoing(MatchOngoingEvent event) {
        try {
            switch (event.getSportsType()) {
                case 0:
                    //足球
                    if (event.getStatus() == 1) {
                        log.info("5分钟后更新足球 {} 游戏状态", event.getMatchId());
                        taskScheduler.schedule(() -> {
                            crawStrategyFactory.getCrawMatchService(platform).updateFootballState(event.getMatchId(), event.getTime());
                        }, Instant.now().plusSeconds(5 * 60));
                    } else if (event.getStatus() == 2) {
                        long interval = event.getMatchTime().getTime() - System.currentTimeMillis() + limitTime;
                        if (interval < 0) {
                            log.info("立即更新足球 {} 游戏状态", event.getMatchId());
                            taskScheduler.execute(() -> crawStrategyFactory.getCrawMatchService(platform)
                                    .updateFootballState(event.getMatchId(), event.getTime()));
                        } else {
                            log.info("{} s 后更新足球 {} 游戏状态", interval / 1000, event.getMatchId());
                            taskScheduler.schedule(() -> {
                                crawStrategyFactory.getCrawMatchService(platform).updateFootballState(event.getMatchId(), event.getTime());
                            }, Instant.now().plusMillis(interval));
                        }
                    }
                    break;
                case 1:
                    //篮球
                    if (event.getStatus() == 1) {
                        taskScheduler.schedule(() -> {
                            log.info("5分钟后更新篮球 {} 游戏状态", event.getMatchId());
                            crawStrategyFactory.getCrawMatchService("LeYu").updateBasketballState(event.getMatchId(), event.getTime());
                        }, Instant.now().plusSeconds(5 * 60));
                    } else if (event.getStatus() == 2) {
                        long interval = event.getMatchTime().getTime() - System.currentTimeMillis() + limitTime;
                        if (interval < 0) {
                            log.info("立即更新篮球 {} 游戏状态", event.getMatchId());
                            taskScheduler.execute(() -> crawStrategyFactory.getCrawMatchService("LeYu").updateBasketballState(event.getMatchId(), event.getTime()));
                        } else {
                            log.info("{} s 后更新篮球 {} 游戏状态", interval / 1000, event.getMatchId());
                            taskScheduler.schedule(() -> {
                                crawStrategyFactory.getCrawMatchService("LeYu").updateBasketballState(event.getMatchId(), event.getTime());
                            }, Instant.now().plusMillis(interval));
                        }
                    }
                    break;
            }
        } catch (Exception e) {
            log.warn("Add player count failed：{}", e.getMessage());
        }
    }
}
