package com.live.common.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.StringUtils;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.live.common.constant.ConstantValue;
import com.live.common.domain.ResponseData;
import com.live.common.domain.ResultPage;
import com.live.common.domain.dto.api.*;
import com.live.common.domain.dto.back.GetArticleListResponse;
import com.live.common.domain.dto.back.GetArticleTagListResponse;
import com.live.common.domain.entity.*;
import com.live.common.domain.request.*;
import com.live.common.domain.request.back.ArticleListRequest;
import com.live.common.domain.request.back.EditArticleInfoRequest;
import com.live.common.enums.StatusEnum;
import com.live.common.mapper.*;
import com.live.common.service.ArticleService;
import com.live.common.service.RedisUtilsService;
import com.live.common.utils.CommonMethod;
import com.live.common.utils.DateUtil;
import com.live.common.utils.RedisCommonChar;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.redis.connection.RedisServer;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.util.*;
import java.util.stream.Collectors;

@Slf4j
@Service
public class ArticleServiceImpl implements ArticleService {

    @Resource
    private TagMapper tagMapper;
    @Resource
    private TeamMapper teamMapper;
    @Resource
    private UserMapper userMapper;
    @Resource
    private SportsMapper sportsMapper;
    @Resource
    private ArticleMapper articleMapper;
    @Resource
    private RedisUtilsService redisUtilsService;
    @Resource
    private ArticleLikeMapper articleLikeMapper;

    @Override
    public ResponseData<?> getArticleTopTagList() {
        List<Tag> tagList = tagMapper.selectList(Wrappers.<Tag>lambdaQuery()
                .eq(Tag::getDeleted, 0)
                .eq(Tag::getSearchTop, 1)
                .orderByDesc(Tag::getSort)
        );

        List<ArticleTopTagListResponse> responses = new ArrayList<>();
        responses.add(ArticleTopTagListResponse.builder()
                .tagName("热门")
                .tagId(0)
                .build());

        tagList.forEach(b -> {
            responses.add(ArticleTopTagListResponse.builder()
                    .tagName(b.getTagName())
                    .tagId(b.getId())
                    .build());
        });
        return ResponseData.successResponse(responses);
    }

    @Override
    public ResponseData<?> getArticleType() {
        List<Tag> tagList = tagMapper.selectList(Wrappers.<Tag>lambdaQuery()
                .eq(Tag::getDeleted, 0)
                .orderByDesc(Tag::getSort)
        );

        List<ArticleTopTagListResponse> responses = tagList.stream().map(b -> ArticleTopTagListResponse.builder()
                .tagId(b.getId())
                .tagName(b.getTagName())
                .build()).collect(Collectors.toList());
        return ResponseData.successResponse(responses);
    }

    @Override
    public ResponseData<?> getTopArticleList(HttpServletRequest request) {
        List<Article> articles = articleMapper.selectList(Wrappers.<Article>lambdaQuery()
                .eq(Article::getDeleted, StatusEnum.ENABLE.getCode())
                .eq(Article::getContentType, 0)
                .eq(Article::getShowHome, 1)
        );

        return ResponseData.successResponse(articles.stream().map(b -> ArticleTopListResponse.builder()
                .coverPicture(ConstantValue.completeAddress(b.getPicUrl()))
                .id(b.getId())
                .title(b.getTitle())
                .build()).collect(Collectors.toList()));
    }

    @Override
    public ResponseData<?> getArticleListByTagId(ArticleByTagIdRequest tagIdRequest, HttpServletRequest request) {
        String userId = null;
        try {
            String phone = CommonMethod.getUserPhone(request);
            User account = userMapper.selectOne(Wrappers.<User>lambdaQuery()
                    .eq(User::getPhone, phone)
            );
            userId = account.getId();
        } catch (RuntimeException e) {
            log.trace("getArticleListByTagId 未携带 Token ");
        }

        int jumpNum = (tagIdRequest.getPageNum() - 1) * tagIdRequest.getPageSize();

        QueryWrapper<Article> wrapper = new QueryWrapper<>();
        wrapper.eq("deleted", StatusEnum.ENABLE.getCode());
        wrapper.eq("content_type", 0);
        if (tagIdRequest.getTagId() != 0) {
            wrapper.apply("FIND_IN_SET({0},tag_id)", tagIdRequest.getTagId());
        } else {
//            wrapper.eq("hot_news", 1);

        }

        long count = articleMapper.selectCount(wrapper);

        wrapper.orderByDesc("create_time");
        wrapper.last(String.format(" limit %s,%s", jumpNum, tagIdRequest.getPageSize()));
        List<Article> articles = articleMapper.selectList(wrapper);

        String finalUserId = userId;
        List<ArticleListResponse> articleList = articles.stream().map(b -> {
            String tagIdString = b.getTagId();
            List<Tag> tags = new ArrayList<>();
            if (StringUtils.isNotBlank(tagIdString)) {
                String[] tagIds = tagIdString.split(",");
                tags = tagMapper.selectBatchIds(Arrays.stream(tagIds)
                        .filter(StringUtils::isNotBlank)
                        .collect(Collectors.toList()));
            }
            long isLike = 0;
            if (finalUserId != null) {
                isLike = articleLikeMapper.selectCount(Wrappers.<ArticleLike>lambdaQuery()
                        .eq(ArticleLike::getDeleted, 0)
                        .eq(ArticleLike::getArticleId, b.getId())
                        .eq(ArticleLike::getUserId, finalUserId)
                );
            }
            long otherLike = articleLikeMapper.selectCount(Wrappers.<ArticleLike>lambdaQuery()
                    .eq(ArticleLike::getDeleted, 0)
                    .eq(ArticleLike::getArticleId, b.getId())
            );
            return ArticleListResponse.builder()
                    .id(b.getId())
                    .title(b.getTitle())
                    .coverPicture(b.getPicUrl())
                    .likeCount(b.getFakeLike() + (int) otherLike)
                    .liked(isLike > 0)
                    .date(DateUtil.format(b.getReleaseTime(), DateUtil.MD_))
                    .time(DateUtil.format(b.getReleaseTime(), DateUtil.HM_))
                    .tags(tags.stream().map(tag -> TagInfoResponse.builder()
                            .name(tag.getTagName())
                            .id(tag.getId())
                            .build()).collect(Collectors.toList()))
                    .build();
        }).collect(Collectors.toList());

        return ResponseData.successResponse(
                new ResultPage<>(tagIdRequest.getPageNum(), tagIdRequest.getPageSize(), (int) count, articleList));
    }

    @Override
    public synchronized ResponseData<?> getArticleInfoById(CommonStringId commonStringId, HttpServletRequest request) {
        Article article = articleMapper.selectById(commonStringId.getId());
        if (article == null) {
            return ResponseData.fail400Response("文章ID错误，未找到文章");
        }

        String userId = null;
        long isLike = 0;
        try {
            String phone = CommonMethod.getUserPhone(request);
            User account = userMapper.selectOne(Wrappers.<User>lambdaQuery()
                    .eq(User::getPhone, phone)
            );
            userId = account.getId();
            isLike = articleLikeMapper.selectCount(Wrappers.<ArticleLike>lambdaQuery()
                    .eq(ArticleLike::getDeleted, 0)
                    .eq(ArticleLike::getArticleId, commonStringId.getId())
                    .eq(ArticleLike::getUserId, userId)
            );
        } catch (RuntimeException e) {
            log.trace("getArticleListByTagId 未携带 Token ");
        }

        String tagIdString = article.getTagId();
        List<Tag> tags = new ArrayList<>();
        List<ArticleTopListResponse> articleTopList = new ArrayList<>();
        Set<Integer> tagIdUnique = new HashSet<>();
        tagIdUnique.add(article.getId());
        if (StringUtils.isNotBlank(tagIdString)) {
            String[] tagIds = tagIdString.split(",");
            tags = tagMapper.selectBatchIds(Arrays.stream(tagIds)
                    .filter(StringUtils::isNotBlank)
                    .collect(Collectors.toList()));

            for (Tag tag : tags) {
                List<Article> articles = articleMapper.selectList(Wrappers.<Article>lambdaQuery()
                        .eq(Article::getDeleted, 0)
                        .eq(Article::getContentType, 0)
                        .apply("FIND_IN_SET({0},tag_id)", tag.getId())
                        .orderByDesc(Article::getReleaseTime)
                        .last(String.format(" limit %s,%s", 0, 3))
                );

                for (Article article1 : articles) {
                    if (!tagIdUnique.contains(article1.getId())) {
                        tagIdUnique.add(article1.getId());
                        articleTopList.add(ArticleTopListResponse.builder()
                                .id(article1.getId())
                                .title(article1.getTitle())
                                .coverPicture(article1.getPicUrl())
                                .build());
                    }
                }
            }
        }

        ArticleInfoResponse articleInfo = ArticleInfoResponse.builder()
                .id(article.getId())
                .authorAvatar("")
                .authorId(article.getAuthorId())
                .authorName(article.getAuthorName())
                .content(article.getContent())
                .coverPicture(article.getPicUrl())
                .likeCount(article.getFakeLike() + (int) isLike)
                .liked(isLike > 0)
                .releaseTime(DateUtil.format(article.getReleaseTime(), DateUtil.YMDHMS_))
                .tags(tags.stream().map(b -> TagInfoResponse.builder()
                        .id(b.getId())
                        .name(b.getTagName())
                        .build()).collect(Collectors.toList()))
                .title(article.getTitle())
                .associateArticle(articleTopList)
                .readNum(article.getReadNum())
                .build();

        if (StringUtils.isNotBlank(article.getAuthorId())) {
            User author = userMapper.selectById(article.getAuthorId());
            if (author != null) {
                articleInfo.setAuthorAvatar(ConstantValue.completeAddress(author.getUserIcon()));
            }
        }

        article.setReadNum(article.getReadNum() + 1);
        articleMapper.updateById(article);
        return ResponseData.successResponse(articleInfo);
    }

    @Override
    public ResponseData<?> likeArticle(CommonIntId commonIntId, HttpServletRequest request) {
        Article article = articleMapper.selectById(commonIntId.getId());
        if (article == null) {
            return ResponseData.fail400Response("文章ID错误，未找到文章");
        }
        try {
            String phone = CommonMethod.getUserPhone(request);
            User account = userMapper.selectOne(Wrappers.<User>lambdaQuery()
                    .eq(User::getPhone, phone)
            );

            String key = RedisCommonChar.REDIS_LOCK + "likeArticle:" + phone;
            if (redisUtilsService.existsKey(key)) {
                return ResponseData.fail400Response("您操作太频繁啦~");
            }

            redisUtilsService.cacheStringInfo(key, "1", 2);
            ArticleLike articleLike = articleLikeMapper.selectOne(Wrappers.<ArticleLike>lambdaQuery()
                    .eq(ArticleLike::getDeleted, 0)
                    .eq(ArticleLike::getArticleId, commonIntId.getId())
                    .eq(ArticleLike::getUserId, account.getId())
            );
            if (articleLike != null) {
                articleLikeMapper.deleteById(articleLike.getId());
            } else {
                articleLikeMapper.insert(ArticleLike.builder()
                        .articleId(commonIntId.getId())
                        .userId(account.getId())
                        .build());
            }
            return ResponseData.successResponse();
        } catch (RuntimeException e) {
            return ResponseData.fail500Response("请先登录~~");
        }
    }

    @Override
    public ResponseData<?> getArticleListByTitleLike(ArticleByTitleLikeRequest titleLikeRequest, HttpServletRequest request) {
        String userId = null;
        try {
            String phone = CommonMethod.getUserPhone(request);
            User account = userMapper.selectOne(Wrappers.<User>lambdaQuery()
                    .eq(User::getPhone, phone)
            );
            userId = account.getId();
        } catch (RuntimeException e) {
            log.trace("getArticleListByTagId 未携带 Token ");
        }

        int jumpNum = (titleLikeRequest.getPageNum() - 1) * titleLikeRequest.getPageSize();

        QueryWrapper<Article> wrapper = new QueryWrapper<>();
        wrapper.eq("deleted", StatusEnum.ENABLE.getCode());
        wrapper.eq("content_type", 0);

        StringBuilder sb = new StringBuilder();
        for (String like : titleLikeRequest.getTitleLikes()) {
            if (StringUtils.isBlank(sb)) {
                sb.append(" (title like '%").append(like).append("%'");
                continue;
            }
            sb.append(" or title like '%").append(like).append("%'");
        }
        if (!StringUtils.isBlank(sb))
            sb.append(")");

        wrapper.apply(sb.toString());
        long count = articleMapper.selectCount(wrapper);

        wrapper.orderByDesc("create_time");
        wrapper.last(String.format(" limit %s,%s", jumpNum, titleLikeRequest.getPageSize()));
        List<Article> articles = articleMapper.selectList(wrapper);

        String finalUserId = userId;
        List<ArticleListResponse> articleList = articles.stream().map(b -> {
            String tagIdString = b.getTagId();
            List<Tag> tags = new ArrayList<>();
            if (StringUtils.isNotBlank(tagIdString)) {
                String[] tagIds = tagIdString.split(",");
                tags = tagMapper.selectBatchIds(Arrays.stream(tagIds)
                        .filter(StringUtils::isNotBlank)
                        .collect(Collectors.toList()));
            }
            long isLike = 0;
            if (finalUserId != null) {
                isLike = articleLikeMapper.selectCount(Wrappers.<ArticleLike>lambdaQuery()
                        .eq(ArticleLike::getDeleted, 0)
                        .eq(ArticleLike::getArticleId, b.getId())
                        .eq(ArticleLike::getUserId, finalUserId)
                );
            }
            long otherLike = articleLikeMapper.selectCount(Wrappers.<ArticleLike>lambdaQuery()
                    .eq(ArticleLike::getDeleted, 0)
                    .eq(ArticleLike::getArticleId, b.getId())
            );
            return ArticleListResponse.builder()
                    .id(b.getId())
                    .title(b.getTitle())
                    .coverPicture(b.getPicUrl())
                    .likeCount(b.getFakeLike() + (int) otherLike)
                    .liked(isLike > 0)
                    .tags(tags.stream().map(tag -> TagInfoResponse.builder()
                            .name(tag.getTagName())
                            .id(tag.getId())
                            .build()).collect(Collectors.toList()))
                    .build();
        }).collect(Collectors.toList());
        return ResponseData.successResponse(
                new ResultPage<>(titleLikeRequest.getPageNum(), titleLikeRequest.getPageSize(), (int) count, articleList));
    }

    @Override
    public ResponseData<?> getArticleListByTeamId(GetArticleListByTeamIdRequest commonPage, HttpServletRequest request) {
        Team team = teamMapper.selectById(commonPage.getId());
        if(team == null)
            return ResponseData.fail400Response("Team id 错误");

        Sports sports = sportsMapper.selectById(team.getSportsDbId());

        int jumpNum = (commonPage.getPageNum() - 1) * commonPage.getPageSize();

        Tag sportsTag = tagMapper.selectOne(Wrappers.<Tag>lambdaQuery()
                .eq(Tag::getTagName, sports.getCompetitionShortName())
                .eq(Tag::getTagType, 1)
                .eq(Tag::getDeleted, 0)
        );
        Tag teamTag = tagMapper.selectOne(Wrappers.<Tag>lambdaQuery()
                .eq(Tag::getTagName, team.getTeamName())
                .eq(Tag::getTagType, 1)
                .eq(Tag::getDeleted, 0)
        );

        QueryWrapper<Article> wrapper = new QueryWrapper<>();
        wrapper.eq("deleted", StatusEnum.ENABLE.getCode());
        wrapper.eq("content_type", 0);

        if(sportsTag != null && teamTag != null){
            wrapper.apply("FIND_IN_SET({0},tag_id)", sportsTag.getId());
            wrapper.or();
            wrapper.apply("FIND_IN_SET({0},tag_id)", teamTag.getId());
        } else if(sportsTag != null){
            wrapper.apply("FIND_IN_SET({0},tag_id)", sportsTag.getId());
        } else if(teamTag != null){
            wrapper.apply("FIND_IN_SET({0},tag_id)", teamTag.getId());
        } else {
            return ResponseData.successResponse(new ResultPage<>(commonPage.getPageNum(), commonPage.getPageSize(), (int) 0,  new ArrayList()));
        }

        long count = articleMapper.selectCount(wrapper);
        wrapper.orderByDesc("release_time");
        wrapper.last(String.format(" limit %s,%s", jumpNum, commonPage.getPageSize()));

        List<Article> articles = articleMapper.selectList(wrapper);
        List<GetArticleListResponse> responses = changeDto(articles);
        return ResponseData.successResponse(new ResultPage<>(commonPage.getPageNum(), commonPage.getPageSize(), (int) count, responses));
    }

    @Override
    public ResponseData<?> getArticleList(ArticleListRequest commonPage, HttpServletRequest request) {
        int jumpNum = (commonPage.getPageNum() - 1) * commonPage.getPageSize();

        QueryWrapper<Article> wrapper = new QueryWrapper<>();
        wrapper.eq("deleted", StatusEnum.ENABLE.getCode());

        long count = articleMapper.selectCount(wrapper);

        wrapper.orderByDesc("create_time");
        wrapper.last(String.format(" limit %s,%s", jumpNum, commonPage.getPageSize()));
        List<Article> quizRecords = articleMapper.selectList(wrapper);

        List<GetArticleListResponse> responses = changeDto(quizRecords);
        return ResponseData.successResponse(
                new ResultPage<>(commonPage.getPageNum(), commonPage.getPageSize(), (int) count, responses));
    }

    private List<GetArticleListResponse> changeDto(List<Article> quizRecords){
        List<GetArticleListResponse> responses = new LinkedList<>();
        for (Article article : quizRecords) {
            long isLike = articleLikeMapper.selectCount(Wrappers.<ArticleLike>lambdaQuery()
                    .eq(ArticleLike::getDeleted, 0)
                    .eq(ArticleLike::getArticleId, article.getId())
            );
            String tagIdString = article.getTagId();
            List<Tag> tags = new ArrayList<>();
            if (StringUtils.isNotBlank(tagIdString)) {
                String[] tagIds = tagIdString.split(",");
                tags = tagMapper.selectBatchIds(Arrays.stream(tagIds)
                        .filter(StringUtils::isNotBlank)
                        .collect(Collectors.toList()));
            }

            responses.add(GetArticleListResponse.builder()
                    .coverPicture(ConstantValue.completeAddress(article.getPicUrl()))
                    .id(article.getId())
                    .fakeLike(article.getFakeLike())
                    .likeSum(article.getFakeLike() + (int) isLike)
                    .title(article.getTitle())
                    .content(article.getContent())
                    .tags(tags.stream().map(b -> TagInfoResponse.builder()
                            .id(b.getId())
                            .name(b.getTagName())
                            .build()).collect(Collectors.toList()))
                    .deleted(article.getDeleted())
                    .readNum(article.getReadNum())
                    .build());
        }
        return responses;
    }

    @Override
    public ResponseData<?> editArticleInfo(EditArticleInfoRequest editArticleInfo, HttpServletRequest request) {
        Article article = articleMapper.selectById(editArticleInfo.getId());
        if (article == null)
            article = new Article();

        article.setReadNum(editArticleInfo.getReadNum());
        article.setContent(editArticleInfo.getContent());
        article.setPicUrl(editArticleInfo.getCoverPicture());
        article.setDeleted(editArticleInfo.getDeleted());
        article.setFakeLike(editArticleInfo.getFakeLike());
        article.setTitle(editArticleInfo.getTitle());
        article.setTagId(editArticleInfo.getTagIds().stream().map(String::valueOf).collect(Collectors.joining(",")));

        if (article.getId() == null || article.getId() == 0) {
            article.setId(null);
            articleMapper.insert(article);
        } else {
            articleMapper.updateById(article);
        }
        return ResponseData.successResponse();
    }

    @Override
    public ResponseData<?> getArticleTagList(CommonPage commonPage, HttpServletRequest request) {
        int jumpNum = (commonPage.getPageNum() - 1) * commonPage.getPageSize();

        long count = tagMapper.selectCount(Wrappers.<Tag>lambdaQuery()
                .eq(Tag::getDeleted, 0)
        );

        List<Tag> tags = tagMapper.selectList(Wrappers.<Tag>lambdaQuery()
                .eq(Tag::getDeleted, 0)
                .orderByDesc(Tag::getSort)
                .last(String.format(" limit %s,%s", jumpNum, commonPage.getPageSize()))
        );

        List<GetArticleTagListResponse> responses = tags.stream().map(b -> GetArticleTagListResponse.builder()
                .id(b.getId())
                .searchTop(b.getSearchTop())
                .sort(b.getSort())
                .tagName(b.getTagName())
                .tagType(b.getTagType())
                .build())
                .collect(Collectors.toList());
        return ResponseData.successResponse(
                new ResultPage<>(commonPage.getPageNum(), commonPage.getPageSize(), (int) count, responses));
    }

}
