package com.live.common.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.live.common.constant.ConstantValue;
import com.live.common.constant.RedisKeySplicing;
import com.live.common.domain.ResponseData;
import com.live.common.domain.ResultPage;
import com.live.common.domain.dto.api.*;
import com.live.common.domain.dto.back.MatchInfoBackResponse;
import com.live.common.domain.dto.back.MatchLiveListResponse;
import com.live.common.domain.dto.back.QuizRecordBackResponse;
import com.live.common.domain.dto.back.SportsResponse;
import com.live.common.domain.entity.*;
import com.live.common.domain.request.*;
import com.live.common.domain.request.back.*;
import com.live.common.enums.StatusEnum;
import com.live.common.mapper.*;
import com.live.common.service.ArticleService;
import com.live.common.service.MatchService;
import com.live.common.service.RedisUtilsService;
import com.live.common.utils.CommonMethod;
import com.live.common.utils.DateUtil;
import com.live.common.utils.IdGen;
import com.live.common.utils.MD5Util;
import com.live.common.utils.match.CommonUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.dao.DuplicateKeyException;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;

@Slf4j
@Service
public class MatchServiceImpl implements MatchService {

    @Resource
    private RoomMapper roomMapper;
    @Resource
    private UserMapper userMapper;
    @Resource
    private TeamMapper teamMapper;
    @Resource
    private MatchMapper matchMapper;
    @Resource
    private FollowMapper followMapper;
    @Resource
    private SportsMapper sportsMapper;
    @Resource
    private ArticleService articleService;
    @Resource
    private MatchOddsMapper matchOddsMapper;
    @Resource
    private MatchLiveMapper matchLiveMapper;
    @Resource
    private QuizRecordMapper quizRecordMapper;
    @Resource
    private TeamReportMapper teamReportMapper;
    @Resource
    private RedisUtilsService redisUtilsService;
    @Resource
    private UserReserveMapper userReserveMapper;
    @Resource
    private MatchRecentMapper matchRecentMapper;
    @Resource
    private IndexCompanyMapper indexCompanyMapper;
    @Resource
    private MatchSummaryMapper matchSummaryMapper;
    @Resource
    private TeamIntegralMapper teamIntegralMapper;
    @Resource
    private FootballStatMapper footballStatMapper;
    @Resource
    private FootballEventMapper footballEventMapper;
    @Resource
    private RoomHostRecordMapper roomHostRecordMapper;
    @Resource
    private TeamCompetitionMapper teamCompetitionMapper;
    @Resource
    private PlayerCompetitionMapper playerCompetitionMapper;

    private String[] header = new String[7];
    private String[] headerFootball = new String[7];

    public MatchServiceImpl() {
        header[0] = "全场";
        header[1] = "赛";
        header[2] = "胜";
        header[3] = "负";
        header[4] = "主场";
        header[5] = "客场";
        header[6] = "胜率";

        headerFootball[0] = "全场";
        headerFootball[1] = "赛";
        headerFootball[2] = "胜";
        headerFootball[3] = "平";
        headerFootball[4] = "负";
        headerFootball[5] = "积分";
        headerFootball[6] = "胜率";
    }

    @Override
    public ResponseData<?> getMatchListBySportsId(GetMatchListByRequest commonStringId, HttpServletRequest request) {
        List<Match> matches;

        QueryWrapper<Match> wrapper = new QueryWrapper<>();
        wrapper.eq("deleted", StatusEnum.ENABLE.getCode());
        wrapper.eq("source_type", 1);

        if (StringUtils.isNotBlank(commonStringId.getSportsId())) {
            wrapper.eq("sports_id", commonStringId.getSportsId());
        }
        if (StringUtils.isNotBlank(commonStringId.getMatchTime())) {
            wrapper.ge("match_time", commonStringId.getMatchTime() + " 00:00:00");
            wrapper.le("match_time", commonStringId.getSportsId() + " 23:59:59");
        }

        //-1 全部 0 展示正在进行 和 未开始 1 正在进行 2未开始 3完场
        //1正在进行 2未开始 3完场 4 未知
        Integer status = commonStringId.getStatus() == null ? 0 : commonStringId.getStatus();
        switch (status) {
            case -1:
                wrapper.notIn("competition_status", 4);
                wrapper.orderByDesc("match_time");
                wrapper.last(String.format(" limit %s,%s", 0, 30));

                matches = matchMapper.selectList(wrapper);
                break;
            case 1:
            case 2:
            case 3:
                wrapper.eq("competition_status", status);
                wrapper.orderByDesc("match_time");
                wrapper.last(String.format(" limit %s,%s", 0, 30));

                matches = matchMapper.selectList(wrapper);
                break;
            default:
                wrapper.in("competition_status", 1, 2);
                wrapper.orderByDesc("match_time");
                wrapper.last(String.format(" limit %s,%s", 0, 30));

                matches = matchMapper.selectList(wrapper);
                break;
        }

        MatchHomeListByDayResponse response = new MatchHomeListByDayResponse();
        for (int i = matches.size() - 1; i >= 0; i--) {
            if (i == matches.size()) {
                response.setMatchDay(DateUtil.format(matches.get(0).getMatchTime(), DateUtil.YMD_));
            }
            Match match = matches.get(i);
            MatchHomeListByDayResponse.MatchInfo matchInfo = new MatchHomeListByDayResponse.MatchInfo();
            matchInfo.setSubscribed(false);

            if (DateUtil.format(match.getMatchTime(), DateUtil.YMD_).equals(response.getMatchDay())) {
                try {
                    String phone = CommonMethod.getUserPhone(request);
                    User account = userMapper.selectOne(Wrappers.<User>lambdaQuery()
                            .eq(User::getPhone, phone)
                    );
                    if (account != null) {
                        UserReserve userReserves = userReserveMapper.selectOne(Wrappers.<UserReserve>lambdaQuery()
                                .eq(UserReserve::getUserId, account.getId())
                                .eq(UserReserve::getMatchId, match.getId())
                                .eq(UserReserve::getDeleted, 0)
                        );
                        if (userReserves != null)
                            matchInfo.setSubscribed(true);
                    }
                } catch (Exception e) {
                    log.warn("未携带 Token");
                }

                Team home = teamMapper.selectById(match.getHomeId());
                Team away = teamMapper.selectById(match.getAwayId());
                Sports sports = sportsMapper.selectById(match.getSportsId());

                matchInfo.setMatchShortName(match.getCompetitionShortName());
                matchInfo.setAway(MatchListInfo.builder()
                        .teamName(away.getTeamName())
                        .teamLogo(away.getTeamIcon())
                        .score(match.getAwayScore())
                        .build());
                matchInfo.setHome(MatchListInfo.builder()
                        .teamName(home.getTeamName())
                        .teamLogo(home.getTeamIcon())
                        .score(match.getHomeScore())
                        .build());
                matchInfo.setMatchTime(DateUtil.format(match.getMatchTime(), DateUtil.HM_));
                matchInfo.setSportsType(sports.getSportsId());

                response.getMatchInfos().add(matchInfo);
            }
        }

        return ResponseData.successResponse(response);
    }

    @Override
    public ResponseData<?> getMatchTypeInfoList(HttpServletRequest request) {
        String key = RedisKeySplicing.requestLock("getMatchTypeInfoList", "0");

        if(redisUtilsService.existsKey(key)){
            String cacheInfo = redisUtilsService.getCacheStringInfo(key);
            GetMatchTypeInfoListResponse response = JSONObject.toJavaObject(JSONObject.parseObject(cacheInfo), GetMatchTypeInfoListResponse.class);
            return ResponseData.successResponse(response);
        }

        List<Sports> sports = sportsMapper.selectList(Wrappers.<Sports>lambdaQuery()
                .eq(Sports::getDeleted, 0)
                .eq(Sports::getSourceType, 1)
        );

        GetMatchTypeInfoListResponse response = new GetMatchTypeInfoListResponse();
        for (Sports s : sports) {
            GetMatchTypeInfoListResponse.MatchTypeInfoResponse matchTypeInfo = GetMatchTypeInfoListResponse.MatchTypeInfoResponse.builder()
                    .icon(s.getCompetitionIcon())
                    .id(s.getId())
                    .shortName(s.getCompetitionShortName())
                    .shortNameEn(s.getCompetitionShortNameEn())
                    .build();
            if (s.getSportsId() == 0) {
                response.getFootball().add(matchTypeInfo);
            } else if (s.getSportsId() == 1) {
                response.getBasketball().add(matchTypeInfo);
            }
        }
        redisUtilsService.cacheStringInfoByDay(key, JSON.toJSONString(response), 7);
        return ResponseData.successResponse(response);
    }

    @Override
    public ResponseData<?> getTeamInfoList(HttpServletRequest request) {
        String key = RedisKeySplicing.requestLock("getTeamInfoList", "0");

        if(redisUtilsService.existsKey(key)){
            String cacheInfo = redisUtilsService.getCacheStringInfo(key);
            GetTeamInfoListResponse response = JSONObject.toJavaObject(JSONObject.parseObject(cacheInfo), GetTeamInfoListResponse.class);
            return ResponseData.successResponse(response);
        }

        List<Team> teams = teamMapper.selectList(Wrappers.<Team>lambdaQuery()
                .eq(Team::getDeleted, 0)
                .eq(Team::getOriginalType, 1)
                .orderByDesc(Team::getCreateTime)
        );

        GetTeamInfoListResponse response = new GetTeamInfoListResponse();
        for (Team team : teams) {
            GetTeamInfoListResponse.GetTeamInfoInfoResponse teamInfo = GetTeamInfoListResponse.GetTeamInfoInfoResponse.builder()
                    .id(team.getId())
                    .teamIcon(team.getTeamIcon())
                    .teamName(team.getTeamName())
                    .teamNameEn(team.getTeamNameEn())
                    .build();

            if (team.getHotTeam() == 1) {
                response.getHotTeam().add(teamInfo);
            } else if (team.getHotTeam() == 0) {
                response.getOtherTeam().add(teamInfo);
            }
        }
        redisUtilsService.cacheStringInfoByDay(key, JSON.toJSONString(response), 7);
        return ResponseData.successResponse(response);
    }

    @Override
    public ResponseData<?> getMatchList(GetMatchListRequest commonStringId, HttpServletRequest request) {
        List<Match> matches;

        QueryWrapper<Match> wrapper = new QueryWrapper<>();
        wrapper.eq("deleted", StatusEnum.ENABLE.getCode());
        wrapper.eq("source_type", 1);
        wrapper.ge("match_time", DateUtil.getStartTime());

        Integer status = commonStringId.getStatus() == null ? 0 : commonStringId.getStatus();
        switch (status) {
            case -1:
                wrapper.notIn("competition_status", 4);
                if (StringUtils.isNotBlank(commonStringId.getId())) {
                    if (commonStringId.getId().length() < 10) {
                        wrapper.apply("FIND_IN_SET({0},category_id)", commonStringId.getId());
                    } else {
                        wrapper.eq("sports_id", commonStringId.getId());
                    }
                }
                wrapper.orderByAsc("match_time");
                matches = matchMapper.selectList(wrapper);
                break;
            case 1:
            case 2:
            case 3:
                wrapper.eq("competition_status", status);
                if (StringUtils.isNotBlank(commonStringId.getId())) {
                    if (commonStringId.getId().length() < 10) {
                        wrapper.apply("FIND_IN_SET({0},category_id)", commonStringId.getId());
                    } else {
                        wrapper.eq("sports_id", commonStringId.getId());
                    }
                }

                wrapper.orderByAsc("match_time");
                matches = matchMapper.selectList(wrapper);
                break;
            default:
                wrapper.in("competition_status", 1, 2);
                if (StringUtils.isNotBlank(commonStringId.getId())) {
                    if (commonStringId.getId().length() < 10) {
                        wrapper.apply("FIND_IN_SET({0},category_id)", commonStringId.getId());
                    } else {
                        wrapper.eq("sports_id", commonStringId.getId());
                    }
                }

                wrapper.orderByAsc("match_time");
                matches = matchMapper.selectList(wrapper);
                break;
        }

        MatchListResponse matchListResponse = conversionByList(matches, true);

        try {
            String phone = CommonMethod.getUserPhone(request);
            User account = userMapper.selectOne(Wrappers.<User>lambdaQuery()
                    .eq(User::getPhone, phone)
            );

            List<UserReserve> userReserves = userReserveMapper.selectList(Wrappers.<UserReserve>lambdaQuery()
                    .eq(UserReserve::getUserId, account.getId())
                    .eq(UserReserve::getDeleted, 0)
            );
            Set<String> reservesId = userReserves.stream().map(UserReserve::getMatchId).collect(Collectors.toSet());

            matchListResponse.getMatchLists().forEach(match -> {
                match.getMatchInfos().forEach(matchInfo -> {
                    if (reservesId.contains(matchInfo.getId())) {
                        matchInfo.setSubscribe(true);
                    }
                });
            });
        } catch (RuntimeException e) {
            log.info("未携带 token !");
        }
        return ResponseData.successResponse(matchListResponse);
    }

    @Override
    public ResponseData<?> getMatchInfo(CommonStringId commonStringId, HttpServletRequest request) {
        if (StringUtils.isBlank(commonStringId.getId())) {
            return ResponseData.fail400Response("赛事 ID 为 null");
        }

        Match match = matchMapper.selectById(commonStringId.getId());
        if (match == null) {
            return ResponseData.fail400Response("赛事 ID 错误");
        }

        Sports sports = sportsMapper.selectById(match.getSportsId());
        Team awayTeam = teamMapper.selectById(match.getAwayId());
        Team homeTeam = teamMapper.selectById(match.getHomeId());

        return ResponseData.successResponse(conversionMatchInfo(match, sports.getSportsId(), awayTeam, homeTeam, true));
    }

    @Override
    public ResponseData<?> userReserveMatch(CommonStringId commonStringId, HttpServletRequest request) {
        if (StringUtils.isBlank(commonStringId.getId())) {
            return ResponseData.fail400Response("赛事 ID 为 null");
        }

        Match match = matchMapper.selectById(commonStringId.getId());
        if (match == null) {
            return ResponseData.fail400Response("赛事 ID 错误");
        }

        String userPhone = CommonMethod.getUserPhone(request);
        User account = userMapper.selectOne(Wrappers.<User>lambdaQuery()
                .eq(User::getPhone, userPhone)
        );

        try {
            userReserveMapper.insert(UserReserve.builder()
                    .matchId(commonStringId.getId())
                    .userId(account.getId())
                    .build());
        } catch (DuplicateKeyException e) {
            //nothing
        }
        return ResponseData.successResponse();
    }

    @Override
    public ResponseData<?> userCancelReserveMatch(CommonStringId commonStringId, HttpServletRequest request) {
        if (StringUtils.isBlank(commonStringId.getId())) {
            return ResponseData.fail400Response("赛事 ID 为 null");
        }

        Match match = matchMapper.selectById(commonStringId.getId());
        if (match == null) {
            return ResponseData.fail400Response("赛事 ID 错误");
        }

        String userPhone = CommonMethod.getUserPhone(request);
        User account = userMapper.selectOne(Wrappers.<User>lambdaQuery()
                .eq(User::getPhone, userPhone)
        );

        userReserveMapper.delete(Wrappers.<UserReserve>lambdaQuery()
                .eq(UserReserve::getMatchId, commonStringId.getId())
                .eq(UserReserve::getUserId, account.getId())
        );
        return ResponseData.successResponse();
    }

    @Override
    public ResponseData<?> userGetReserveMatch(CommonPage commonPage, HttpServletRequest request) {
        int jumpNum = (commonPage.getPageNum() - 1) * commonPage.getPageSize();

        String userPhone = CommonMethod.getUserPhone(request);
        User account = userMapper.selectOne(Wrappers.<User>lambdaQuery()
                .eq(User::getPhone, userPhone)
        );

        long count = userReserveMapper.selectCount(Wrappers.<UserReserve>lambdaQuery()
                .eq(UserReserve::getUserId, account.getId())
                .eq(UserReserve::getDeleted, 0)
        );

        List<UserReserve> userReserves = userReserveMapper.selectList(Wrappers.<UserReserve>lambdaQuery()
                .eq(UserReserve::getUserId, account.getId())
                .eq(UserReserve::getDeleted, 0)
                .last(String.format(" limit %s,%s", jumpNum, commonPage.getPageSize()))
        );

        List<Match> matches = new ArrayList<>();
        if (userReserves.size() > 0) {
            matches = matchMapper.selectList(Wrappers.<Match>lambdaQuery()
                    .in(Match::getId, userReserves.stream().map(UserReserve::getMatchId).collect(Collectors.toSet()))
                    .eq(Match::getDeleted, 0)
                    .orderByDesc(Match::getMatchTime)
            );
        }

        MatchHomeListByDayResponse response = new MatchHomeListByDayResponse();
        for (Match match : matches) {
            MatchHomeListByDayResponse.MatchInfo matchInfo = new MatchHomeListByDayResponse.MatchInfo();
            matchInfo.setSubscribed(true);

            Team home = teamMapper.selectById(match.getHomeId());
            Team away = teamMapper.selectById(match.getAwayId());
            Sports sports = sportsMapper.selectById(match.getSportsId());

            matchInfo.setMatchShortName(match.getCompetitionShortName());
            matchInfo.setAway(MatchListInfo.builder()
                    .teamName(away.getTeamName())
                    .teamLogo(away.getTeamIcon())
                    .score(match.getAwayScore())
                    .build());
            matchInfo.setHome(MatchListInfo.builder()
                    .teamName(home.getTeamName())
                    .teamLogo(home.getTeamIcon())
                    .score(match.getHomeScore())
                    .build());
            matchInfo.setMatchTime(DateUtil.format(match.getMatchTime(), DateUtil.HM_));
            matchInfo.setSportsType(sports.getSportsId());

            response.getMatchInfos().add(matchInfo);
        }

        return ResponseData.successResponse(new ResultPage<>(commonPage.getPageNum(), commonPage.getPageSize(), (int) count, response));
    }

    @Override
    public ResponseData<?> userGetAnchorList(HttpServletRequest request) {
        String userPhone = CommonMethod.getUserPhone(request);
        User account = userMapper.selectOne(Wrappers.<User>lambdaQuery()
                .eq(User::getPhone, userPhone)
        );

        List<Follow> follows = followMapper.selectList(Wrappers.<Follow>lambdaQuery()
                .eq(Follow::getUserId, account.getId())
                .eq(Follow::getDeleted, 0)
        );

        List<UserGetAnchorResponse> userGetAnchor = new ArrayList<>();
        if (follows != null && follows.size() > 0) {
            List<Room> rooms = roomMapper.selectList(Wrappers.<Room>lambdaQuery()
                    .in(Room::getUserId, follows.stream().map(Follow::getFollowId).collect(Collectors.toList()))
                    .eq(Room::getDeleted, 0)
            );

            userGetAnchor = rooms.stream().map(room -> {
                User user = userMapper.selectById(room.getUserId());

                return UserGetAnchorResponse.builder()
                        .liveStatus(room.getType())
                        .roomId(room.getHuanXinRoomId())
                        .roomName(room.getRoomName())
                        .anchorAvatar(ConstantValue.completeAddress(user.getUserIcon()))
                        .anchorName(user.getUserName())
                        .roomAnnouncement(room.getAnnouncement())
                        .matchCutImg(ConstantValue.completeAddress(room.getRoomCover()))
                        .build();
            }).collect(Collectors.toList());
        }
        return ResponseData.successResponse(userGetAnchor);
    }

    @Override
    public ResponseData<?> userFollowAnchor(CommonStringId commonIntId, HttpServletRequest request) {
        if (StringUtils.isBlank(commonIntId.getId())) {
            return ResponseData.fail400Response("房间 ID 为 null");
        }

        Room room = roomMapper.selectById(commonIntId.getId());
        if (room == null) {
            return ResponseData.fail400Response("房间错误");
        }

        String userPhone = CommonMethod.getUserPhone(request);
        User account = userMapper.selectOne(Wrappers.<User>lambdaQuery()
                .eq(User::getPhone, userPhone)
        );

        try {
            followMapper.insert(Follow.builder()
                    .userId(account.getId())
                    .followId(room.getUserId())
                    .build());
        } catch (DuplicateKeyException e) {
            //nothing
        }
        return ResponseData.successResponse();
    }

    @Override
    public ResponseData<?> userUnFollowAnchor(CommonStringId commonIntId, HttpServletRequest request) {
        if (StringUtils.isBlank(commonIntId.getId())) {
            return ResponseData.fail400Response("房间 ID 为 null");
        }

        Room room = roomMapper.selectById(commonIntId.getId());
        if (room == null) {
            return ResponseData.fail400Response("房间错误");
        }

        String userPhone = CommonMethod.getUserPhone(request);
        User account = userMapper.selectOne(Wrappers.<User>lambdaQuery()
                .eq(User::getPhone, userPhone)
        );

        followMapper.delete(Wrappers.<Follow>lambdaQuery()
                .eq(Follow::getFollowId, room.getUserId())
                .eq(Follow::getUserId, account.getId())
        );
        return ResponseData.successResponse();
    }

    @Override
    public MatchListResponse conversionByList(List<Match> matches, boolean otherAnchors) {
        String dataMd5 = MD5Util.string2MD5(JSONObject.toJSONString(matches));
        String key = RedisKeySplicing.cacheMatchList(dataMd5);
        if (redisUtilsService.existsKey(key)) {
            String cacheInfo = redisUtilsService.getCacheStringInfo(key);
            MatchListResponse response = JSONObject.toJavaObject(JSONObject.parseObject(cacheInfo), MatchListResponse.class);

            for (MatchListResponse.MatchResponse matchResponse : response.getMatchLists()) {
                for (MatchListResponse.MatchInfoResponse infoResponse : matchResponse.getMatchInfos()) {
                    List<MatchListResponse.GuideResponse> guideResponseList;
                    if (otherAnchors) {
                        guideResponseList = guidesByMatchId(infoResponse.getId());
                        infoResponse.setGuides(guideResponseList);
                    }
                }
            }
            return response;
        }

        LinkedHashMap<String, List<MatchListResponse.MatchInfoResponse>> dateMatchList = new LinkedHashMap<>();
        for (Match match : matches) {
            try {
                Sports sports = sportsMapper.selectById(match.getSportsId());
                Team awayTeam = teamMapper.selectById(match.getAwayId());
                Team homeTeam = teamMapper.selectById(match.getHomeId());

                dateMatchList.computeIfAbsent(DateUtil.format(match.getMatchTime(), DateUtil.YMD_),
                        value -> new ArrayList<>()).add(
                        conversionMatchInfo(match, sports.getSportsId(), awayTeam, homeTeam, otherAnchors));
            } catch (RuntimeException e) {
                e.printStackTrace();
            }
        }
        MatchListResponse matchListResponse = MatchListResponse.builder()
                .matchLists(dateMatchList.entrySet().stream().map(b -> MatchListResponse.MatchResponse.builder()
                        .matchInfos(b.getValue())
                        .matchTime(b.getKey())
                        .build()).collect(Collectors.toList()))
                .build();

        redisUtilsService.cacheStringInfo(key, JSONObject.toJSONString(matchListResponse), 5 * 60 * 60L);
        return matchListResponse;
    }

    @Override
    public ResponseData<?> matchInfoBasketball(CommonStringId commonIntId, HttpServletRequest request) {
        Match match = matchMapper.selectById(commonIntId.getId());
        if (match == null) {
            return ResponseData.fail500Response("ID错误，篮球比赛不存在");
        }
        Sports sports = sportsMapper.selectById(match.getSportsId());
        if (sports.getSportsId() != 1) {
            return ResponseData.fail500Response("此接口获取篮球比赛详情");
        }

        Team awayTeam = teamMapper.selectById(match.getAwayId());
        Team homeTeam = teamMapper.selectById(match.getHomeId());

        MatchInfoNewResponse matchInfoResponse = MatchInfoNewResponse.builder()
                .status(match.getCompetitionStatus())
                .matchDay(DateUtil.format(match.getMatchTime(), DateUtil.MD_))
                .matchTime(DateUtil.format(match.getMatchTime(), DateUtil.HM_))
                .sportType(1)//足球0 篮球1
                .teamMatchInfo(MatchInfoNewResponse.TeamMatchInfo.builder()
                        .matchShortName(sports.getCompetitionShortName())
                        .awayTeam(MatchListInfo.builder()
                                .score(match.getAwayScore())
                                .teamLogo(awayTeam.getTeamIcon())
                                .teamName(awayTeam.getTeamName())
                                .build())
                        .homeTeam(MatchListInfo.builder()
                                .score(match.getHomeScore())
                                .teamName(homeTeam.getTeamName())
                                .teamLogo(homeTeam.getTeamIcon())
                                .build())
                        .build())
                //一手情报
                .matchTeamMsg(getMatchTeamMsg(homeTeam.getId(), awayTeam.getId(), match.getMatchTime()))
                //积分榜
                .teamIntegralDto(getTeamIntegralDtoByDb(homeTeam.getId(), awayTeam.getId()))
                //分析数据
                .matchRecentDtos(getMatchRecentByDbDto(match.getHomeId(), match.getAwayId()))
                .build();
        if (match.getCompetitionStatus() == 2) {
            return ResponseData.successResponse(matchInfoResponse);
        }

        TeamCompetition homeCompetition = null;
        TeamCompetition awayCompetition = null;
        if (StringUtils.isNotBlank(match.getCompetitionHomeId())) {
            homeCompetition = teamCompetitionMapper.selectById(match.getCompetitionHomeId());
            awayCompetition = teamCompetitionMapper.selectById(match.getCompetitionAwayId());

            matchInfoResponse.setAwayCompetition(conversion(awayCompetition));
            matchInfoResponse.setHomeCompetition(conversion(homeCompetition));
            //队员比分数据
            List<PlayerCompetition> homePlayerCompetitions = playerCompetitionMapper.selectList(Wrappers.<PlayerCompetition>lambdaQuery()
                    .eq(PlayerCompetition::getDeleted, 0)
                    .eq(PlayerCompetition::getTeamCompetitionId, homeCompetition.getId())
            );
            matchInfoResponse.setHomePlayerList(homePlayerCompetitions.stream().map(this::conversionPlayer).collect(Collectors.toList()));

            List<PlayerCompetition> awayPlayerCompetitions = playerCompetitionMapper.selectList(Wrappers.<PlayerCompetition>lambdaQuery()
                    .eq(PlayerCompetition::getDeleted, 0)
                    .eq(PlayerCompetition::getTeamCompetitionId, awayCompetition.getId())
            );
            matchInfoResponse.setHomePlayerList(awayPlayerCompetitions.stream().map(this::conversionPlayer).collect(Collectors.toList()));
        }
        return ResponseData.successResponse(matchInfoResponse);
    }

    @Override
    public ResponseData<?> matchInfoFootball(CommonStringId commonIntId, HttpServletRequest request) {
        Match match = matchMapper.selectById(commonIntId.getId());
        if (match == null) {
            return ResponseData.fail500Response("ID错误，足球比赛不存在");
        }
        Sports sports = sportsMapper.selectById(match.getSportsId());
        if (sports.getSportsId() != 0) {
            return ResponseData.fail500Response("此接口获取足球比赛详情");
        }

        Team awayTeam = teamMapper.selectById(match.getAwayId());
        Team homeTeam = teamMapper.selectById(match.getHomeId());

        MatchInfoFootballResponse result = MatchInfoFootballResponse.builder()
                .matchGameInfo(MatchInfoFootballResponse.MatchGameInfo.builder()
                        .sportType(0)
                        .competitionName(match.getCompetitionName())
                        .competitionShortName(match.getCompetitionShortName())
                        .matchDay(DateUtil.format(match.getMatchTime(), DateUtil.MD_))
                        .matchTime(DateUtil.format(match.getMatchTime(), DateUtil.HM_))
                        .status(match.getCompetitionStatus())
                        .teamInfo(MatchInfoFootballResponse.TeamMatchInfo.builder()
                                .awayTeam(MatchListInfo.builder()
                                        .score(match.getAwayScore())
                                        .teamLogo(awayTeam.getTeamIcon())
                                        .teamName(awayTeam.getTeamName())
                                        .build())
                                .homeTeam(MatchListInfo.builder()
                                        .score(match.getHomeScore())
                                        .teamName(homeTeam.getTeamName())
                                        .teamLogo(homeTeam.getTeamIcon())
                                        .build())
                                .build())
                        .build())
                //一手情报
                .matchTeamMsg(getMatchTeamMsg(homeTeam.getId(), awayTeam.getId(), match.getMatchTime()))
                //积分榜
                .teamIntegralDto(getTeamIntegralDtoByDbFootball(homeTeam.getId(), awayTeam.getId()))
                //分析数据
                .matchRecentDtos(getMatchRecentByDbDto(match.getHomeId(), match.getAwayId()))
                .build();

        if (match.getCompetitionStatus() == 2) {
            return ResponseData.successResponse(result);
        }

        List<FootballStat> gameStat = footballStatMapper.selectList(Wrappers.<FootballStat>lambdaQuery()
                .eq(FootballStat::getDeleted, 0)
                .eq(FootballStat::getMatchId, commonIntId.getId())
                .orderByDesc(FootballStat::getUpdateTime)
        );

        List<FootballEvent> matchGoals = footballEventMapper.selectList(Wrappers.<FootballEvent>lambdaQuery()
                .eq(FootballEvent::getDeleted, 0)
                .eq(FootballEvent::getMatchId, commonIntId.getId())
                .orderByAsc(FootballEvent::getMinute)
        );

        result.setGameStat(gameStat.stream().map(b -> MatchInfoFootballResponse.MatchGameStat.builder()
                .statAway(b.getStatAway())
                .statHome(b.getStatHome())
                .statName(b.getStatName())
                .build()).collect(Collectors.toList()));

        result.setMatchGoals(matchGoals.stream().map(b -> MatchInfoFootballResponse.MatchGoal.builder()
                .assistedName(b.getAssistedName())
                .downName(b.getDownName())
                .eventName(b.getEventName())
                .minute(b.getMinute())
                .pName(b.getPName())
                .score(b.getScore())
                .type(b.getType())
                .upName(b.getUpName())
                .build())
                .collect(Collectors.toList()));
        return ResponseData.successResponse(result);
    }

    //LeYu 历史交锋 近期战绩
    private List<MatchInfoFootballResponse.MatchRecentDto> getMatchRecentDto(String matchId) {
        List<MatchRecent> matchRecents = matchRecentMapper.selectList(Wrappers.<MatchRecent>lambdaQuery()
                .eq(MatchRecent::getDeleted, 0)
                .eq(MatchRecent::getMatchId, matchId)
        );

        return matchRecents.stream().map(b -> MatchInfoFootballResponse.MatchRecentDto.builder()
                .teamRecordLost(b.getTeamRecordLost())
                .teamRecordName(b.getTeamRecordName())
                .teamRecordTie(b.getTeamRecordTie())
                .teamRecordWin(b.getTeamRecordWin())
                .competitionData(StringUtils.isNotBlank(b.getCompetitionData()) ? JSONObject.parseArray(b.getCompetitionData(), String[].class) : null)
                .build()).collect(Collectors.toList());
    }

    //AlStat 历史交锋 近期战绩
    private List<MatchInfoFootballResponse.MatchRecentDto> getMatchRecentByDbDto(String homeId, String awayId) {
        List<MatchInfoFootballResponse.MatchRecentDto> recentDtos = new ArrayList<>();
        try {
            //历史交锋
            List<Match> historyClash = matchMapper.selectList(Wrappers.<Match>lambdaQuery()
                    .eq(Match::getDeleted, 0)
                    .eq(Match::getHomeId, homeId)
                    .eq(Match::getAwayId, awayId)
                    .eq(Match::getCompetitionStatus, 3)
                    .orderByDesc(Match::getMatchTime)
            );
            String[][] historyMatchData = new String[historyClash.size()][5];
            int historyLost = 0;
            int historyWin = 0;
            int historyTie = 0;
            int h = 0;
            for (Match match : historyClash) {
                Sports sports = sportsMapper.selectById(match.getSportsId());
                historyMatchData[h][0] = sports.getCompetitionShortName();
                historyMatchData[h][1] = DateUtil.format(match.getMatchTime(), DateUtil.YMD_);
                historyMatchData[h][2] = "主";
                historyMatchData[h][3] = match.getHomeScore() + "-" + match.getAwayScore();
                historyMatchData[h][4] = "客";
                if (match.getHomeScore() > match.getAwayScore()) {
                    historyWin++;
                } else if (match.getHomeScore().equals(match.getAwayScore())) {
                    historyTie++;
                } else {
                    historyLost++;
                }
                h++;
            }
            recentDtos.add(MatchInfoFootballResponse.MatchRecentDto.builder()
                    .competitionData(JSONObject.parseArray(JSON.toJSONString(historyMatchData), String[].class))
                    .teamRecordLost(historyLost + "负")
                    .teamRecordName("历史交锋")
                    .teamRecordWin(historyWin + "胜")
                    .teamRecordTie(historyTie + "平")
                    .build());

            //主队主场
            List<Match> homeMainMatch = matchMapper.selectList(Wrappers.<Match>lambdaQuery()
                    .eq(Match::getDeleted, 0)
                    .eq(Match::getHomeId, homeId)
                    .eq(Match::getCompetitionStatus, 3)
                    .orderByDesc(Match::getMatchTime)
            );
            //主队客场
            List<Match> homeMinorMatch = matchMapper.selectList(Wrappers.<Match>lambdaQuery()
                    .eq(Match::getDeleted, 0)
                    .eq(Match::getAwayId, homeId)
                    .eq(Match::getCompetitionStatus, 3)
                    .orderByDesc(Match::getMatchTime)
            );

            String[][] homeMatchData = new String[homeMainMatch.size() + homeMinorMatch.size()][5];
            int homeLost = 0;
            int homeWin = 0;
            int homeTie = 0;
            int i = 0;
            for (Match match : homeMainMatch) {
                Sports sports = sportsMapper.selectById(match.getSportsId());
                Team team = teamMapper.selectById(match.getAwayId());
                homeMatchData[i][0] = sports.getCompetitionShortName();
                homeMatchData[i][1] = DateUtil.format(match.getMatchTime(), DateUtil.YMD_);
                homeMatchData[i][2] = "主";
                homeMatchData[i][3] = match.getHomeScore() + "-" + match.getAwayScore();
                homeMatchData[i][4] = team.getTeamName();
                if (match.getHomeScore() > match.getAwayScore()) {
                    homeWin++;
                } else if (match.getHomeScore().equals(match.getAwayScore())) {
                    homeTie++;
                } else {
                    homeLost++;
                }
                i++;
            }
            for (Match match : homeMinorMatch) {
                Sports sports = sportsMapper.selectById(match.getSportsId());
                Team team = teamMapper.selectById(match.getHomeId());
                homeMatchData[i][0] = sports.getCompetitionShortName();
                homeMatchData[i][1] = DateUtil.format(match.getMatchTime(), DateUtil.YMD_);
                homeMatchData[i][2] = "客";
                homeMatchData[i][3] = match.getHomeScore() + "-" + match.getAwayScore();
                homeMatchData[i][4] = team.getTeamName();
                if (match.getAwayScore() > match.getHomeScore()) {
                    homeWin++;
                } else if (match.getAwayScore().equals(match.getHomeScore())) {
                    homeTie++;
                } else {
                    homeLost++;
                }
                i++;
            }
            recentDtos.add(MatchInfoFootballResponse.MatchRecentDto.builder()
                    .competitionData(JSONObject.parseArray(JSON.toJSONString(homeMatchData), String[].class))
                    .teamRecordLost(homeLost + "负")
                    .teamRecordName("最近战绩")
                    .teamRecordWin(homeWin + "胜")
                    .teamRecordTie(homeTie + "平")
                    .build());

            //客队主场
            List<Match> awayMainMatch = matchMapper.selectList(Wrappers.<Match>lambdaQuery()
                    .eq(Match::getDeleted, 0)
                    .eq(Match::getHomeId, awayId)
                    .eq(Match::getCompetitionStatus, 3)
                    .orderByDesc(Match::getMatchTime)
            );
            //客队客场
            List<Match> awayMinorMatch = matchMapper.selectList(Wrappers.<Match>lambdaQuery()
                    .eq(Match::getDeleted, 0)
                    .eq(Match::getAwayId, awayId)
                    .eq(Match::getCompetitionStatus, 3)
                    .orderByDesc(Match::getMatchTime)
            );

            String[][] awayMatchData = new String[awayMainMatch.size() + awayMinorMatch.size()][5];
            int awayLost = 0;
            int awayWin = 0;
            int awayTie = 0;
            i = 0;
            for (Match match : awayMainMatch) {
                Sports sports = sportsMapper.selectById(match.getSportsId());
                Team team = teamMapper.selectById(match.getAwayId());
                awayMatchData[i][0] = sports.getCompetitionShortName();
                awayMatchData[i][1] = DateUtil.format(match.getMatchTime(), DateUtil.YMD_);
                awayMatchData[i][2] = "主";
                awayMatchData[i][3] = match.getHomeScore() + "-" + match.getAwayScore();
                awayMatchData[i][4] = team.getTeamName();
                if (match.getHomeScore() > match.getAwayScore()) {
                    awayWin++;
                } else if (match.getHomeScore().equals(match.getAwayScore())) {
                    awayTie++;
                } else {
                    awayLost++;
                }
                i++;
            }
            for (Match match : awayMinorMatch) {
                Sports sports = sportsMapper.selectById(match.getSportsId());
                Team team = teamMapper.selectById(match.getHomeId());
                awayMatchData[i][0] = sports.getCompetitionShortName();
                awayMatchData[i][1] = DateUtil.format(match.getMatchTime(), DateUtil.YMD_);
                awayMatchData[i][2] = "客";
                awayMatchData[i][3] = match.getHomeScore() + "-" + match.getAwayScore();
                awayMatchData[i][4] = team.getTeamName();
                if (match.getAwayScore() > match.getHomeScore()) {
                    awayWin++;
                } else if (match.getAwayScore().equals(match.getHomeScore())) {
                    awayTie++;
                } else {
                    awayLost++;
                }
                i++;
            }
            recentDtos.add(MatchInfoFootballResponse.MatchRecentDto.builder()
                    .competitionData(JSONObject.parseArray(JSON.toJSONString(awayMatchData), String[].class))
                    .teamRecordLost(awayLost + "负")
                    .teamRecordName("最近战绩")
                    .teamRecordWin(awayWin + "胜")
                    .teamRecordTie(awayTie + "平")
                    .build());
        } catch (Exception e) {
            e.printStackTrace();
        }
        return recentDtos;
    }

    private MatchInfoFootballResponse.TeamIntegralDto getTeamIntegralDtoByDbFootball(String homeTeamId, String awayTeamId) {
        String[] homeSum = new String[7];
        String[] homeMain = new String[7];
        String[] homeKe = new String[7];

        String[] awaySum = new String[7];
        String[] awayMain = new String[7];
        String[] awayKe = new String[7];
        try {
            homeSum[0] = "总";
            homeMain[0] = "主";
            homeKe[0] = "客";

            List<Match> matchesHomeMain = matchMapper.selectList(Wrappers.<Match>lambdaQuery()
                    .eq(Match::getDeleted, 0)
                    .eq(Match::getHomeId, homeTeamId)
                    .eq(Match::getCompetitionStatus, 3)
                    .orderByDesc(Match::getCreateTime)
            );
            List<Match> matchesHome = matchMapper.selectList(Wrappers.<Match>lambdaQuery()
                    .eq(Match::getDeleted, 0)
                    .eq(Match::getAwayId, homeTeamId)
                    .eq(Match::getCompetitionStatus, 3)
                    .orderByDesc(Match::getCreateTime)
            );

            homeMain[1] = matchesHomeMain.size() + "";
            long homeWins = matchesHomeMain.stream().filter(b -> b.getHomeScore() > b.getAwayScore()).count();
            long homeDraw = matchesHomeMain.stream().filter(b -> b.getHomeScore().equals(b.getAwayScore())).count();
            long homeLose = matchesHomeMain.stream().filter(b -> b.getHomeScore() < b.getAwayScore()).count();
            homeMain[2] = homeWins + "";
            homeMain[3] = homeDraw + "";
            homeMain[4] = homeLose + "";
            homeMain[5] = homeWins * 3 + homeDraw + "";
            homeMain[6] = matchesHomeMain.size() == 0 ? "0%" : homeWins * 100 / matchesHomeMain.size() + "%";

            homeKe[1] = matchesHome.size() + "";
            long homeKeWins = matchesHome.stream().filter(b -> b.getHomeScore() < b.getAwayScore()).count();
            long homeKeDraw = matchesHome.stream().filter(b -> b.getHomeScore().equals(b.getAwayScore())).count();
            long homeKeLose = matchesHome.stream().filter(b -> b.getHomeScore() > b.getAwayScore()).count();
            homeKe[2] = homeKeWins + "";
            homeKe[3] = homeKeDraw + "";
            homeKe[4] = homeKeLose + "";
            homeKe[5] = homeKeWins * 3 + homeKeDraw + "";
            homeKe[6] = matchesHome.size() == 0 ? "0%" : homeKeWins * 100 / matchesHome.size() + "%";

            homeSum[1] = matchesHomeMain.size() + matchesHome.size() + "";
            homeSum[2] = homeWins + homeKeWins + "";
            homeSum[3] = homeDraw + homeKeDraw + "";
            homeSum[4] = homeLose + homeKeLose + "";
            homeSum[5] = (homeWins + homeKeWins) * 3 + (homeDraw + homeKeDraw) + "";
            homeSum[6] = (matchesHomeMain.size() + matchesHome.size()) == 0 ? "0%" : (homeWins + homeKeWins) * 100 / (matchesHomeMain.size() + matchesHome.size()) + "%";

            // away
            awaySum[0] = "总";
            awayMain[0] = "主";
            awayKe[0] = "客";

            List<Match> matchesAwayMain = matchMapper.selectList(Wrappers.<Match>lambdaQuery()
                    .eq(Match::getDeleted, 0)
                    .eq(Match::getHomeId, awayTeamId)
                    .eq(Match::getCompetitionStatus, 3)
                    .orderByDesc(Match::getCreateTime)
            );
            List<Match> matchesAway = matchMapper.selectList(Wrappers.<Match>lambdaQuery()
                    .eq(Match::getDeleted, 0)
                    .eq(Match::getAwayId, awayTeamId)
                    .eq(Match::getCompetitionStatus, 3)
                    .orderByDesc(Match::getCreateTime)
            );

            awayMain[1] = matchesAwayMain.size() + "";
            long awayWins = matchesAwayMain.stream().filter(b -> b.getHomeScore() > b.getAwayScore()).count();
            long awayDraw = matchesAwayMain.stream().filter(b -> b.getHomeScore().equals(b.getAwayScore())).count();
            long awayLose = matchesAwayMain.stream().filter(b -> b.getHomeScore() < b.getAwayScore()).count();
            awayMain[2] = awayWins + "";
            awayMain[3] = awayDraw + "";
            awayMain[4] = awayLose + "";
            awayMain[5] = awayWins * 3 + awayDraw + "";
            awayMain[6] = matchesAwayMain.size() == 0 ? "0%" : awayWins * 100 / matchesAwayMain.size() + "%";

            awayKe[1] = matchesAway.size() + "";
            long awayKeWins = matchesAway.stream().filter(b -> b.getHomeScore() < b.getAwayScore()).count();
            long awayKeDraw = matchesAway.stream().filter(b -> b.getHomeScore().equals(b.getAwayScore())).count();
            long awayKeLose = matchesAway.stream().filter(b -> b.getHomeScore() > b.getAwayScore()).count();
            awayKe[2] = awayKeWins + "";
            awayKe[3] = awayKeDraw + "";
            awayKe[4] = awayKeLose + "";
            awayKe[5] = awayKeWins * 3 + awayKeDraw + "";
            awayKe[6] = matchesAway.size() == 0 ? "0%" : awayKeWins * 100 / matchesAway.size() + "%";

            awaySum[1] = matchesAwayMain.size() + matchesAway.size() + "";
            awaySum[2] = awayWins + awayKeWins + "";
            awaySum[3] = awayDraw + awayKeDraw + "";
            awaySum[4] = awayLose + awayKeLose + "";
            awaySum[5] = (awayWins + awayKeWins) * 3 + (awayDraw + awayKeDraw) + "";
            awaySum[6] = (matchesAwayMain.size() + matchesAway.size()) == 0 ? "0%" : (awayWins + awayKeWins) * 100 / (matchesAwayMain.size() + matchesAway.size()) + "%";
        } catch (Exception e) {
            e.printStackTrace();
        }

        return MatchInfoFootballResponse.TeamIntegralDto.builder()
                .awayIntegral(MatchInfoFootballResponse.TeamIntegralInfoDto.builder()
                        .header(headerFootball)
                        .integralAway(awayKe)
                        .integralHome(awayMain)
                        .integralSum(awaySum)
                        .build())
                .homeIntegral(MatchInfoFootballResponse.TeamIntegralInfoDto.builder()
                        .header(headerFootball)
                        .integralAway(homeKe)
                        .integralHome(homeMain)
                        .integralSum(homeSum)
                        .build())
                .build();
    }

    private MatchInfoFootballResponse.TeamIntegralDto getTeamIntegralDtoByDb(String homeTeamId, String awayTeamId) {
        String[] homeSum = new String[7];
        String[] awaySum = new String[7];
        try {
            homeSum[0] = "总";
            awaySum[0] = "总";

            List<Match> matchesHomeMain = matchMapper.selectList(Wrappers.<Match>lambdaQuery()
                    .eq(Match::getDeleted, 0)
                    .eq(Match::getHomeId, homeTeamId)
                    .eq(Match::getCompetitionStatus, 3)
                    .orderByDesc(Match::getCreateTime)
            );
            List<Match> matchesHome = matchMapper.selectList(Wrappers.<Match>lambdaQuery()
                    .eq(Match::getDeleted, 0)
                    .eq(Match::getAwayId, homeTeamId)
                    .eq(Match::getCompetitionStatus, 3)
                    .orderByDesc(Match::getCreateTime)
            );
            homeSum[1] = matchesHomeMain.size() + matchesHome.size() + "";
            int homeMainWins = 0;
            int homeMainLose = 0;
            for (Match m : matchesHomeMain) {
                if (m.getHomeScore() > m.getAwayScore()) {
                    homeMainWins++;
                } else {
                    homeMainLose++;
                }
            }
            homeSum[4] = homeMainWins + "-" + homeMainLose;
            int awayMainWins = 0;
            int awayMinLose = 0;
            for (Match m : matchesHome) {
                if (m.getAwayScore() > m.getHomeScore()) {
                    awayMainWins++;
                    homeMainWins++;
                } else {
                    awayMinLose++;
                    homeMainLose++;
                }
            }
            homeSum[2] = homeMainWins + "";
            homeSum[3] = homeMainLose + "";
            homeSum[5] = awayMainWins + "-" + awayMinLose;
            homeSum[6] = (homeMainWins + homeMainLose) == 0 ? "0%" : homeMainWins * 100 / (homeMainWins + homeMainLose) + "%";

            // away

            List<Match> matchesAwayMain = matchMapper.selectList(Wrappers.<Match>lambdaQuery()
                    .eq(Match::getDeleted, 0)
                    .eq(Match::getHomeId, awayTeamId)
                    .eq(Match::getCompetitionStatus, 3)
                    .orderByDesc(Match::getCreateTime)
            );
            List<Match> matchesAway = matchMapper.selectList(Wrappers.<Match>lambdaQuery()
                    .eq(Match::getDeleted, 0)
                    .eq(Match::getAwayId, awayTeamId)
                    .eq(Match::getCompetitionStatus, 3)
                    .orderByDesc(Match::getCreateTime)
            );
            awaySum[1] = matchesAwayMain.size() + matchesAway.size() + "";
            homeMainWins = 0;
            homeMainLose = 0;
            for (Match m : matchesAwayMain) {
                if (m.getHomeScore() > m.getAwayScore()) {
                    homeMainWins++;
                } else {
                    homeMainLose++;
                }
            }
            awaySum[4] = homeMainWins + "-" + homeMainLose;
            awayMainWins = 0;
            awayMinLose = 0;
            for (Match m : matchesAway) {
                if (m.getAwayScore() > m.getHomeScore()) {
                    awayMainWins++;
                    homeMainWins++;
                } else {
                    awayMinLose++;
                    homeMainLose++;
                }
            }
            awaySum[2] = homeMainWins + "";
            awaySum[3] = homeMainLose + "";
            awaySum[5] = awayMainWins + "-" + awayMinLose;
            awaySum[6] = (homeMainWins + homeMainLose) == 0 ? "0%" : homeMainWins * 100 / (homeMainWins + homeMainLose) + "%";
        } catch (Exception e) {
            e.printStackTrace();
        }
        return MatchInfoFootballResponse.TeamIntegralDto.builder()
                .awayIntegral(MatchInfoFootballResponse.TeamIntegralInfoDto.builder()
                        .header(header)
                        .integralAway(null)
                        .integralHome(null)
                        .integralSum(homeSum)
                        .build())
                .homeIntegral(MatchInfoFootballResponse.TeamIntegralInfoDto.builder()
                        .header(header)
                        .integralAway(null)
                        .integralHome(null)
                        .integralSum(awaySum)
                        .build())
                .build();
    }

    private MatchInfoFootballResponse.TeamIntegralDto getTeamIntegralDto(String homeTeamId, String awayTeamId, Date matchTime) {
        Optional<TeamIntegral> homeIntegral = teamIntegralMapper.selectList(Wrappers.<TeamIntegral>lambdaQuery()
                .eq(TeamIntegral::getDeleted, 0)
                .eq(TeamIntegral::getTeamId, homeTeamId)
                .le(TeamIntegral::getCreateTime, matchTime)
                .orderByDesc(TeamIntegral::getCreateTime)
        ).stream().findFirst();

        Optional<TeamIntegral> awayIntegral = teamIntegralMapper.selectList(Wrappers.<TeamIntegral>lambdaQuery()
                .eq(TeamIntegral::getDeleted, 0)
                .eq(TeamIntegral::getTeamId, awayTeamId)
                .le(TeamIntegral::getCreateTime, matchTime)
                .orderByDesc(TeamIntegral::getCreateTime)
        ).stream().findFirst();

        return MatchInfoFootballResponse.TeamIntegralDto.builder()
                .awayIntegral(MatchInfoFootballResponse.TeamIntegralInfoDto.builder()
                        .header(homeIntegral.filter(f -> StringUtils.isNotBlank(f.getIntegralHeader())).map(b -> b.getIntegralHeader().split(",")).orElse(null))
                        .integralAway(homeIntegral.filter(f -> StringUtils.isNotBlank(f.getIntegralAway())).map(b -> b.getIntegralAway().split(",")).orElse(null))
                        .integralHome(homeIntegral.filter(f -> StringUtils.isNotBlank(f.getIntegralHome())).map(b -> b.getIntegralHome().split(",")).orElse(null))
                        .integralSum(homeIntegral.filter(f -> StringUtils.isNotBlank(f.getIntegralSum())).map(b -> b.getIntegralSum().split(",")).orElse(null))
                        .build())
                .homeIntegral(MatchInfoFootballResponse.TeamIntegralInfoDto.builder()
                        .header(awayIntegral.filter(f -> StringUtils.isNotBlank(f.getIntegralHeader())).map(b -> b.getIntegralHeader().split(",")).orElse(null))
                        .integralAway(awayIntegral.filter(f -> StringUtils.isNotBlank(f.getIntegralAway())).map(b -> b.getIntegralAway().split(",")).orElse(null))
                        .integralHome(awayIntegral.filter(f -> StringUtils.isNotBlank(f.getIntegralHome())).map(b -> b.getIntegralHome().split(",")).orElse(null))
                        .integralSum(awayIntegral.filter(f -> StringUtils.isNotBlank(f.getIntegralSum())).map(b -> b.getIntegralSum().split(",")).orElse(null))
                        .build())
                .build();
    }

    private MatchInfoFootballResponse.MatchTeamMsg getMatchTeamMsg(String homeTeamId, String awayTeamId, Date matchTime) {
        Optional<TeamReport> homeReport = teamReportMapper.selectList(Wrappers.<TeamReport>lambdaQuery()
                .eq(TeamReport::getDeleted, 0)
                .eq(TeamReport::getTeamId, homeTeamId)
                .le(TeamReport::getCreateTime, matchTime)
                .orderByDesc(TeamReport::getCreateTime)
        ).stream().findFirst();

        Optional<TeamReport> awayReport = teamReportMapper.selectList(Wrappers.<TeamReport>lambdaQuery()
                .eq(TeamReport::getDeleted, 0)
                .eq(TeamReport::getTeamId, awayTeamId)
                .le(TeamReport::getCreateTime, matchTime)
                .orderByDesc(TeamReport::getCreateTime)
        ).stream().findFirst();

        return MatchInfoFootballResponse.MatchTeamMsg.builder()
                .homeMsg(MatchInfoFootballResponse.TeamMsg.builder()
                        .goodMsg(homeReport.map(TeamReport::getGodMsg).orElse(null))
                        .badMsg(homeReport.map(TeamReport::getBadMsg).orElse(null))
                        .build())
                .awayMsg(MatchInfoFootballResponse.TeamMsg.builder()
                        .goodMsg(awayReport.map(TeamReport::getGodMsg).orElse(null))
                        .badMsg(awayReport.map(TeamReport::getBadMsg).orElse(null))
                        .build())
                .build();
    }

    @Override
    public ResponseData<?> getMatchSummaryById(MatchSummaryRequest commonPage) {
        int jumpNum = (commonPage.getPageNum() - 1) * commonPage.getPageSize();

        QueryWrapper<MatchSummary> wrapper = new QueryWrapper<>();
        wrapper.eq("deleted", StatusEnum.ENABLE.getCode());
        wrapper.eq("match_id", commonPage.getMatchId());

        long count = matchSummaryMapper.selectCount(wrapper);

        wrapper.orderByDesc("sort");
        wrapper.last(String.format(" limit %s,%s", jumpNum, commonPage.getPageSize()));

        List<MatchSummary> matchSummaries = matchSummaryMapper.selectList(wrapper);
        List<MatchSummaryResponse> summaryResponses = matchSummaries.stream().map(b ->
                MatchSummaryResponse.builder()
                        .matchTime(b.getMatchTime())
                        .msg(b.getMessage())
                        .sort(b.getSort())
                        .build())
                .collect(Collectors.toList());
        return ResponseData.successResponse(new ResultPage<>(commonPage.getPageNum(), commonPage.getPageSize(), (int) count, summaryResponses));
    }

    @Override
    public ResponseData<?> getMatchLiveList(CommonStringId commonStringId) {
        List<MatchLive> matchLives = matchLiveMapper.selectList(Wrappers.<MatchLive>lambdaQuery()
                .eq(MatchLive::getDeleted, StatusEnum.ENABLE.getCode())
                .eq(MatchLive::getMatchId, commonStringId.getId())
        );

        List<GetMatchLiveListResponse> responses = matchLives.stream().map(b -> GetMatchLiveListResponse.builder()
                .jumpLiveUrl(b.getSourceUrl())
                .liveName(b.getSourceName())
                .provider(b.getCreateBy())
                .build())
                .collect(Collectors.toList());
        return ResponseData.successResponse(responses);
    }

    @Override
    public ResponseData<MatchFootballOddsResponse> matchFootballOdds(CommonStringId commonIntId, HttpServletRequest request) {
        Match match = matchMapper.selectById(commonIntId.getId());
        if (match == null)
            return ResponseData.fail500Response("ID错误，比赛不存在");

        Sports sports = sportsMapper.selectById(match.getSportsId());
        if (sports.getSportsId() != 0)
            return ResponseData.fail500Response("此接口获取足球比赛指数");

        List<MatchOdds> odds = matchOddsMapper.selectList(Wrappers.<MatchOdds>lambdaQuery()
                .eq(MatchOdds::getDeleted, StatusEnum.ENABLE.getCode())
                .eq(MatchOdds::getMatchId, commonIntId.getId())
        );

        Map<Integer, Map<Integer, List<MatchOdds>>> companyInfos = new HashMap<>();
        for (MatchOdds matchOdds : odds) {
            companyInfos.computeIfAbsent(matchOdds.getCompanyId(), k -> new HashMap<>())
                    .computeIfAbsent(matchOdds.getOddsType(), m -> new ArrayList<>()).add(matchOdds);
        }

        List<MatchFootballOddsResponse> responses = new ArrayList<>();
        for (Integer companyId : companyInfos.keySet()) {
            IndexCompany indexCompany = indexCompanyMapper.selectById(companyId);
            if (indexCompany == null)
                continue;

            List<MatchFootballOddsResponse.CompanyInfoZuQiu> companyInfoList = new ArrayList<>();
            MatchFootballOddsResponse.CompanyInfoZuQiu responseInfo = MatchFootballOddsResponse.CompanyInfoZuQiu.builder()
                    .companyName(indexCompany.getCompanyName())
                    .country(indexCompany.getCountry())
                    .ouZhi(MatchFootballOddsResponse.OuZhiOddsInfo.builder().build())
                    .yaZhi(MatchFootballOddsResponse.YaZhiOddsZuQiuInfo.builder().build())
                    .daXiao(MatchFootballOddsResponse.DaXiaoOddsInfo.builder().build())
                    .build();

            for (Integer oddsType : companyInfos.get(companyId).keySet()) {
                for (MatchOdds matchOdds : companyInfos.get(companyId).get(oddsType)) {
                    if (oddsType == 0) {
                        MatchFootballOddsResponse.FootballOdds footballOdds = MatchFootballOddsResponse.FootballOdds.builder()
                                .win(matchOdds.getFieldOne())
                                .type(matchOdds.getMatchType() == 0 ? "初" : "即")
                                .draw(matchOdds.getFieldTwo())
                                .loss(matchOdds.getFieldThree())
                                .build();
                        if (matchOdds.getMatchType() == 0) {
                            responseInfo.getOuZhi().setInitialOdds(footballOdds);
                        } else {
                            responseInfo.getOuZhi().setNowOdds(footballOdds);
                        }
                    } else if (oddsType == 1) {
                        MatchFootballOddsResponse.FootballYaZhiOdds footballOdds = MatchFootballOddsResponse.FootballYaZhiOdds.builder()
                                .type(matchOdds.getMatchType() == 0 ? "初" : "即")
                                .handicap(matchOdds.getFieldOne())
                                .home(matchOdds.getFieldTwo())
                                .away(matchOdds.getFieldThree())
                                .build();
                        if (matchOdds.getMatchType() == 0) {
                            responseInfo.getYaZhi().setInitialOdds(footballOdds);
                        } else {
                            responseInfo.getYaZhi().setNowOdds(footballOdds);
                        }
                    } else if (oddsType == 2) {
                        MatchFootballOddsResponse.FootballDaXiaoOdds footballOdds = MatchFootballOddsResponse.FootballDaXiaoOdds.builder()
                                .type(matchOdds.getMatchType() == 0 ? "初" : "即")
                                .handicap(matchOdds.getFieldOne())
                                .over(matchOdds.getFieldTwo())
                                .under(matchOdds.getFieldThree())
                                .build();
                        if (matchOdds.getMatchType() == 0) {
                            responseInfo.getDaXiao().setInitialOdds(footballOdds);
                        } else {
                            responseInfo.getDaXiao().setNowOdds(footballOdds);
                        }
                    }
                }
            }
            companyInfoList.add(responseInfo);

            responses.add(MatchFootballOddsResponse.builder()
                    .companyInfos(companyInfoList)
                    .build());
        }
        distinguishFootballMaxAndMin(responses);
        return ResponseData.successResponse(responses);
    }

    private void distinguishFootballMaxAndMin(List<MatchFootballOddsResponse> responses) {
        MatchFootballOddsResponse.CompanyInfoZuQiu historyMax = MatchFootballOddsResponse.CompanyInfoZuQiu.builder()
                .companyName("最高值")
                .country("")
                .daXiao(MatchFootballOddsResponse.DaXiaoOddsInfo.builder()
                        .initialOdds(MatchFootballOddsResponse.FootballDaXiaoOdds.builder()
                                .handicap(0.00)
                                .over(0.00)
                                .type("初")
                                .under(0.00)
                                .build())
                        .nowOdds(MatchFootballOddsResponse.FootballDaXiaoOdds.builder()
                                .handicap(0.00)
                                .over(0.00)
                                .type("即")
                                .under(0.00)
                                .build())
                        .build())
                .ouZhi(MatchFootballOddsResponse.OuZhiOddsInfo.builder()
                        .initialOdds(MatchFootballOddsResponse.FootballOdds.builder()
                                .loss(0.00)
                                .type("初")
                                .draw(0.00)
                                .win(0.00)
                                .build())
                        .nowOdds(MatchFootballOddsResponse.FootballOdds.builder()
                                .win(0.00)
                                .type("即")
                                .draw(0.00)
                                .loss(0.00)
                                .build())
                        .build())
                .yaZhi(MatchFootballOddsResponse.YaZhiOddsZuQiuInfo.builder()
                        .initialOdds(MatchFootballOddsResponse.FootballYaZhiOdds.builder()
                                .away(0.00)
                                .handicap(0.00)
                                .home(0.00)
                                .type("初")
                                .build())
                        .nowOdds(MatchFootballOddsResponse.FootballYaZhiOdds.builder()
                                .away(0.00)
                                .handicap(0.00)
                                .home(0.00)
                                .type("即")
                                .build())
                        .build())
                .build();

        MatchFootballOddsResponse.CompanyInfoZuQiu historyMin = new MatchFootballOddsResponse.CompanyInfoZuQiu();

        AtomicBoolean init = new AtomicBoolean(true);
        for (MatchFootballOddsResponse company : responses) {
            for (MatchFootballOddsResponse.CompanyInfoZuQiu companyInfo : company.getCompanyInfos()) {
                if (init.get()) {
                    historyMin = JSON.toJavaObject(JSONObject.parseObject(JSONObject.toJSONString(companyInfo)), MatchFootballOddsResponse.CompanyInfoZuQiu.class);
                    historyMin.setCompanyName("最低值");
                    historyMin.setCountry("");
                    init.set(false);
                }

                //ouzhi
                if (companyInfo.getOuZhi().getInitialOdds() != null) {
                    //max
                    Double maxInitLoss = Math.max(historyMax.getOuZhi().getInitialOdds().getLoss(), companyInfo.getOuZhi().getInitialOdds().getLoss());
                    historyMax.getOuZhi().getInitialOdds().setLoss(maxInitLoss);
                    Double maxInitWin = Math.max(historyMax.getOuZhi().getInitialOdds().getWin(), companyInfo.getOuZhi().getInitialOdds().getWin());
                    historyMax.getOuZhi().getInitialOdds().setWin(maxInitWin);
                    Double maxInitDraw = Math.max(historyMax.getOuZhi().getInitialOdds().getDraw(), companyInfo.getOuZhi().getInitialOdds().getDraw());
                    historyMax.getOuZhi().getInitialOdds().setDraw(maxInitDraw);
                    //min
                    if (historyMin.getOuZhi().getInitialOdds() == null) {
                        historyMin.getOuZhi().setInitialOdds(JSON.toJavaObject(JSONObject.parseObject(JSONObject.toJSONString(companyInfo.getOuZhi().getInitialOdds())),
                                MatchFootballOddsResponse.FootballOdds.class));
                    } else {
                        Double minInitLoss = Math.min(historyMin.getOuZhi().getInitialOdds().getLoss(), companyInfo.getOuZhi().getInitialOdds().getLoss());
                        historyMin.getOuZhi().getInitialOdds().setLoss(minInitLoss);
                        Double minInitWin = Math.min(historyMin.getOuZhi().getInitialOdds().getWin(), companyInfo.getOuZhi().getInitialOdds().getWin());
                        historyMin.getOuZhi().getInitialOdds().setWin(minInitWin);
                        Double minInitDraw = Math.min(historyMin.getOuZhi().getInitialOdds().getDraw(), companyInfo.getOuZhi().getInitialOdds().getDraw());
                        historyMin.getOuZhi().getInitialOdds().setDraw(minInitDraw);
                    }
                }
                if (companyInfo.getOuZhi().getNowOdds() != null) {
                    //max
                    Double maxNowLoss = Math.max(historyMax.getOuZhi().getNowOdds().getLoss(), companyInfo.getOuZhi().getNowOdds().getLoss());
                    historyMax.getOuZhi().getNowOdds().setLoss(maxNowLoss);
                    Double maxNowWin = Math.max(historyMax.getOuZhi().getNowOdds().getWin(), companyInfo.getOuZhi().getNowOdds().getWin());
                    historyMax.getOuZhi().getNowOdds().setWin(maxNowWin);
                    Double maxNowDraw = Math.max(historyMax.getOuZhi().getNowOdds().getDraw(), companyInfo.getOuZhi().getNowOdds().getDraw());
                    historyMax.getOuZhi().getNowOdds().setDraw(maxNowDraw);
                    //min
                    if (historyMin.getOuZhi().getNowOdds() == null) {
                        historyMin.getOuZhi().setNowOdds(JSON.toJavaObject(JSONObject.parseObject(JSONObject.toJSONString(companyInfo.getOuZhi().getNowOdds())),
                                MatchFootballOddsResponse.FootballOdds.class));
                    } else {
                        Double minNowLoss = Math.min(historyMin.getOuZhi().getNowOdds().getLoss(), companyInfo.getOuZhi().getNowOdds().getLoss());
                        historyMin.getOuZhi().getNowOdds().setLoss(minNowLoss);
                        Double minNowWin = Math.min(historyMin.getOuZhi().getNowOdds().getWin(), companyInfo.getOuZhi().getNowOdds().getWin());
                        historyMin.getOuZhi().getNowOdds().setWin(minNowWin);
                        Double minNowDraw = Math.min(historyMin.getOuZhi().getNowOdds().getDraw(), companyInfo.getOuZhi().getNowOdds().getDraw());
                        historyMin.getOuZhi().getNowOdds().setDraw(minNowDraw);
                    }
                }
                //yazhi
                if (companyInfo.getYaZhi().getInitialOdds() != null) {
                    //max
                    Double maxInitAway = Math.max(historyMax.getYaZhi().getInitialOdds().getAway(), companyInfo.getYaZhi().getInitialOdds().getAway());
                    historyMax.getYaZhi().getInitialOdds().setAway(maxInitAway);
                    Double maxInitHome = Math.max(historyMax.getYaZhi().getInitialOdds().getHome(), companyInfo.getYaZhi().getInitialOdds().getHome());
                    historyMax.getYaZhi().getInitialOdds().setHome(maxInitHome);
                    Double maxInitHandicap = Math.max(historyMax.getYaZhi().getInitialOdds().getHandicap(), companyInfo.getYaZhi().getInitialOdds().getHandicap());
                    historyMax.getYaZhi().getInitialOdds().setHandicap(maxInitHandicap);
                    //min
                    if (historyMin.getYaZhi().getInitialOdds() == null) {
                        historyMin.getYaZhi().setInitialOdds(JSON.toJavaObject(JSONObject.parseObject(JSONObject.toJSONString(companyInfo.getYaZhi().getInitialOdds())),
                                MatchFootballOddsResponse.FootballYaZhiOdds.class));
                    } else {
                        Double minInitAway = Math.min(historyMin.getYaZhi().getInitialOdds().getAway(), companyInfo.getYaZhi().getInitialOdds().getAway());
                        historyMin.getYaZhi().getInitialOdds().setAway(minInitAway);
                        Double minInitHome = Math.min(historyMin.getYaZhi().getInitialOdds().getHome(), companyInfo.getYaZhi().getInitialOdds().getHome());
                        historyMin.getYaZhi().getInitialOdds().setHome(minInitHome);
                        Double minInitHandicap = Math.min(historyMin.getYaZhi().getInitialOdds().getHandicap(), companyInfo.getYaZhi().getInitialOdds().getHandicap());
                        historyMin.getYaZhi().getInitialOdds().setHandicap(minInitHandicap);
                    }
                }
                if (companyInfo.getYaZhi().getNowOdds() != null) {
                    //max
                    Double maxNowAway = Math.max(historyMax.getYaZhi().getNowOdds().getAway(), companyInfo.getYaZhi().getNowOdds().getAway());
                    historyMax.getYaZhi().getNowOdds().setAway(maxNowAway);
                    Double maxNowHome = Math.max(historyMax.getYaZhi().getNowOdds().getHome(), companyInfo.getYaZhi().getNowOdds().getHome());
                    historyMax.getYaZhi().getNowOdds().setHome(maxNowHome);
                    Double maxNowHandicap = Math.max(historyMax.getYaZhi().getNowOdds().getHandicap(), companyInfo.getYaZhi().getNowOdds().getHandicap());
                    historyMax.getYaZhi().getNowOdds().setHandicap(maxNowHandicap);
                    //min
                    if (historyMin.getYaZhi().getNowOdds() == null) {
                        historyMin.getYaZhi().setNowOdds(JSON.toJavaObject(JSONObject.parseObject(JSONObject.toJSONString(companyInfo.getYaZhi().getNowOdds())),
                                MatchFootballOddsResponse.FootballYaZhiOdds.class));
                    } else {
                        Double minNowAway = Math.min(historyMin.getYaZhi().getNowOdds().getAway(), companyInfo.getYaZhi().getNowOdds().getAway());
                        historyMin.getYaZhi().getNowOdds().setAway(minNowAway);
                        Double minNowHome = Math.min(historyMin.getYaZhi().getNowOdds().getHome(), companyInfo.getYaZhi().getNowOdds().getHome());
                        historyMin.getYaZhi().getNowOdds().setHome(minNowHome);
                        Double minNowHandicap = Math.min(historyMin.getYaZhi().getNowOdds().getHandicap(), companyInfo.getYaZhi().getNowOdds().getHandicap());
                        historyMin.getYaZhi().getNowOdds().setHandicap(minNowHandicap);
                    }
                }
                //daxiao
                if (companyInfo.getDaXiao().getInitialOdds() != null) {
                    //max
                    Double maxInitOver = Math.max(historyMax.getDaXiao().getInitialOdds().getOver(), companyInfo.getDaXiao().getInitialOdds().getOver());
                    historyMax.getDaXiao().getInitialOdds().setOver(maxInitOver);
                    Double maxInitUnder = Math.max(historyMax.getDaXiao().getInitialOdds().getUnder(), companyInfo.getDaXiao().getInitialOdds().getUnder());
                    historyMax.getDaXiao().getInitialOdds().setUnder(maxInitUnder);
                    Double maxInitHandicapDaXiao = Math.max(historyMax.getDaXiao().getInitialOdds().getHandicap(), companyInfo.getDaXiao().getInitialOdds().getHandicap());
                    historyMax.getDaXiao().getInitialOdds().setHandicap(maxInitHandicapDaXiao);
                    //min
                    if (historyMin.getDaXiao().getInitialOdds() == null) {
                        historyMin.getDaXiao().setInitialOdds(JSON.toJavaObject(JSONObject.parseObject(JSONObject.toJSONString(companyInfo.getDaXiao().getInitialOdds())),
                                MatchFootballOddsResponse.FootballDaXiaoOdds.class));
                    } else {
                        Double minInitOver = Math.min(historyMin.getDaXiao().getInitialOdds().getOver(), companyInfo.getDaXiao().getInitialOdds().getOver());
                        historyMin.getDaXiao().getInitialOdds().setOver(minInitOver);
                        Double minInitUnder = Math.min(historyMin.getDaXiao().getInitialOdds().getUnder(), companyInfo.getDaXiao().getInitialOdds().getUnder());
                        historyMin.getDaXiao().getInitialOdds().setUnder(minInitUnder);
                        Double minInitHandicapDaXiao = Math.min(historyMin.getDaXiao().getInitialOdds().getHandicap(), companyInfo.getDaXiao().getInitialOdds().getHandicap());
                        historyMin.getDaXiao().getInitialOdds().setHandicap(minInitHandicapDaXiao);
                    }
                }
                if (companyInfo.getDaXiao().getNowOdds() != null) {
                    //max
                    Double maxNowOver = Math.max(historyMax.getDaXiao().getNowOdds().getOver(), companyInfo.getDaXiao().getNowOdds().getOver());
                    historyMax.getDaXiao().getNowOdds().setOver(maxNowOver);
                    Double maxNowUnder = Math.max(historyMax.getDaXiao().getNowOdds().getUnder(), companyInfo.getDaXiao().getNowOdds().getUnder());
                    historyMax.getDaXiao().getNowOdds().setUnder(maxNowUnder);
                    Double maxNowHandicapDaXiao = Math.max(historyMax.getDaXiao().getNowOdds().getHandicap(), companyInfo.getDaXiao().getNowOdds().getHandicap());
                    historyMax.getDaXiao().getNowOdds().setHandicap(maxNowHandicapDaXiao);
                    //min
                    if (historyMin.getDaXiao().getNowOdds() == null) {
                        historyMin.getDaXiao().setNowOdds(JSON.toJavaObject(JSONObject.parseObject(JSONObject.toJSONString(companyInfo.getDaXiao().getNowOdds())),
                                MatchFootballOddsResponse.FootballDaXiaoOdds.class));
                    } else {
                        Double minNowOver = Math.min(historyMin.getDaXiao().getNowOdds().getOver(), companyInfo.getDaXiao().getNowOdds().getOver());
                        historyMin.getDaXiao().getNowOdds().setOver(minNowOver);
                        Double minNowUnder = Math.min(historyMin.getDaXiao().getNowOdds().getUnder(), companyInfo.getDaXiao().getNowOdds().getUnder());
                        historyMin.getDaXiao().getNowOdds().setUnder(minNowUnder);
                        Double minNowHandicapDaXiao = Math.min(historyMin.getDaXiao().getNowOdds().getHandicap(), companyInfo.getDaXiao().getNowOdds().getHandicap());
                        historyMin.getDaXiao().getNowOdds().setHandicap(minNowHandicapDaXiao);
                    }
                }
            }
        }
        responses.add(MatchFootballOddsResponse.builder().companyInfos(Collections.singletonList(historyMax)).build());
        responses.add(MatchFootballOddsResponse.builder().companyInfos(Collections.singletonList(historyMin)).build());
    }

    @Override
    public ResponseData<MatchBasketballOddsResponse> matchBasketballOdds(CommonStringId commonIntId, HttpServletRequest request) {
        Match match = matchMapper.selectById(commonIntId.getId());
        if (match == null)
            return ResponseData.fail500Response("ID错误，比赛不存在");

        Sports sports = sportsMapper.selectById(match.getSportsId());
        if (sports.getSportsId() != 1)
            return ResponseData.fail500Response("此接口获取篮球比赛指数");

        List<MatchOdds> odds = matchOddsMapper.selectList(Wrappers.<MatchOdds>lambdaQuery()
                .eq(MatchOdds::getDeleted, StatusEnum.ENABLE.getCode())
                .eq(MatchOdds::getMatchId, commonIntId.getId())
        );

        Map<Integer, Map<Integer, List<MatchOdds>>> companyInfos = new HashMap<>();
        for (MatchOdds matchOdds : odds) {
            companyInfos.computeIfAbsent(matchOdds.getCompanyId(), k -> new HashMap<>())
                    .computeIfAbsent(matchOdds.getOddsType(), m -> new ArrayList<>()).add(matchOdds);
        }

        List<MatchBasketballOddsResponse> responses = new LinkedList<>();

        for (Integer companyId : companyInfos.keySet()) {
            IndexCompany indexCompany = indexCompanyMapper.selectById(companyId);
            if (indexCompany == null)
                continue;

            List<MatchBasketballOddsResponse.CompanyInfo> companyInfoList = new ArrayList<>();
            MatchBasketballOddsResponse.CompanyInfo responseInfo = MatchBasketballOddsResponse.CompanyInfo.builder()
                    .companyName(indexCompany.getCompanyName())
                    .country(indexCompany.getCountry())
                    .ouZhi(MatchBasketballOddsResponse.OuZhiOddsInfo.builder().build())
                    .ranFen(MatchBasketballOddsResponse.YaZhiOddsInfo.builder().build())
                    .daXiao(MatchBasketballOddsResponse.DaXiaoOddsInfo.builder().build())
                    .build();

            for (Integer oddsType : companyInfos.get(companyId).keySet()) {
                for (MatchOdds matchOdds : companyInfos.get(companyId).get(oddsType)) {
                    if (oddsType == 0) {
                        MatchBasketballOddsResponse.BasketballOdds basketball = MatchBasketballOddsResponse.BasketballOdds.builder()
                                .win(matchOdds.getFieldOne())
                                .type(matchOdds.getMatchType() == 0 ? "初" : "即")
                                .loss(matchOdds.getFieldThree())
                                .build();
                        if (matchOdds.getMatchType() == 0) {
                            responseInfo.getOuZhi().setInitialOdds(basketball);
                        } else {
                            responseInfo.getOuZhi().setNowOdds(basketball);
                        }
                    } else if (oddsType == 1) {
                        MatchBasketballOddsResponse.BasketballYaZhiOdds basketballOdds = MatchBasketballOddsResponse.BasketballYaZhiOdds.builder()
                                .type(matchOdds.getMatchType() == 0 ? "初" : "即")
                                .handicap(matchOdds.getFieldOne())
                                .home(matchOdds.getFieldTwo())
                                .away(matchOdds.getFieldThree())
                                .build();
                        if (matchOdds.getMatchType() == 0) {
                            responseInfo.getRanFen().setInitialOdds(basketballOdds);
                        } else {
                            responseInfo.getRanFen().setNowOdds(basketballOdds);
                        }
                    } else if (oddsType == 2) {
                        MatchBasketballOddsResponse.BasketballDaXiaoOdds basketballOdds = MatchBasketballOddsResponse.BasketballDaXiaoOdds.builder()
                                .type(matchOdds.getMatchType() == 0 ? "初" : "即")
                                .handicap(matchOdds.getFieldOne())
                                .over(matchOdds.getFieldTwo())
                                .under(matchOdds.getFieldThree())
                                .build();
                        if (matchOdds.getMatchType() == 0) {
                            responseInfo.getDaXiao().setInitialOdds(basketballOdds);
                        } else {
                            responseInfo.getDaXiao().setNowOdds(basketballOdds);
                        }
                    }
                }
            }
            companyInfoList.add(responseInfo);

            responses.add(MatchBasketballOddsResponse.builder()
                    .companyInfos(companyInfoList)
                    .build());
        }
        distinguishBasketballMaxAndMin(responses);
        return ResponseData.successResponse(responses);
    }

    private void distinguishBasketballMaxAndMin(List<MatchBasketballOddsResponse> companyInfos) {
        MatchBasketballOddsResponse.CompanyInfo historyMax = MatchBasketballOddsResponse.CompanyInfo.builder()
                .companyName("最高值")
                .country("")
                .daXiao(MatchBasketballOddsResponse.DaXiaoOddsInfo.builder()
                        .initialOdds(MatchBasketballOddsResponse.BasketballDaXiaoOdds.builder()
                                .handicap(0.00)
                                .over(0.00)
                                .type("初")
                                .under(0.00)
                                .build())
                        .nowOdds(MatchBasketballOddsResponse.BasketballDaXiaoOdds.builder()
                                .handicap(0.00)
                                .over(0.00)
                                .type("即")
                                .under(0.00)
                                .build())
                        .build())
                .ouZhi(MatchBasketballOddsResponse.OuZhiOddsInfo.builder()
                        .initialOdds(MatchBasketballOddsResponse.BasketballOdds.builder()
                                .loss(0.00)
                                .type("初")
                                .win(0.00)
                                .build())
                        .nowOdds(MatchBasketballOddsResponse.BasketballOdds.builder()
                                .win(0.00)
                                .type("即")
                                .loss(0.00)
                                .build())
                        .build())
                .ranFen(MatchBasketballOddsResponse.YaZhiOddsInfo.builder()
                        .initialOdds(MatchBasketballOddsResponse.BasketballYaZhiOdds.builder()
                                .away(0.00)
                                .handicap(0.00)
                                .home(0.00)
                                .type("初")
                                .build())
                        .nowOdds(MatchBasketballOddsResponse.BasketballYaZhiOdds.builder()
                                .away(0.00)
                                .handicap(0.00)
                                .home(0.00)
                                .type("即")
                                .build())
                        .build())
                .build();

        MatchBasketballOddsResponse.CompanyInfo historyMin = new MatchBasketballOddsResponse.CompanyInfo();

        AtomicBoolean init = new AtomicBoolean(true);
        for (MatchBasketballOddsResponse company : companyInfos) {
            for (MatchBasketballOddsResponse.CompanyInfo companyInfo : company.getCompanyInfos()) {
                if (init.get()) {
                    historyMin = JSON.toJavaObject(JSONObject.parseObject(JSONObject.toJSONString(companyInfo)), MatchBasketballOddsResponse.CompanyInfo.class);
                    historyMin.setCompanyName("最低值");
                    historyMin.setCountry("");
                    init.set(false);
                }

                //ouzhi
                if (companyInfo.getOuZhi().getInitialOdds() != null) {
                    //max
                    Double maxInitLoss = Math.max(historyMax.getOuZhi().getInitialOdds().getLoss(), companyInfo.getOuZhi().getInitialOdds().getLoss());
                    historyMax.getOuZhi().getInitialOdds().setLoss(maxInitLoss);
                    Double maxInitWin = Math.max(historyMax.getOuZhi().getInitialOdds().getWin(), companyInfo.getOuZhi().getInitialOdds().getWin());
                    historyMax.getOuZhi().getInitialOdds().setWin(maxInitWin);
                    //min
                    if (historyMin.getOuZhi().getInitialOdds() == null) {
                        historyMin.getOuZhi().setInitialOdds(JSON.toJavaObject(JSONObject.parseObject(JSONObject.toJSONString(companyInfo.getOuZhi().getInitialOdds())),
                                MatchBasketballOddsResponse.BasketballOdds.class));
                    } else {
                        Double minInitLoss = Math.min(historyMin.getOuZhi().getInitialOdds().getLoss(), companyInfo.getOuZhi().getInitialOdds().getLoss());
                        historyMin.getOuZhi().getInitialOdds().setLoss(minInitLoss);
                        Double minInitWin = Math.min(historyMin.getOuZhi().getInitialOdds().getWin(), companyInfo.getOuZhi().getInitialOdds().getWin());
                        historyMin.getOuZhi().getInitialOdds().setWin(minInitWin);
                    }
                }
                if (companyInfo.getOuZhi().getNowOdds() != null) {
                    //max
                    Double maxNowLoss = Math.max(historyMax.getOuZhi().getNowOdds().getLoss(), companyInfo.getOuZhi().getNowOdds().getLoss());
                    historyMax.getOuZhi().getNowOdds().setLoss(maxNowLoss);
                    Double maxNowWin = Math.max(historyMax.getOuZhi().getNowOdds().getWin(), companyInfo.getOuZhi().getNowOdds().getWin());
                    historyMax.getOuZhi().getNowOdds().setWin(maxNowWin);
                    //min
                    if (historyMin.getOuZhi().getNowOdds() == null) {
                        historyMin.getOuZhi().setNowOdds(JSON.toJavaObject(JSONObject.parseObject(JSONObject.toJSONString(companyInfo.getOuZhi().getNowOdds())),
                                MatchBasketballOddsResponse.BasketballOdds.class));
                    } else {
                        Double minNowLoss = Math.min(historyMin.getOuZhi().getNowOdds().getLoss(), companyInfo.getOuZhi().getNowOdds().getLoss());
                        historyMin.getOuZhi().getNowOdds().setLoss(minNowLoss);
                        Double minNowWin = Math.min(historyMin.getOuZhi().getNowOdds().getWin(), companyInfo.getOuZhi().getNowOdds().getWin());
                        historyMin.getOuZhi().getNowOdds().setWin(minNowWin);
                    }
                }
                //ranfen
                if (companyInfo.getRanFen().getInitialOdds() != null) {
                    //max
                    Double maxInitAway = Math.max(historyMax.getRanFen().getInitialOdds().getAway(), companyInfo.getRanFen().getInitialOdds().getAway());
                    historyMax.getRanFen().getInitialOdds().setAway(maxInitAway);
                    Double maxInitHome = Math.max(historyMax.getRanFen().getInitialOdds().getHome(), companyInfo.getRanFen().getInitialOdds().getHome());
                    historyMax.getRanFen().getInitialOdds().setHome(maxInitHome);
                    Double maxInitHandicap = Math.max(historyMax.getRanFen().getInitialOdds().getHandicap(), companyInfo.getRanFen().getInitialOdds().getHandicap());
                    historyMax.getRanFen().getInitialOdds().setHandicap(maxInitHandicap);
                    //min
                    if (historyMin.getRanFen().getInitialOdds() == null) {
                        historyMin.getRanFen().setInitialOdds(JSON.toJavaObject(JSONObject.parseObject(JSONObject.toJSONString(companyInfo.getRanFen().getInitialOdds())),
                                MatchBasketballOddsResponse.BasketballYaZhiOdds.class));
                    } else {
                        Double minInitAway = Math.min(historyMin.getRanFen().getInitialOdds().getAway(), companyInfo.getRanFen().getInitialOdds().getAway());
                        historyMin.getRanFen().getInitialOdds().setAway(minInitAway);
                        Double minInitHome = Math.min(historyMin.getRanFen().getInitialOdds().getHome(), companyInfo.getRanFen().getInitialOdds().getHome());
                        historyMin.getRanFen().getInitialOdds().setHome(minInitHome);
                        Double minInitHandicap = Math.min(historyMin.getRanFen().getInitialOdds().getHandicap(), companyInfo.getRanFen().getInitialOdds().getHandicap());
                        historyMin.getRanFen().getInitialOdds().setHandicap(minInitHandicap);
                    }
                }
                if (companyInfo.getRanFen().getNowOdds() != null) {
                    //max
                    Double maxNowAway = Math.max(historyMax.getRanFen().getNowOdds().getAway(), companyInfo.getRanFen().getNowOdds().getAway());
                    historyMax.getRanFen().getNowOdds().setAway(maxNowAway);
                    Double maxNowHome = Math.max(historyMax.getRanFen().getNowOdds().getHome(), companyInfo.getRanFen().getNowOdds().getHome());
                    historyMax.getRanFen().getNowOdds().setHome(maxNowHome);
                    Double maxNowHandicap = Math.max(historyMax.getRanFen().getNowOdds().getHandicap(), companyInfo.getRanFen().getNowOdds().getHandicap());
                    historyMax.getRanFen().getNowOdds().setHandicap(maxNowHandicap);
                    //min
                    if (historyMin.getRanFen().getNowOdds() == null) {
                        historyMin.getRanFen().setNowOdds(JSON.toJavaObject(JSONObject.parseObject(JSONObject.toJSONString(companyInfo.getRanFen().getNowOdds())),
                                MatchBasketballOddsResponse.BasketballYaZhiOdds.class));
                    } else {
                        Double minNowAway = Math.min(historyMin.getRanFen().getNowOdds().getAway(), companyInfo.getRanFen().getNowOdds().getAway());
                        historyMin.getRanFen().getNowOdds().setAway(minNowAway);
                        Double minNowHome = Math.min(historyMin.getRanFen().getNowOdds().getHome(), companyInfo.getRanFen().getNowOdds().getHome());
                        historyMin.getRanFen().getNowOdds().setHome(minNowHome);
                        Double minNowHandicap = Math.min(historyMin.getRanFen().getNowOdds().getHandicap(), companyInfo.getRanFen().getNowOdds().getHandicap());
                        historyMin.getRanFen().getNowOdds().setHandicap(minNowHandicap);
                    }
                }
                //daxiao
                if (companyInfo.getDaXiao().getInitialOdds() != null) {
                    //max
                    Double maxInitOver = Math.max(historyMax.getDaXiao().getInitialOdds().getOver(), companyInfo.getDaXiao().getInitialOdds().getOver());
                    historyMax.getDaXiao().getInitialOdds().setOver(maxInitOver);
                    Double maxInitUnder = Math.max(historyMax.getDaXiao().getInitialOdds().getUnder(), companyInfo.getDaXiao().getInitialOdds().getUnder());
                    historyMax.getDaXiao().getInitialOdds().setUnder(maxInitUnder);
                    Double maxInitHandicapDaXiao = Math.max(historyMax.getDaXiao().getInitialOdds().getHandicap(), companyInfo.getDaXiao().getInitialOdds().getHandicap());
                    historyMax.getDaXiao().getInitialOdds().setHandicap(maxInitHandicapDaXiao);
                    //min
                    if (historyMin.getDaXiao().getInitialOdds() == null) {
                        historyMin.getDaXiao().setInitialOdds(JSON.toJavaObject(JSONObject.parseObject(JSONObject.toJSONString(companyInfo.getDaXiao().getInitialOdds())),
                                MatchBasketballOddsResponse.BasketballDaXiaoOdds.class));
                    } else {
                        Double minInitOver = Math.min(historyMin.getDaXiao().getInitialOdds().getOver(), companyInfo.getDaXiao().getInitialOdds().getOver());
                        historyMin.getDaXiao().getInitialOdds().setOver(minInitOver);
                        Double minInitUnder = Math.min(historyMin.getDaXiao().getInitialOdds().getUnder(), companyInfo.getDaXiao().getInitialOdds().getUnder());
                        historyMin.getDaXiao().getInitialOdds().setUnder(minInitUnder);
                        Double minInitHandicapDaXiao = Math.min(historyMin.getDaXiao().getInitialOdds().getHandicap(), companyInfo.getDaXiao().getInitialOdds().getHandicap());
                        historyMin.getDaXiao().getInitialOdds().setHandicap(minInitHandicapDaXiao);
                    }
                }
                if (companyInfo.getDaXiao().getNowOdds() != null) {
                    //max
                    Double maxNowOver = Math.max(historyMax.getDaXiao().getNowOdds().getOver(), companyInfo.getDaXiao().getNowOdds().getOver());
                    historyMax.getDaXiao().getNowOdds().setOver(maxNowOver);
                    Double maxNowUnder = Math.max(historyMax.getDaXiao().getNowOdds().getUnder(), companyInfo.getDaXiao().getNowOdds().getUnder());
                    historyMax.getDaXiao().getNowOdds().setUnder(maxNowUnder);
                    Double maxNowHandicapDaXiao = Math.max(historyMax.getDaXiao().getNowOdds().getHandicap(), companyInfo.getDaXiao().getNowOdds().getHandicap());
                    historyMax.getDaXiao().getNowOdds().setHandicap(maxNowHandicapDaXiao);
                    //min
                    if (historyMin.getDaXiao().getNowOdds() == null) {
                        historyMin.getDaXiao().setNowOdds(JSON.toJavaObject(JSONObject.parseObject(JSONObject.toJSONString(companyInfo.getDaXiao().getNowOdds())),
                                MatchBasketballOddsResponse.BasketballDaXiaoOdds.class));
                    } else {
                        Double minNowOver = Math.min(historyMin.getDaXiao().getNowOdds().getOver(), companyInfo.getDaXiao().getNowOdds().getOver());
                        historyMin.getDaXiao().getNowOdds().setOver(minNowOver);
                        Double minNowUnder = Math.min(historyMin.getDaXiao().getNowOdds().getUnder(), companyInfo.getDaXiao().getNowOdds().getUnder());
                        historyMin.getDaXiao().getNowOdds().setUnder(minNowUnder);
                        Double minNowHandicapDaXiao = Math.min(historyMin.getDaXiao().getNowOdds().getHandicap(), companyInfo.getDaXiao().getNowOdds().getHandicap());
                        historyMin.getDaXiao().getNowOdds().setHandicap(minNowHandicapDaXiao);
                    }
                }
            }
        }
        companyInfos.add(MatchBasketballOddsResponse.builder().companyInfos(Collections.singletonList(historyMax)).build());
        companyInfos.add(MatchBasketballOddsResponse.builder().companyInfos(Collections.singletonList(historyMin)).build());
    }

    @Override
    public ResponseData<?> getMatchNewsById(GetMatchNewsByIdRequest getMatchNewsById, HttpServletRequest request) {
        Match match = matchMapper.selectById(getMatchNewsById.getMatchId());
        if (match == null)
            return ResponseData.fail400Response("赛事 ID 错误");

        Team home = teamMapper.selectById(match.getHomeId());
        Team away = teamMapper.selectById(match.getAwayId());

        Set<String> likes = new HashSet<>();
        likes.add(match.getCompetitionShortName());
        likes.add(match.getCompetitionName());
        likes.add(home.getTeamName());
        likes.add(away.getTeamName());

        ArticleByTitleLikeRequest tagIdRequest = new ArticleByTitleLikeRequest();
        tagIdRequest.setPageNum(getMatchNewsById.getPageNum());
        tagIdRequest.setPageSize(getMatchNewsById.getPageSize());
        tagIdRequest.setTitleLikes(likes);
        return articleService.getArticleListByTitleLike(tagIdRequest, request);
    }

    @Override
    public ResponseData<?> getHotRankInfoById(CommonIntId commonIntId, HttpServletRequest request) {
        QuizRecord quizRecord = quizRecordMapper.selectById(commonIntId.getId());
        if (quizRecord == null)
            return ResponseData.fail400Response("ID错误，未找到");

        User user = userMapper.selectById(quizRecord.getUserId());
        List<QuizRecord> userQuizRecords = quizRecordMapper.selectList(Wrappers.<QuizRecord>lambdaQuery()
                .eq(QuizRecord::getDeleted, 0)
                .eq(QuizRecord::getUserId, quizRecord.getUserId())
                .orderByDesc(QuizRecord::getCreateTime)
        );
        List<Match> matches = matchMapper.selectBatchIds(userQuizRecords.stream().map(QuizRecord::getMatchId).collect(Collectors.toList()));
        Map<String, Match> matchMap = matches.stream().collect(Collectors.toMap(Match::getId, b -> b));

        int continuousWin = 0;
        for (QuizRecord b : userQuizRecords) {
            Match lm = matchMap.get(b.getMatchId());
            if (lm != null && CommonUtil.winOrLose(lm, b.getBetPosition())) {
                continuousWin++;
                continue;
            }
            break;
        }
        int sumWin = 0;
        List<String> recentRecord = new LinkedList<>();
        for (QuizRecord b : userQuizRecords) {
            Match lm = matchMap.get(b.getMatchId());
            if (lm != null && CommonUtil.winOrLose(lm, b.getBetPosition())) {
                sumWin++;
                recentRecord.add("红");
                continue;
            }
            recentRecord.add("黑");
        }

        Match match = matchMapper.selectById(quizRecord.getMatchId());
        Sports sports = sportsMapper.selectById(match.getSportsId());
        Team home = teamMapper.selectById(match.getHomeId());
        Team away = teamMapper.selectById(match.getAwayId());

        List<MatchOdds> matchOdds = matchOddsMapper.selectList(Wrappers.<MatchOdds>lambdaQuery()
                .eq(MatchOdds::getDeleted, StatusEnum.ENABLE.getCode())
                .eq(MatchOdds::getMatchId, commonIntId.getId())
                .eq(MatchOdds::getMatchType, 1)
                .eq(MatchOdds::getOddsType, 0)
                .orderByAsc(MatchOdds::getCompanyId)
        );
        GetHotRankInfoByIdResponse.TeamIndex teamIndex = null;
        if (matchOdds.size() > 0) {
            teamIndex = GetHotRankInfoByIdResponse.TeamIndex.builder()
                    .wins(matchOdds.get(0).getFieldOne())
                    .draw(matchOdds.get(0).getFieldTwo())
                    .lose(matchOdds.get(0).getFieldThree())
                    .build();
        }

        long curHour = System.currentTimeMillis() / (60 * 60 * 1000);
        long creHour = quizRecord.getCreateTime().getTime() / (60 * 60 * 1000);

        GetHotRankInfoByIdResponse hotRankInfo = GetHotRankInfoByIdResponse.builder()
                .anchorDesc(user.getUserDesc())
                .anchorIcon(ConstantValue.completeAddress(user.getUserIcon()))
                .anchorId(user.getId())
                .anchorName(user.getUserName())
                .id(quizRecord.getId())
                .sum(userQuizRecords.size())
                .wins(sumWin)
                .continuousWin(continuousWin)
                .releaseTime(curHour - creHour + "小时前发布")
                .content(quizRecord.getContent())
                .title(quizRecord.getTitle())
                .winOrLose(CommonUtil.winOrLose(match, quizRecord.getBetPosition()))
                .winPosition(CommonUtil.winPosition(match))
                .betPosition(quizRecord.getBetPosition())
                .recentRecord(recentRecord)
                .teamVs(DateUtil.format(match.getMatchTime(), DateUtil.MDHM_) + " " + sports.getCompetitionShortName() + " " +
                        home.getTeamName() + " VS " + away.getTeamName())
                .matchOdds(teamIndex)
                .teamInfo(GetHotRankInfoByIdResponse.MatchList.builder()
                        .home(MatchListInfo.builder()
                                .score(match.getHomeScore())
                                .teamLogo(home.getTeamIcon())
                                .teamName(home.getTeamName())
                                .build())
                        .away(MatchListInfo.builder()
                                .score(match.getAwayScore())
                                .teamLogo(away.getTeamIcon())
                                .teamName(away.getTeamName())
                                .build())
                        .build())
                .build();
        return ResponseData.successResponse(hotRankInfo);
    }

    @Override
    public ResponseData<?> getHotRankList(GetMatchNewsByIdRequest matchNewsByIdRequest, HttpServletRequest request) {
        Match match = matchMapper.selectById(matchNewsByIdRequest.getMatchId());
        if (match == null)
            return ResponseData.successResponse(new ResultPage<>(matchNewsByIdRequest.getPageNum(), matchNewsByIdRequest.getPageSize(), 0, null));

        int jumpNum = (matchNewsByIdRequest.getPageNum() - 1) * matchNewsByIdRequest.getPageSize();

        long count = quizRecordMapper.selectCount(Wrappers.<QuizRecord>lambdaQuery()
                .eq(QuizRecord::getDeleted, 0)
                .eq(QuizRecord::getMatchId, matchNewsByIdRequest.getMatchId())
        );

        List<QuizRecord> quizRecords = quizRecordMapper.selectList(Wrappers.<QuizRecord>lambdaQuery()
                .eq(QuizRecord::getDeleted, 0)
                .eq(QuizRecord::getMatchId, matchNewsByIdRequest.getMatchId())
                .orderByDesc(QuizRecord::getCreateTime)
                .last(String.format(" limit %s,%s", jumpNum, matchNewsByIdRequest.getPageSize()))
        );

        List<HotRankDto> getHotRankList = new ArrayList<>();
        for (QuizRecord quizRecord : quizRecords) {
            User user = userMapper.selectById(quizRecord.getUserId());
            List<QuizRecord> userQuizRecords = quizRecordMapper.selectList(Wrappers.<QuizRecord>lambdaQuery()
                    .eq(QuizRecord::getDeleted, 0)
                    .eq(QuizRecord::getUserId, quizRecord.getUserId())
                    .orderByDesc(QuizRecord::getCreateTime)
            );
            List<Match> matches = matchMapper.selectBatchIds(userQuizRecords.stream().map(QuizRecord::getMatchId).collect(Collectors.toList()));
            Map<String, Match> matchMap = matches.stream().collect(Collectors.toMap(Match::getId, b -> b));

            int continuousWin = 0;
            for (QuizRecord b : userQuizRecords) {
                Match lm = matchMap.get(b.getMatchId());
                if (lm != null && CommonUtil.winOrLose(lm, b.getBetPosition())) {
                    continuousWin++;
                    continue;
                }
                break;
            }
            int sumWin = 0;
            for (QuizRecord b : userQuizRecords) {
                Match lm = matchMap.get(b.getMatchId());
                if (lm != null && CommonUtil.winOrLose(lm, b.getBetPosition())) {
                    sumWin++;
                }
            }

            long curHour = System.currentTimeMillis() / (60 * 60 * 1000);
            long creHour = quizRecord.getCreateTime().getTime() / (60 * 60 * 1000);

            getHotRankList.add(HotRankDto.builder()
                    .anchorDesc(user.getUserDesc())
                    .anchorIcon(ConstantValue.completeAddress(user.getUserIcon()))
                    .anchorId(user.getId())
                    .anchorName(user.getUserName())
                    .id(quizRecord.getId())
                    .sum(userQuizRecords.size())
                    .wins(sumWin)
                    .continuousWin(continuousWin)
                    .releaseTime(curHour - creHour + "小时前发布")
                    .title(quizRecord.getTitle())
                    .winOrLose(CommonUtil.winOrLose(match, quizRecord.getBetPosition()))
                    .betPosition(quizRecord.getBetPosition())
                    .build());
        }
        return ResponseData.successResponse(new ResultPage<>(matchNewsByIdRequest.getPageNum(), matchNewsByIdRequest.getPageSize(),
                (int) count, getHotRankList));
    }

    @Override
    public ResponseData<?> getHotRankAssociate(RankAssociateRequest commonIntId, HttpServletRequest request) {
        QuizRecord quizRecord = quizRecordMapper.selectById(commonIntId.getId());
        if (quizRecord == null)
            return ResponseData.fail400Response("ID错误，文章未找到");

        int jumpNum = (commonIntId.getPageNum() - 1) * commonIntId.getPageSize();

        long count = quizRecordMapper.selectCount(Wrappers.<QuizRecord>lambdaQuery()
                .eq(QuizRecord::getDeleted, 0)
                .ne(QuizRecord::getId, commonIntId.getId())
                .eq(QuizRecord::getMatchId, quizRecord.getMatchId())
        );

        List<QuizRecord> quizRecords = quizRecordMapper.selectList(Wrappers.<QuizRecord>lambdaQuery()
                .eq(QuizRecord::getDeleted, 0)
                .eq(QuizRecord::getMatchId, quizRecord.getMatchId())
                .ne(QuizRecord::getId, commonIntId.getId())
                .orderByDesc(QuizRecord::getCreateTime)
                .last(String.format(" limit %s,%s", jumpNum, commonIntId.getPageSize()))
        );

        Match match = matchMapper.selectById(quizRecord.getMatchId());
        Sports sports = sportsMapper.selectById(match.getSportsId());
        Team home = teamMapper.selectById(match.getHomeId());
        Team away = teamMapper.selectById(match.getAwayId());
        List<RankAssociateResponse> getHotRankList = new ArrayList<>();
        for (QuizRecord b : quizRecords) {
            long curHour = System.currentTimeMillis() / (60 * 60 * 1000);
            long creHour = quizRecord.getCreateTime().getTime() / (60 * 60 * 1000);

            getHotRankList.add(RankAssociateResponse.builder()
                    .id(b.getId())
                    .matchId(b.getMatchId())
                    .releaseTime(curHour - creHour + "小时前发布")
                    .title(b.getTitle())
                    .matchDesc(DateUtil.format(match.getMatchTime(), DateUtil.MDHM_) + " " + sports.getCompetitionShortName() + " " +
                            home.getTeamName() + " VS " + away.getTeamName())
                    .build());
        }
        return ResponseData.successResponse(new ResultPage<>(commonIntId.getPageNum(), commonIntId.getPageSize(),
                (int) count, getHotRankList));
    }

    @Override
    public ResponseData<?> backGetMatchList(CommonPage commonPage, HttpServletRequest request) {
        int jumpNum = (commonPage.getPageNum() - 1) * commonPage.getPageSize();

        QueryWrapper<Match> wrapper = new QueryWrapper<>();
        wrapper.eq("deleted", StatusEnum.ENABLE.getCode());
        wrapper.eq("source_type", 0);

        long count = matchMapper.selectCount(wrapper);

        wrapper.orderByDesc("match_time");
        wrapper.last(String.format(" limit %s,%s", jumpNum, commonPage.getPageSize()));

        List<Match> matches = matchMapper.selectList(wrapper);

        List<MatchInfoBackResponse> matchList = new ArrayList<>();
        for (Match match : matches) {
            try {
                Sports sports = sportsMapper.selectById(match.getSportsId());
                Team awayTeam = teamMapper.selectById(match.getAwayId());
                Team homeTeam = teamMapper.selectById(match.getHomeId());

                matchList.add(MatchInfoBackResponse.builder()
                        .id(match.getId())
                        .matchName(sports.getCompetitionName())
                        .sportsId(sports.getId())
                        .matchShortName(sports.getCompetitionShortName())
                        .sportsType(sports.getSportsId())
                        .status(match.getCompetitionStatus())
                        .matchDate(DateUtil.format(match.getMatchTime(), DateUtil.YMD_))
                        .matchTime(DateUtil.format(match.getMatchTime(), DateUtil.HM_))
                        .awayTeam(TeamResponse.builder()
                                .id(awayTeam.getId())
                                .teamIcon(awayTeam.getTeamIcon())
                                .teamName(awayTeam.getTeamName())
                                .score(match.getAwayScore())
                                .build())
                        .homeTeam(TeamResponse.builder()
                                .id(homeTeam.getId())
                                .teamIcon(homeTeam.getTeamIcon())
                                .teamName(homeTeam.getTeamName())
                                .score(match.getHomeScore())
                                .build())
                        .build());
            } catch (RuntimeException e) {
                e.printStackTrace();
            }
        }
        return ResponseData.successResponse(new ResultPage<>(commonPage.getPageNum(), commonPage.getPageSize(), (int) count, matchList));
    }

    @Override
    public ResponseData<?> getSportsList(HttpServletRequest request) {
        List<Sports> sports = sportsMapper.selectList(Wrappers.<Sports>lambdaQuery()
                .eq(Sports::getDeleted, StatusEnum.ENABLE.getCode())
        );

        List<SportsResponse> sportsResponses = sports.stream().map(b -> SportsResponse.builder()
                .competitionIcon(b.getCompetitionIcon())
                .competitionName(b.getCompetitionName())
                .competitionShortName(b.getCompetitionShortName())
                .sportsType(b.getSportsId())
                .sportsName(b.getSportsName())
                .id(b.getId())
                .build())
                .collect(Collectors.toList());
        return ResponseData.successResponse(sportsResponses);
    }

    @Override
    public ResponseData<?> addMatch(AddMatchRequest addMatchRequest, HttpServletRequest request) {
        Sports sports = sportsMapper.selectById(addMatchRequest.getSportsId());
        if (sports == null) {
            return ResponseData.fail400Response("SportsId 错误没有对应赛事类型");
        }

        Team home = teamMapper.selectById(addMatchRequest.getHomeId());
        Team away = teamMapper.selectById(addMatchRequest.getAwayId());
        if (home == null || away == null) {
            return ResponseData.fail400Response("Team 错误没有对应队伍");
        }

        String categoryId = "4";
        if (sports.getSportsId() == 0) {
            categoryId = "2";
        } else if (sports.getSportsId() == 1) {
            categoryId = "3";
        }

        matchMapper.insert(Match.builder()
                .awayId(addMatchRequest.getAwayId())
                .homeId(addMatchRequest.getHomeId())
                .categoryId(categoryId)
                .awayScore(addMatchRequest.getAwaySource())
                .homeScore(addMatchRequest.getHomeSource())
                .matchTime(DateUtil.parse(addMatchRequest.getStartTime(), DateUtil.YMDHM_))
                .sportsId(addMatchRequest.getSportsId())
                .competitionName(sports.getCompetitionName())
                .competitionShortName(sports.getCompetitionShortName())
                .build());
        return ResponseData.successResponse();
    }

    @Override
    public ResponseData<?> updateMatch(UpdateMatchRequest updateMatch, HttpServletRequest request) {
        Match match = matchMapper.selectById(updateMatch.getId());
        if (match == null) {
            return ResponseData.fail400Response("Id 错误没有对应赛程");
        }

        Sports sports = sportsMapper.selectById(updateMatch.getSportsId());
        if (sports == null) {
            return ResponseData.fail400Response("SportsId 错误没有对应赛事类型");
        }

        Team home = teamMapper.selectById(updateMatch.getHomeId());
        Team away = teamMapper.selectById(updateMatch.getAwayId());
        if (home == null || away == null) {
            return ResponseData.fail400Response("Team 错误没有对应队伍");
        }

        String categoryId = "4";
        if (sports.getSportsId() == 0) {
            categoryId = "2";
        } else if (sports.getSportsId() == 1) {
            categoryId = "3";
        }

        match.setAwayScore(updateMatch.getAwaySource());
        match.setAwayId(updateMatch.getAwayId());
        match.setHomeScore(updateMatch.getHomeSource());
        match.setHomeId(updateMatch.getHomeId());
        match.setCategoryId(categoryId);
        match.setSportsId(updateMatch.getSportsId());
        match.setMatchTime(DateUtil.parse(updateMatch.getStartTime(), DateUtil.YMDHM_));
        match.setDeleted(match.getDeleted());
        matchMapper.updateById(match);
        return ResponseData.successResponse();
    }

    @Override
    public ResponseData<?> getMatchLiveConfigList(MatchLiveListRequest commonPage, HttpServletRequest request) {
        int jumpNum = (commonPage.getPageNum() - 1) * commonPage.getPageSize();

        QueryWrapper<MatchLive> wrapper = new QueryWrapper<>();
        wrapper.eq("deleted", StatusEnum.ENABLE.getCode());
        if (StringUtils.isNotBlank(commonPage.getMatchId())) {
            wrapper.eq("match_id", commonPage.getMatchId());
        }

        long count = matchLiveMapper.selectCount(wrapper);

        wrapper.last(String.format(" limit %s,%s", jumpNum, commonPage.getPageSize()));
        List<MatchLive> matchLives = matchLiveMapper.selectList(wrapper);

        List<MatchLiveListResponse> responses = matchLives.stream().map(b -> MatchLiveListResponse.builder()
                .id(b.getId())
                .liveName(b.getSourceName())
                .liveUrl(b.getSourceUrl())
                .matchId(b.getMatchId())
                .createUser(b.getCreateBy())
                .deleted(b.getDeleted())
                .build())
                .collect(Collectors.toList());
        return ResponseData.successResponse(new ResultPage<>(commonPage.getPageNum(), commonPage.getPageSize(), (int) count, responses));
    }

    @Override
    public ResponseData<?> getHotRankListBack(HotRankListRequest commonPage, HttpServletRequest request) {
        int jumpNum = (commonPage.getPageNum() - 1) * commonPage.getPageSize();

        QueryWrapper<QuizRecord> wrapper = new QueryWrapper<>();
        wrapper.eq("deleted", StatusEnum.ENABLE.getCode());
        if (StringUtils.isNotBlank(commonPage.getMatchId())) {
            wrapper.eq("match_id", commonPage.getMatchId());
        }
        if (StringUtils.isNotBlank(commonPage.getUserId())) {
            wrapper.eq("user_id", commonPage.getUserId());
        }

        long count = quizRecordMapper.selectCount(wrapper);

        wrapper.orderByDesc("create_time");
        wrapper.last(String.format(" limit %s,%s", jumpNum, commonPage.getPageSize()));
        List<QuizRecord> quizRecords = quizRecordMapper.selectList(wrapper);

        List<QuizRecordBackResponse> responses = new LinkedList<>();
        for (QuizRecord quizRecord : quizRecords) {
            try {
                Match match = matchMapper.selectById(quizRecord.getMatchId());
                Team home = teamMapper.selectById(match.getHomeId());
                Team away = teamMapper.selectById(match.getAwayId());
                User user = userMapper.selectById(quizRecord.getUserId());

                responses.add(QuizRecordBackResponse.builder()
                        .content(quizRecord.getContent())
                        .betPosition(quizRecord.getBetPosition())
                        .matchId(quizRecord.getMatchId())
                        .title(quizRecord.getTitle())
                        .userId(quizRecord.getUserId())
                        .homeVsAway(home.getTeamName() + "VS" + away.getTeamName())
                        .id(quizRecord.getId())
                        .matchName(match.getCompetitionShortName())
                        .userName(user.getUserName())
                        .winOrLose(CommonUtil.winOrLose(match, quizRecord.getBetPosition()) ? "红" : "黑")
                        .deleted(quizRecord.getDeleted())
                        .build());
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return ResponseData.successResponse(new ResultPage<>(commonPage.getPageNum(), commonPage.getPageSize(), (int) count, responses));
    }


    @Override
    public ResponseData<?> editHotRankInfo(EditHotRankInfoRequest rankListRequest, HttpServletRequest request) {
        QuizRecord quizRecord = quizRecordMapper.selectById(rankListRequest.getId());
        if (quizRecord == null)
            quizRecord = new QuizRecord();

        quizRecord.setContent(rankListRequest.getContent());
        quizRecord.setBetPosition(rankListRequest.getBetPosition());
        quizRecord.setTitle(rankListRequest.getTitle());
        quizRecord.setMatchId(rankListRequest.getMatchId());
        quizRecord.setDeleted(rankListRequest.getDeleted());
        quizRecord.setUserId(rankListRequest.getUserId());

        if (quizRecord.getId() == null || quizRecord.getId() == 0) {
            quizRecord.setId(null);
            quizRecordMapper.insert(quizRecord);
        } else {
            quizRecordMapper.updateById(quizRecord);
        }
        return ResponseData.successResponse();
    }

    @Override
    public ResponseData<?> matchLiveConfig(MatchLiveConfigRequest matchLiveConfigRequest, HttpServletRequest request) {
        Match match = matchMapper.selectById(matchLiveConfigRequest.getMatchId());
        if (match == null)
            return ResponseData.fail400Response("MatchId 错误，未找到");

        for (MatchLiveConfigRequest.MatchLiveInfo matchLiveInfo : matchLiveConfigRequest.getLiveConfig()) {
            if (matchLiveInfo.getId() != null && matchLiveInfo.getId() != 0) {
                MatchLive matchLive = matchLiveMapper.selectById(matchLiveInfo.getId());
                if (matchLive == null)
                    return ResponseData.fail400Response("ID 错误，未找到");

                //删除
                if (matchLiveInfo.getDeleted() == 1) {
                    matchLiveMapper.deleteById(matchLive.getId());
                    continue;
                }

                //修改
                matchLive.setSourceUrl(matchLiveInfo.getLiveUrl());
                matchLive.setSourceName(matchLive.getSourceName());
                matchLive.setCreateBy(matchLive.getCreateBy());
                matchLive.setMatchId(matchLiveConfigRequest.getMatchId());
                matchLiveMapper.updateById(matchLive);
            } else {
                if (matchLiveInfo.getDeleted() == 1)
                    continue;

                if (StringUtils.isNotBlank(matchLiveInfo.getLiveUrl())) {
                    matchLiveMapper.insert(MatchLive.builder()
                            .matchId(matchLiveConfigRequest.getMatchId())
                            .sourceName(matchLiveInfo.getLiveName())
                            .sourceUrl(matchLiveInfo.getLiveUrl())
                            .createBy(matchLiveInfo.getCreateUser())
                            .build());
                }
            }
        }
        return ResponseData.successResponse();
    }

    /**
     * @param match
     * @param sportsType
     * @param awayTeam
     * @param homeTeam
     * @param otherAnchors true 展示关联主播 false 不展示
     * @return
     */
    private MatchListResponse.MatchInfoResponse conversionMatchInfo(Match match, Integer sportsType, Team awayTeam, Team homeTeam, boolean otherAnchors) {
        return MatchListResponse.MatchInfoResponse.builder()
                .id(match.getId())
                .matchName(match.getCompetitionName())
                .matchShortName(match.getCompetitionShortName())
                .sportsType(sportsType)
                .status(match.getCompetitionStatus())
                .matchDate(DateUtil.format(match.getMatchTime(), DateUtil.YMD_))
                .matchTime(DateUtil.format(match.getMatchTime(), DateUtil.HM_))
                .subscribe(false)
                .awayTeam(TeamResponse.builder()
                        .id(awayTeam.getId())
                        .teamIcon(awayTeam.getTeamIcon())
                        .teamName(awayTeam.getTeamName())
                        .score(match.getAwayScore())
                        .build())
                .homeTeam(TeamResponse.builder()
                        .id(homeTeam.getId())
                        .teamIcon(homeTeam.getTeamIcon())
                        .teamName(homeTeam.getTeamName())
                        .score(match.getHomeScore())
                        .build())
                .homeScore(match.getHomeScore())
                .awayScore(match.getAwayScore())
                .build();
    }

    private List<MatchListResponse.GuideResponse> guidesByMatchId(String matchId) {
        List<MatchListResponse.GuideResponse> guideResponseList = new ArrayList<>();
        List<RoomHostRecord> anchors = roomHostRecordMapper.selectList(Wrappers.<RoomHostRecord>lambdaQuery()
                .eq(RoomHostRecord::getMatchId, matchId)
                .eq(RoomHostRecord::getDeleted, 0)
        );
        for (RoomHostRecord anchorMatch : anchors) {
            User user = userMapper.selectById(anchorMatch.getHostUserId());
            Room room = roomMapper.selectById(anchorMatch.getRoomId());

            guideResponseList.add(MatchListResponse.GuideResponse.builder()
                    .avatar(ConstantValue.completeAddress(user.getUserIcon()))
                    .id(user.getId())
                    .name(user.getUserName())
                    .roomId(room.getHuanXinRoomId())
                    .status(room.getType())
                    .recommend(room.getRecommend())
                    .matchCutImg(ConstantValue.completeAddress(room.getRoomCover()))
                    .build());
        }
        return guideResponseList;
    }

    private MatchInfoNewResponse.PlayerCompetitionInfo conversionPlayer(PlayerCompetition playerCompetition) {
        return MatchInfoNewResponse.PlayerCompetitionInfo.builder()
                .isFirst(playerCompetition.getAbsentReason() == null ? 1 : Integer.parseInt(playerCompetition.getAbsentReason()))
                .assists(playerCompetition.getAssists())
                .attackBoard(playerCompetition.getAttackBoard())
                .blocks(playerCompetition.getBlocks())
                .defensiveBoard(playerCompetition.getDefensiveBoard())
                .error(playerCompetition.getError())
                .foul(playerCompetition.getFoul())
                .freeThrowNum(playerCompetition.getFreeThrowNum())
                .freeThrows(playerCompetition.getFreeThrows())
                .playerName(playerCompetition.getPlayerName())
                .playingTime(playerCompetition.getPlayingTime())
                .score(playerCompetition.getScore())
                .shootNum(playerCompetition.getShootNum())
                .shots(playerCompetition.getShots())
                .steals(playerCompetition.getSteals())
                .threeShotNum(playerCompetition.getThreeShotNum())
                .threeShots(playerCompetition.getThreeShots())
                .totalBoards(playerCompetition.getTotalBoards())
                .twoScore(playerCompetition.getTwoScore())
                .build();
    }

    private MatchInfoNewResponse.TeamCompetitionInfo conversion(TeamCompetition teamCompetition) {
        return MatchInfoNewResponse.TeamCompetitionInfo.builder()
                .assists(teamCompetition.getAssists())
                .attackBoard(teamCompetition.getAttackBoard())
                .blocks(teamCompetition.getBlocks())
                .defensiveBoard(teamCompetition.getDefensiveBoard())
                .error(teamCompetition.getError())
                .foul(teamCompetition.getFoul())
                .freeThrowNum(teamCompetition.getFreeThrowNum())
                .freeThrows(teamCompetition.getFreeThrows())
                .homeOrAway(teamCompetition.getHomeOrAway() == 0 ? "主场" : "客场")
                .result(teamCompetition.getResult() == 0 ? "胜" : "负")
                .score(teamCompetition.getScore())
                .scoreSettle(teamCompetition.getScoreSettle())
                .shootNum(teamCompetition.getShootNum())
                .shots(teamCompetition.getShots())
                .steals(teamCompetition.getSteals())
                .threeShotNum(teamCompetition.getThreeShotNum())
                .threeShots(teamCompetition.getThreeShots())
                .totalBoards(teamCompetition.getTotalBoards())
                .twoScore(teamCompetition.getTwoScore())
                .build();
    }
}
