package com.live.common.service.impl;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.toolkit.StringUtils;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.live.common.constant.ConstantValue;
import com.live.common.domain.ResponseData;
import com.live.common.domain.ResultPage;
import com.live.common.domain.dto.api.AnnouncementResponse;
import com.live.common.domain.dto.api.ShareConfigResponse;
import com.live.common.domain.dto.api.VersionResponse;
import com.live.common.domain.entity.Announcement;
import com.live.common.domain.entity.Room;
import com.live.common.domain.entity.SystemConfig;
import com.live.common.domain.entity.Version;
import com.live.common.domain.request.CommonPage;
import com.live.common.domain.request.ImageCallbackRequest;
import com.live.common.domain.request.VersionRequest;
import com.live.common.enums.SystemConfigEnum;
import com.live.common.mapper.AnnouncementMapper;
import com.live.common.mapper.RoomMapper;
import com.live.common.mapper.SystemConfigMapper;
import com.live.common.mapper.VersionMapper;
import com.live.common.service.CommonService;
import com.live.common.utils.MD5Util;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.util.List;
import java.util.stream.Collectors;

@Slf4j
@Service
public class CommonServiceImpl implements CommonService {

    @Resource
    private RoomMapper roomMapper;
    @Resource
    private VersionMapper versionMapper;
    @Resource
    private SystemConfigMapper systemConfigMapper;
    @Resource
    private AnnouncementMapper announcementMapper;

    @Override
    public ResponseData<?> checkVersion(VersionRequest version, HttpServletRequest request) {
        VersionResponse versionResponse = new VersionResponse();
        if (StringUtils.isBlank(version.getClientName()) || StringUtils.isBlank(version.getVersion())) {
            return ResponseData.fail400Response("字段不能为空");
        }

        List<Version> versions = versionMapper.selectList(Wrappers.<Version>lambdaQuery()
                .eq(Version::getClientName, version.getClientName())
        );

        if (versions.size() == 0) {
            return ResponseData.fail400Response("没有该 ClientName 数据");
        }

        int id = 0;
        int maxId = 0;
        for (Version vs : versions) {
            if (version.getVersion().equals(vs.getVersion())) {
                id = vs.getId();
            }
            if (vs.getId() > maxId) {
                maxId = vs.getId();

                versionResponse.setClientName(vs.getClientName());
                versionResponse.setVersion(vs.getVersion());
                versionResponse.setIntroduction(vs.getIntroduction());
                versionResponse.setDownloadLink(vs.getDownloadLink());
            }
        }

        if (id == 0) {
            return ResponseData.fail500Response("未知版本号");
        }

        versionResponse.setForceUpdate(false);
        for (Version vs : versions) {
            if (vs.getId() >= id && vs.getForceUpdate() == 1) {
                versionResponse.setForceUpdate(true);
                break;
            }
        }
        return ResponseData.successResponse(versionResponse);
    }

    @Override
    public ResponseData<?> getAvatarUrl(HttpServletRequest request) {
        return ResponseData.successResponse(ConstantValue.AVATAR_URL);
    }

    @Override
    public ResponseData<?> getMessageList(CommonPage commonPage, HttpServletRequest request) {
        int jumpNum = (commonPage.getPageNum() - 1) * commonPage.getPageSize();

        long count = announcementMapper.selectCount(Wrappers.<Announcement>lambdaQuery()
                .eq(Announcement::getDeleted, 0)
        );

        List<Announcement> announcements = announcementMapper.selectList(Wrappers.<Announcement>lambdaQuery()
                .eq(Announcement::getDeleted, 0)
                .orderByDesc(Announcement::getCreateTime)
                .last(String.format(" limit %s,%s", jumpNum, commonPage.getPageSize()))
        );

        return ResponseData.successResponse(new ResultPage<>(commonPage.getPageNum(), commonPage.getPageSize(), (int) count,
                announcements.stream().map(b -> AnnouncementResponse.builder()
                        .id(b.getId())
                        .content(b.getContent())
                        .title(b.getTitle())
                        .build()).collect(Collectors.toList())));
    }

    @Override
    public ResponseData<?> getShareConfig(HttpServletRequest request) {
        SystemConfig shareTitle = systemConfigMapper.selectOne(Wrappers.<SystemConfig>lambdaQuery()
                .eq(SystemConfig::getDeleted, 0)
                .eq(SystemConfig::getConfigName, SystemConfigEnum.SHARE_TITLE)
        );
        SystemConfig shareContent = systemConfigMapper.selectOne(Wrappers.<SystemConfig>lambdaQuery()
                .eq(SystemConfig::getDeleted, 0)
                .eq(SystemConfig::getConfigName, SystemConfigEnum.SHARE_CONTENT)
        );

        return ResponseData.successResponse(ShareConfigResponse.builder()
                .shareTitle(shareTitle.getConfigValue())
                .shareContent(shareContent.getConfigValue())
                .build());
    }

    @Override
    public void imageCallback(ImageCallbackRequest callbackRequest, HttpServletRequest request) {
        String callbackKey = "livesKBcsk2GYK5421jpg";
        String sign = MD5Util.string2MD5(callbackKey + callbackRequest.getT());
        if(sign.equals(callbackRequest.getSign())){
            Room room = roomMapper.selectOne(Wrappers.<Room>lambdaQuery()
                    .eq(Room::getDeleted, 0)
                    .eq(Room::getUserId, callbackRequest.getChannel_id())
            );
            if(room != null && room.getUseCutImg() == 1 && room.getType() == 1){
                room.setRoomCover(callbackRequest.getPic_url().replace("/live/", ""));
                roomMapper.updateById(room);
            }
        } else {
            log.warn("直播截图验签不通过:{}", JSON.toJSONString(callbackRequest));
        }
    }

}
