package com.live.server.controller;

import com.live.common.domain.ResponseData;
import com.live.common.domain.request.CommonIntId;
import com.live.common.domain.request.CommonStringId;
import com.live.common.domain.request.SendGiftRequest;
import com.live.common.service.RoomService;
import io.swagger.annotations.*;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;

@Slf4j
@CrossOrigin
@RestController
@RequestMapping("/room")
@Api(tags = "直播房间接口 API 文档")
public class RoomController {

    @Resource
    private RoomService roomService;

    @PostMapping(value = "/getRoomInfo")
    @ApiOperation(value = "根据房间 id 根据获取主播信息")
    @ApiResponses({
            @ApiResponse(code = 200, message = "成功处理请求"),
            @ApiResponse(code = 401, message = "没有权限访问该服务"),
            @ApiResponse(code = 403, message = "权限不足无法访问该服务"),
            @ApiResponse(code = 404, message = "未发现该服务"),
            @ApiResponse(code = 500, message = "服务器内部错误")
    })
    public ResponseData<?> getRoomInfo(@RequestBody CommonStringId commonStringId, HttpServletRequest request) {
        return roomService.getRoomInfo(commonStringId, request);
    }

    @PostMapping(value = "/getAnchorMatch")
    @ApiOperation(value = "根据房间 id 获取主播赛程(可带可不带token，带token判断预约)")
    @ApiResponses({
            @ApiResponse(code = 200, message = "成功处理请求"),
            @ApiResponse(code = 401, message = "没有权限访问该服务"),
            @ApiResponse(code = 403, message = "权限不足无法访问该服务"),
            @ApiResponse(code = 404, message = "未发现该服务"),
            @ApiResponse(code = 500, message = "服务器内部错误")
    })
    @ApiImplicitParams({
            @ApiImplicitParam(name = "token", value = "登录凭证(登录后 token 值)", dataType = "String", paramType = "header", required = true)
    })
    public ResponseData<?> getAnchorMatchList(@RequestBody CommonStringId stringId, HttpServletRequest request) {
        return roomService.anchorMatch(stringId, request);
    }

    @PostMapping(value = "/leaderboard")
    @ApiOperation(value = "根据房间 id 获取主播排行榜")
    @ApiResponses({
            @ApiResponse(code = 200, message = "成功处理请求"),
            @ApiResponse(code = 401, message = "没有权限访问该服务"),
            @ApiResponse(code = 403, message = "权限不足无法访问该服务"),
            @ApiResponse(code = 404, message = "未发现该服务"),
            @ApiResponse(code = 500, message = "服务器内部错误")
    })
    public ResponseData<?> leaderboard(@RequestBody CommonStringId commonStringId, HttpServletRequest request) {
        return roomService.leaderboard(commonStringId);
    }

    @PostMapping(value = "/getGiftList")
    @ApiOperation(value = "获取礼物列表")
    @ApiResponses({
            @ApiResponse(code = 200, message = "成功处理请求"),
            @ApiResponse(code = 401, message = "没有权限访问该服务"),
            @ApiResponse(code = 403, message = "权限不足无法访问该服务"),
            @ApiResponse(code = 404, message = "未发现该服务"),
            @ApiResponse(code = 500, message = "服务器内部错误")
    })
    public ResponseData<?> getGiftList(HttpServletRequest request) {
        return roomService.getGiftList(request);
    }

    @PostMapping(value = "/sendGift")
    @ApiOperation(value = "赠送礼物")
    @ApiResponses({
            @ApiResponse(code = 200, message = "成功处理请求"),
            @ApiResponse(code = 401, message = "没有权限访问该服务"),
            @ApiResponse(code = 403, message = "权限不足无法访问该服务"),
            @ApiResponse(code = 404, message = "未发现该服务"),
            @ApiResponse(code = 500, message = "服务器内部错误")
    })
    @ApiImplicitParams({
            @ApiImplicitParam(name = "token", value = "登录凭证(登录后 token 值)", dataType = "String", paramType = "header", required = true)
    })
    public ResponseData<?> sendGift(@RequestBody SendGiftRequest sendGiftRequest, HttpServletRequest request) {
        return roomService.sendGift(sendGiftRequest, request);
    }

    @PostMapping(value = "/getHotAnchor")
    @ApiOperation(value = "PC-获取热门主播列表")
    @ApiResponses({
            @ApiResponse(code = 200, message = "成功处理请求"),
            @ApiResponse(code = 401, message = "没有权限访问该服务"),
            @ApiResponse(code = 403, message = "权限不足无法访问该服务"),
            @ApiResponse(code = 404, message = "未发现该服务"),
            @ApiResponse(code = 500, message = "服务器内部错误")
    })
    public ResponseData<?> getHotAnchor(HttpServletRequest request) {
        return roomService.getHotAnchor(request);
    }

    @PostMapping(value = "/getChatMessage")
    @ApiOperation(value = "获取假聊天消息")
    @ApiResponses({
            @ApiResponse(code = 200, message = "成功处理请求"),
            @ApiResponse(code = 401, message = "没有权限访问该服务"),
            @ApiResponse(code = 403, message = "权限不足无法访问该服务"),
            @ApiResponse(code = 404, message = "未发现该服务"),
            @ApiResponse(code = 500, message = "服务器内部错误")
    })
    public ResponseData<?> getChatMessage(HttpServletRequest request) {
        return roomService.getChatMessage(request);
    }
}
