package com.live.common.service.impl;

import com.baomidou.mybatisplus.core.toolkit.StringUtils;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.live.common.constant.ConstantValue;
import com.live.common.constant.RedisKeySplicing;
import com.live.common.domain.ResponseData;
import com.live.common.domain.dto.api.LiveMatchTypeResponse;
import com.live.common.domain.dto.api.PushUrlResponse;
import com.live.common.domain.dto.api.RoomInfoByUserResponse;
import com.live.common.domain.entity.*;
import com.live.common.domain.request.CommonStringId;
import com.live.common.domain.request.CreateRoomRequest;
import com.live.common.domain.request.HXRoomRequest;
import com.live.common.enums.StatusEnum;
import com.live.common.exception.ExceptionCode;
import com.live.common.mapper.*;
import com.live.common.service.HuanXinIMService;
import com.live.common.service.LiveService;
import com.live.common.service.RedisUtilsService;
import com.live.common.utils.*;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.stream.Collectors;

import static com.live.common.utils.TencentYunVideoUtil.*;

@Slf4j
@Service
public class LiveServiceImpl implements LiveService {

    @Resource
    private UserMapper userMapper;
    @Resource
    private RoomMapper roomMapper;
    @Resource
    private MatchMapper matchMapper;
    @Resource
    private CategoryMapper categoryMapper;
    @Resource
    private HuanXinIMService huanXinIMService;
    @Resource
    private RedisUtilsService redisUtilsService;
    @Resource
    private RoomHostRecordMapper roomHostRecordMapper;

    @Value("${tencent.pushKey}")
    private String push_key;
    @Value("${tencent.playKey}")
    private String play_key;
    @Value("${tencent.pushUrl}")
    private String pushUrl;
    @Value("${tencent.playUrl}")
    private String playUrl;
    @Value("${surroundings}")
    private String surroundings;

    @Override
    public ResponseData<?> getMatchListByType(CommonStringId commonStringId, HttpServletRequest request) {
        List<Match> matches = matchMapper.getMatchListByCategory(commonStringId.getId(), DateUtil.format(new Date(), DateUtil.YMDHMS_));

        List<LiveMatchTypeResponse> matchTypeResponses = matches.stream().map(b -> LiveMatchTypeResponse.builder()
                .matchId(b.getId())
                .matchName(b.getCompetitionName())
                .matchShortName(b.getCompetitionShortName())
                .date(DateUtil.format(b.getMatchTime(), DateUtil.YMD_))
                .time(DateUtil.format(b.getMatchTime(), DateUtil.HM_))
                .build()).collect(Collectors.toList());
        return ResponseData.successResponse(matchTypeResponses);
    }

    @Override
    public ResponseData<?> getRoomInfoByUser(HttpServletRequest request) {
        String phone = CommonMethod.getUserPhone(request);
        User account = userMapper.selectOne(Wrappers.<User>lambdaQuery()
                .eq(User::getPhone, phone)
        );

        if (account.getType() == 0) {
            return ResponseData.failResponse(ExceptionCode.NO_PERMISSION_ACCESS, "请先申请成为主播");
        }

        List<Room> rooms = roomMapper.selectList(Wrappers.<Room>lambdaQuery()
                .eq(Room::getUserId, account.getId())
                .eq(Room::getDeleted, 0)
        );
        if (rooms.size() == 0) {
            return ResponseData.successResponse();
        }

        Room room = rooms.get(0);
        Category category = categoryMapper.selectById(room.getCategoryId());
        String categoryName = null;
        if (category != null) {
            categoryName = category.getSportsName();
        }

        RoomInfoByUserResponse response = RoomInfoByUserResponse.builder()
                .roomId(room.getHuanXinRoomId())
                .roomCutImg(ConstantValue.completeAddress(room.getRoomCover()))
                .roomAnnouncement(room.getAnnouncement())
                .roomName(room.getRoomName())
                .matchTypeName(categoryName)
                .matchId(room.getMatchId())
                .matchType(convertCategorys(room.getCategoryId()))
                .status(room.getType())
                .roomNum(room.getRoomNum())
                .build();
        return ResponseData.successResponse(response);
    }

    public static Integer convertCategorys(Integer type) {
        switch (type) {
            case 2:
                return 0;
            case 3:
                return 1;
            default:
                return 2;
        }
    }

    @Override
    public ResponseData<?> createRoom(CreateRoomRequest roomRequest, HttpServletRequest request) {
        String phone = CommonMethod.getUserPhone(request);
        User account = userMapper.selectOne(Wrappers.<User>lambdaQuery()
                .eq(User::getPhone, phone)
        );

        if (account.getType() == 0) {
            return ResponseData.failResponse(ExceptionCode.NO_PERMISSION_ACCESS, "请先申请成为主播");
        }

        String key = RedisKeySplicing.createRoom(phone);
        if (StringUtils.isBlank(redisUtilsService.getCacheStringInfo(key))) {
            redisUtilsService.cacheStringInfo(key, "1", 3);
            Room room = roomMapper.selectOne(Wrappers.<Room>lambdaQuery()
                    .eq(Room::getUserId, account.getId())
                    .eq(Room::getType, 1)
                    .eq(Room::getDeleted, StatusEnum.ENABLE.getCode())
            );
            if (room != null) {
                return ResponseData.fail400Response("只能有一个房间");
            }
            createRoomLast(roomRequest, account, createPlayUrlById(account.getId()));
            redisUtilsService.removeByKey(key);
        } else {
            return ResponseData.failResponse(ExceptionCode.NO_PERMISSION_ACCESS, "正在创建房间,请稍等");
        }
        return ResponseData.successResponse();
    }

    @Override
    public ResponseData<?> createRoom(CreateRoomRequest roomRequest, String userId, String liveUrl) {
        User account = userMapper.selectById(userId);

        if (account.getType() == 0) {
            return ResponseData.failResponse(ExceptionCode.NO_PERMISSION_ACCESS, "请先申请成为主播");
        }

        Room room = roomMapper.selectOne(Wrappers.<Room>lambdaQuery()
                .eq(Room::getUserId, account.getId())
                .eq(Room::getDeleted, StatusEnum.ENABLE.getCode())
        );
        if (room != null) {
            return ResponseData.fail400Response("只能有一个房间");
        }
        createRoomLast(roomRequest, account, liveUrl);
        return ResponseData.successResponse();
    }

    @Transactional
    public void createRoomLast(CreateRoomRequest roomRequest, User account, String liveUrl) {
        String matchId = StringUtils.isBlank(roomRequest.getMatchId()) ? null : roomRequest.getMatchId();

        String userName = "live_" + account.getId() + surroundings;

        String huanXinRoomId = huanXinIMService.createRoom(HXRoomRequest.builder()
                .description(roomRequest.getRoomAnnouncement())
                .name(roomRequest.getRoomName())
                .owner(userName)
                .build());

        String roomNum = null;
        do {
            roomNum = IdGen.randomLong(5);
        } while (roomMapper.selectCount(Wrappers.<Room>lambdaQuery()
                .eq(Room::getRoomNum, roomNum)
        ) > 0);

        roomMapper.insert(Room.builder()
                .huanXinRoomId(huanXinRoomId)
                .roomNum(roomNum)
                .announcement(roomRequest.getRoomAnnouncement())
                .categoryId(CommonMethod.convertCategory(roomRequest.getType()))
                .roomCover(roomRequest.getRoomCutImg())
                .roomName(roomRequest.getRoomName())
                .matchId(matchId)
                .useCutImg(roomRequest.getUseCutImg())
                .userId(account.getId())
                .liveUrl(liveUrl)
                .source(0)
                .hotNum(IdGen.generatorNum(20000,60000))
                .build());

        if (matchId != null) {
            Match match = matchMapper.selectById(matchId);

            roomHostRecordMapper.insert(RoomHostRecord.builder()
                    .hostUserId(account.getId())
                    .endTime(DateUtil.addDate(match.getMatchTime(), 20, Calendar.HOUR))
                    .startTime(match.getMatchTime())
                    .matchId(match.getId())
                    .matchTime(match.getMatchTime())
                    .roomId(huanXinRoomId)
                    .receivedGold(0)
                    .build());
        }
    }

    @Override
    public ResponseData<?> editRoom(CreateRoomRequest roomRequest, HttpServletRequest request) {
        String phone = CommonMethod.getUserPhone(request);
        User account = userMapper.selectOne(Wrappers.<User>lambdaQuery()
                .eq(User::getPhone, phone)
        );

        if (account.getType() == 0) {
            return ResponseData.failResponse(ExceptionCode.NO_PERMISSION_ACCESS, "请先申请成为主播");
        }

        Room room = roomMapper.selectOne(Wrappers.<Room>lambdaQuery()
                .eq(Room::getUserId, account.getId())
                .eq(Room::getHuanXinRoomId, roomRequest.getRoomId())
                .eq(Room::getDeleted, StatusEnum.ENABLE.getCode())
        );
        if (room == null) {
            return ResponseData.fail400Response("请先创建一个房间");
        }

        room.setAnnouncement(roomRequest.getRoomAnnouncement());
        room.setCategoryId(roomRequest.getType());
        room.setMatchId(roomRequest.getMatchId());
        room.setRoomCover(roomRequest.getRoomCutImg());
        room.setRoomName(roomRequest.getRoomName());
        room.setType(roomRequest.getStatus());
        room.setDeleted(roomRequest.getDeleted());
        roomMapper.updateById(room);

        if (roomRequest.getDeleted() == 1) {
            roomHostRecordMapper.delete(Wrappers.<RoomHostRecord>lambdaQuery()
                    .eq(RoomHostRecord::getRoomId, roomRequest.getRoomId())
                    .eq(RoomHostRecord::getHostUserId, account.getId())
            );
        }
        return ResponseData.successResponse();
    }

    @Override
    public ResponseData<?> createPushUrl(HttpServletRequest request) {
        String phone = CommonMethod.getUserPhone(request);
        User account = userMapper.selectOne(Wrappers.<User>lambdaQuery()
                .eq(User::getPhone, phone)
        );

        if (account.getType() == 0) {
            return ResponseData.failResponse(ExceptionCode.NO_PERMISSION_ACCESS, "请先申请成为主播");
        }

        Long time = DateUtil.getTmEndTime().getTime() / 1000;
        StringBuffer sb = new StringBuffer();
        sb.append(account.getId()).append("?").append(SafeUrl.getSafeUrl(push_key, account.getId(), time));

        return ResponseData.successResponse(PushUrlResponse.builder()
                .serverUrl(pushUrl)
                .userPushUrl(sb.toString())
                .build());
    }

    @Override
    public ResponseData<?> createAliPushUrl(HttpServletRequest request) {
        String phone = CommonMethod.getUserPhone(request);
        User account = userMapper.selectOne(Wrappers.<User>lambdaQuery()
                .eq(User::getPhone, phone)
        );

        if (account.getType() == 0) {
            return ResponseData.failResponse(ExceptionCode.NO_PERMISSION_ACCESS, "请先申请成为主播");
        }


        return null;
    }

    @Override
    public ResponseData<?> createPlayUrl(HttpServletRequest request) {
        String phone = CommonMethod.getUserPhone(request);
        User account = userMapper.selectOne(Wrappers.<User>lambdaQuery()
                .eq(User::getPhone, phone)
        );

        if (account.getType() == 0) {
            return ResponseData.failResponse(ExceptionCode.NO_PERMISSION_ACCESS, "请先申请成为主播");
        }
        return ResponseData.successResponse(createPlayUrlById(account.getId()));
    }

    private String createPlayUrlById(String userId) {
        Long time = DateUtil.getTmEndTime().getTime() / 1000;
        StringBuffer sb = new StringBuffer();
        sb.append(playUrl + userId + ".m3u8").append("?")
                .append(SafeUrl.getSafeUrl(play_key, userId, time));
        return sb.toString();
    }

}
