package com.live.common.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.live.common.constant.ConstantValue;
import com.live.common.constant.RedisKeySplicing;
import com.live.common.domain.ResponseData;
import com.live.common.domain.ResultPage;
import com.live.common.domain.dto.PushStreamDTO;
import com.live.common.domain.dto.api.*;
import com.live.common.domain.dto.back.GetRoomListResponse;
import com.live.common.domain.entity.*;
import com.live.common.domain.request.*;
import com.live.common.domain.request.back.CreateAccountRequest;
import com.live.common.domain.request.back.EditRobotMsgRequest;
import com.live.common.domain.request.back.EditRoomRequest;
import com.live.common.domain.request.back.GetRoomListRequest;
import com.live.common.enums.SystemConfigEnum;
import com.live.common.mapper.*;
import com.live.common.service.*;
import com.live.common.utils.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.util.*;
import java.util.stream.Collectors;

import static com.live.common.exception.ExceptionCode.INSUFFICIENT_USER_COINS;

@Slf4j
@Service
public class RoomServiceImpl implements RoomService {

    @Resource
    private RoomMapper roomMapper;
    @Resource
    private UserMapper userMapper;
    @Resource
    private GiftMapper giftMapper;
    @Resource
    private MatchMapper matchMapper;
    @Resource
    private RobotMsgMapper robotMsgMapper;
    @Resource
    private UserService userService;
    @Resource
    private LiveService liveService;
    @Resource
    private FollowMapper followMapper;
    @Resource
    private AdvertMapper advertMapper;
    @Resource
    private MatchService matchService;
    @Resource
    private CategoryMapper categoryMapper;
    @Resource
    private UserGoldMapper userGoldMapper;
    @Resource
    private GiftRecordMapper giftRecordMapper;
    @Resource
    private RedisUtilsService redisUtilsService;
    @Resource
    private UserReserveMapper userReserveMapper;
    @Resource
    private SystemConfigMapper systemConfigMapper;
    @Resource
    private RoomHostRecordMapper roomHostRecordMapper;
    @Value("${surroundings}")
    private String surroundings;

//    @Override
//    public ResponseData<?> getRoomInfo(CommonStringId commonStringId, HttpServletRequest request) {
//        if (StringUtils.isBlank(commonStringId.getId())) {
//            return ResponseData.fail400Response("请传入房间ID");
//        }
//
//        Room room = roomMapper.selectById(commonStringId.getId());
//        if (room == null) {
//            return ResponseData.fail400Response("房间ID错误");
//        }
//        User anchor = userMapper.selectById(room.getUserId());
//
//        GetRoomInfoResponse getRoomInfoResponse = GetRoomInfoResponse.builder()
//                .roomId(room.getHuanXinRoomId())
//                .roomNum(room.getRoomNum())
//                .roomName(room.getRoomName())
//                .roomAnnouncement(room.getAnnouncement())
//                .anchorAvatar(ConstantValue.completeAddress(anchor.getUserIcon()))
//                .anchorDesc(anchor.getUserDesc())
//                .anchorName(anchor.getUserName())
//                .isFollow(false)
//                .hotNum(room.getHotNum())
//                .userGold(0)
//                .userId("")
//                .liveUrl(room.getLiveUrl())
//                .userName("")
//                .userLevel(0)
//                .qqNum(StringUtils.isBlank(room.getQqNum()) ? "" : room.getQqNum())
//                .qqGroupNum(StringUtils.isBlank(room.getQqGroupNum()) ? "" : room.getQqGroupNum())
//                .wxNum(StringUtils.isBlank(room.getWxNum()) ? "" : room.getWxNum())
//                .roomStatus(room.getType())
//                .matchId(room.getMatchId())
//                .build();
//
//        String phone = null;
//        User account = null;
//        try {
//            phone = CommonMethod.getUserPhone(request);
//            account = userMapper.selectOne(Wrappers.<User>lambdaQuery()
//                    .eq(User::getPhone, phone)
//            );
//        } catch (RuntimeException e) {
//            log.info("未携带 token 创建游客账户!");
//            try {
//                String deviceId = CommonMethod.getDeviceId(request);
//                account = userMapper.selectOne(Wrappers.<User>lambdaQuery()
//                        .eq(User::getDeviceId, deviceId)
//                        .eq(User::getDeleted, 0)
//                );
//                if (account == null) {
//                    account = userService.registerByDeviceId(deviceId, IPUtil.getClientIP(request));
//                }
//            } catch (RuntimeException dev) {
//                log.error("未携带 DeviceId");
//            }
//        }
//
//        if (account != null) {
//            getRoomInfoResponse.setHuanXinUserId("live_" + account.getId() + surroundings);
//            getRoomInfoResponse.setUserId(account.getId());
//            getRoomInfoResponse.setUserName(account.getUserName());
//            getRoomInfoResponse.setUserLevel(account.getLevel());
//            Follow follow = followMapper.selectOne(Wrappers.<Follow>lambdaQuery()
//                    .eq(Follow::getUserId, account.getId())
//                    .eq(Follow::getFollowId, anchor.getId())
//                    .eq(Follow::getDeleted, 0)
//            );
//            if (follow != null) {
//                getRoomInfoResponse.setFollow(true);
//            }
//
//            UserGold userGold = userGoldMapper.selectById(account.getId());
//            if (userGold != null) {
//                getRoomInfoResponse.setUserGold(userGold.getGold());
//            }
//        }
//        return ResponseData.successResponse(getRoomInfoResponse);
//    }

//    @Override
//    public ResponseData<?> anchorMatch(CommonStringId commonIntId, HttpServletRequest request) {
//        if (commonIntId.getId() == null) {
//            return ResponseData.fail400Response("请传入房间ID");
//        }
//
//        Room room = roomMapper.selectById(commonIntId.getId());
//        if (room == null) {
//            return ResponseData.fail400Response("房间ID错误");
//        }
//
//        List<RoomHostRecord> roomHostRecords = roomHostRecordMapper.selectList(Wrappers.<RoomHostRecord>lambdaQuery()
//                .eq(RoomHostRecord::getDeleted, 0)
//                .eq(RoomHostRecord::getRoomId, commonIntId.getId())
//                .ge(RoomHostRecord::getStartTime, new Date())
//        );
//
//        if (roomHostRecords != null && roomHostRecords.size() > 0) {
//            Set<String> matches = roomHostRecords.stream().map(RoomHostRecord::getMatchId).collect(Collectors.toSet());
//            List<Match> matchDb = matchMapper.selectBatchIds(matches);
//
//            MatchListResponse matchListResponse = matchService.conversionByList(matchDb, false);
//
//            try {
//                String phone = CommonMethod.getUserPhone(request);
//                User account = userMapper.selectOne(Wrappers.<User>lambdaQuery()
//                        .eq(User::getPhone, phone)
//                );
//
//                List<UserReserve> userReserves = userReserveMapper.selectList(Wrappers.<UserReserve>lambdaQuery()
//                        .eq(UserReserve::getUserId, account.getId())
//                        .eq(UserReserve::getDeleted, 0)
//                );
//                Set<String> reservesId = userReserves.stream().map(UserReserve::getMatchId).collect(Collectors.toSet());
//
//                matchListResponse.getMatchLists().forEach(match -> {
//                    match.getMatchInfos().forEach(matchInfo -> {
//                        if (reservesId.contains(matchInfo.getId())) {
//                            matchInfo.setSubscribe(true);
//                        }
//                    });
//                });
//            } catch (RuntimeException e) {
//                log.info("未携带 token !");
//            }
//            return ResponseData.successResponse(matchListResponse);
//        }
//        return ResponseData.successResponse(MatchListResponse.builder()
//                .matchLists(new ArrayList<>())
//                .build());
//    }

//    @Override
//    public ResponseData<?> leaderboard(CommonStringId commonStringId) {
//        if (StringUtils.isBlank(commonStringId.getId())) {
//            return ResponseData.fail400Response("请传入房间ID");
//        }
//
//        Room room = roomMapper.selectById(commonStringId.getId());
//        if (room == null) {
//            return ResponseData.fail400Response("房间ID错误");
//        }
//
//        List<GetLeaderboardResponse> leaderboard = giftRecordMapper.getLeaderboard(room.getUserId());
//        List<LeaderboardResponse> responseList = leaderboard.stream().map(b -> {
//            User user = userMapper.selectById(b.getUserId());
//            return LeaderboardResponse.builder()
//                    .goldNum(b.getGiftGold())
//                    .userAvatar(ConstantValue.completeAddress(user.getUserIcon()))
//                    .userId(user.getId())
//                    .userLevel(user.getLevel())
//                    .userName(user.getUserName())
//                    .build();
//        }).collect(Collectors.toList());
//        return ResponseData.successResponse(responseList);
//    }

    @Override
    public ResponseData<?> getGiftList(HttpServletRequest request) {
        List<Gift> gifts = giftMapper.selectList(Wrappers.<Gift>lambdaQuery()
                .eq(Gift::getDeleted, 0)
                .le(Gift::getIsOnline, 1)
                .orderByAsc(Gift::getGiftGold)
        );

        List<GiftResponse> giftResponses = gifts.stream().map(gift -> GiftResponse.builder()
                .giftImg(ConstantValue.completeAddress(gift.getGiftIcon()))
                .giftName(gift.getGiftName())
                .gold(gift.getGiftGold())
                .id(gift.getId())
                .build()).collect(Collectors.toList());
        return ResponseData.successResponse(giftResponses);
    }

    @Override
    @Transactional
    public ResponseData<?> sendGift(SendGiftRequest sendGiftRequest, HttpServletRequest request) {
        Room room = roomMapper.selectById(sendGiftRequest.getRoomId());
        if (room == null) {
            return ResponseData.fail400Response("房间ID错误");
        }

        Gift gift = giftMapper.selectById(sendGiftRequest.getGiftId());
        if (gift == null) {
            return ResponseData.fail400Response("礼物ID错误");
        }

        try {
            String phone = CommonMethod.getUserPhone(request);
            User account = userMapper.selectOne(Wrappers.<User>lambdaQuery()
                    .eq(User::getPhone, phone)
            );

            String key = RedisKeySplicing.getPayLock(account.getId());
            if (!redisUtilsService.existsKey(key)) {
                redisUtilsService.cacheStringInfo(key, "1", 5L);

                UserGold userGold = userGoldMapper.selectById(account.getId());
                if (userGold.getGold() < gift.getGiftGold()) {
                    return ResponseData.failResponse(INSUFFICIENT_USER_COINS, "金币不足,请充值");
                }

                userGoldMapper.reducedAmount(gift.getGiftGold(), account.getId());

                giftRecordMapper.insert(GiftRecord.builder()
                        .giftGold(gift.getGiftGold())
                        .giftId(gift.getId())
                        .hostUserId(room.getUserId())
                        .userId(account.getId())
                        .build());

                redisUtilsService.removeByKey(key);
                return ResponseData.successResponse();
            }
        } catch (RuntimeException e) {
            log.info("未携带 token !");
        }
        return ResponseData.failResponse();
    }

    @Override
    public ResponseData<?> getBannerList(HttpServletRequest request) {
        List<Advert> adverts = advertMapper.selectList(Wrappers.<Advert>lambdaQuery()
                .eq(Advert::getDeleted, 0)
                .eq(Advert::getAdType, 1)
                .orderByDesc(Advert::getCreateTime)
        );

        List<BannerListResponse> bannerListResponses = adverts.stream().map(ad -> BannerListResponse.builder()
                .adDes(ad.getAdDesc())
                .adUrl(ad.getAdUrl())
                .picPath(ConstantValue.completeAddress(ad.getPicPath()))
                .title(ad.getAdTitle())
                .build()).collect(Collectors.toList());
        return ResponseData.successResponse(bannerListResponses);
    }

    @Override
    public ResponseData<?> getTopMatchList(HttpServletRequest request) {
        List<Match> matches = matchMapper.selectList(Wrappers.<Match>lambdaQuery()
                .eq(Match::getDeleted, 0)
                .eq(Match::getSourceType, 1)
                .le(Match::getMatchTime, DateUtil.getEndTime())
                .ge(Match::getMatchTime, DateUtil.getStartTime())
                .notIn(Match::getCompetitionStatus, 3, 4)
                .orderByAsc(Match::getMatchTime)
        );

        MatchListResponse matchListResponse = matchService.conversionByList(matches, true);

        try {
            String phone = CommonMethod.getUserPhone(request);
            User account = userMapper.selectOne(Wrappers.<User>lambdaQuery()
                    .eq(User::getPhone, phone)
            );

            List<UserReserve> userReserves = userReserveMapper.selectList(Wrappers.<UserReserve>lambdaQuery()
                    .eq(UserReserve::getUserId, account.getId())
                    .eq(UserReserve::getDeleted, 0)
            );
            Set<String> reservesId = userReserves.stream().map(UserReserve::getMatchId).collect(Collectors.toSet());

            matchListResponse.getMatchLists().forEach(match -> {
                match.getMatchInfos().forEach(matchInfo -> {
                    if (reservesId.contains(matchInfo.getId())) {
                        matchInfo.setSubscribe(true);
                    }
                });
            });
        } catch (RuntimeException e) {
            log.info("未携带 token !");
        }
        return ResponseData.successResponse(matchListResponse);
    }

    @Override
    public ResponseData<?> getHotLiveList(CommonPage commonPage, HttpServletRequest request) {
        int jumpNum = (commonPage.getPageNum() - 1) * commonPage.getPageSize();

        long roomNum = roomMapper.selectCount(Wrappers.<Room>lambdaQuery()
                .eq(Room::getType, 1)
                .eq(Room::getDeleted, 0)
                .eq(Room::getShowHot, 1)
        );

        List<Room> rooms = roomMapper.selectList(Wrappers.<Room>lambdaQuery()
                .eq(Room::getType, 1)
                .eq(Room::getDeleted, 0)
                .eq(Room::getShowHot, 1)
                .orderByDesc(Room::getHotNum)
                .last(String.format(" limit %s,%s", jumpNum, commonPage.getPageSize()))
        );

        return ResponseData.successResponse(new ResultPage<>(commonPage.getPageNum(), commonPage.getPageSize(), (int) roomNum, conversion(rooms)));
    }

    @Override
    public ResponseData<?> getTopLiveList(TopLiveRequest commonPage, HttpServletRequest request) {
        int jumpNum = (commonPage.getPageNum() - 1) * commonPage.getPageSize();
        QueryWrapper<Room> wrapper = new QueryWrapper<>();
        wrapper.eq("type", 1);
        wrapper.eq("deleted", 0);
        if (commonPage.getType() != null && commonPage.getType() != -1) {
            wrapper.apply("FIND_IN_SET({0},category_id)", CommonMethod.convertCategory(commonPage.getType()));
        }

        long roomNum = roomMapper.selectCount(wrapper);

        wrapper.orderByDesc("hot_num");
        wrapper.last(String.format(" limit %s,%s", jumpNum, commonPage.getPageSize()));
        List<Room> rooms = roomMapper.selectList(wrapper);

        ResultPage result = new ResultPage<>(commonPage.getPageNum(),
                commonPage.getPageSize(), (int) roomNum, conversion(rooms));
        return ResponseData.successResponse(result);
    }

    @Override
    public ResponseData<?> getVideoRecommendList(VideoRecommendRequest commonPage, HttpServletRequest request) {
        int jumpNum = (commonPage.getPageNum() - 1) * commonPage.getPageSize();

        QueryWrapper<Room> wrapper = new QueryWrapper<>();
        wrapper.eq("type", 1);
        wrapper.eq("deleted", 0);

        if (commonPage.getRoomId() != null) {
            wrapper.ne("huan_xin_room_id", commonPage.getRoomId());
        }

        long roomNum = roomMapper.selectCount(wrapper);

        wrapper.orderByDesc("hot_num");
        wrapper.last(String.format(" limit %s,%s", jumpNum, commonPage.getPageSize()));
        List<Room> rooms = roomMapper.selectList(wrapper);

        return ResponseData.successResponse(new ResultPage<>(commonPage.getPageNum(),
                commonPage.getPageSize(), (int) roomNum, conversion(rooms)));
    }

    @Override
    public ResponseData<?> getChatMessage(HttpServletRequest request) {
        List<RobotMsg> robotMsgs = robotMsgMapper.selectList(Wrappers.<RobotMsg>lambdaQuery()
                .eq(RobotMsg::getDeleted, 0)
        );
        String msgContent = "";
        if (robotMsgs.size() > 0) {
            Collections.shuffle(robotMsgs);
            msgContent = robotMsgs.get(0).getFakeMsg();
        } else {
            msgContent = ChatFakeMsgUtils.generateContent();
        }
        return ResponseData.successResponse(FakeChatMsgResponse.builder()
                .sendUser(ChatFakeMsgUtils.generateName())
                .msgContent(msgContent)
                .build());
    }

    @Override
    public ResponseData<?> getHotAnchor(HttpServletRequest request) {
        List<User> users = userMapper.selectList(Wrappers.<User>lambdaQuery()
                .eq(User::getHotAnchor, 1)
                .eq(User::getDeleted, 0)
                .eq(User::getType, 1)
        );

        List<MatchListResponse.GuideResponse> guides = new ArrayList<>();
        for (User user : users) {
            Room room = roomMapper.selectOne(Wrappers.<Room>lambdaQuery()
                    .eq(Room::getDeleted, 0)
                    .eq(Room::getType, 1)
                    .eq(Room::getUserId, user.getId())
            );
            if (room != null) {
                guides.add(MatchListResponse.GuideResponse.builder()
                        .id(user.getId())
                        .status(room.getType())
                        .name(user.getUserName())
                        .recommend(room.getRecommend())
                        .roomId(room.getHuanXinRoomId())
                        .avatar(ConstantValue.completeAddress(user.getUserIcon()))
                        .build());
            }
        }
        return ResponseData.successResponse(guides);
    }

    @Override
    public ResponseData<?> getPushStreamUrl(CommonStringId commonStringId, HttpServletRequest request) {
        SystemConfig systemConfig = systemConfigMapper.selectOne(Wrappers.<SystemConfig>lambdaQuery()
                .eq(SystemConfig::getConfigName, SystemConfigEnum.LIVE_PUSH_URL.getTitle())
                .eq(SystemConfig::getDeleted, 0)
        );

        String safeUrl = PushStreamUtil.getSafeUrl("cd3f88a5a396b9ecff178d492337b57d", commonStringId.getId(), System.currentTimeMillis());
        PushStreamDTO streamDTO = new PushStreamDTO();
        streamDTO.setPushUrl(systemConfig.getConfigValue());
        streamDTO.setPushKey(safeUrl);
        return ResponseData.successResponse(streamDTO);
    }

    @Override
    public ResponseData<?> getLiveStreamingToPc(HttpServletRequest request) {
        List<Room> homeRoom = roomMapper.selectList(Wrappers.<Room>lambdaQuery()
                .eq(Room::getShowHome, 1)
                .eq(Room::getDeleted, 0)
                .eq(Room::getType, 1)
                .orderByDesc(Room::getHotNum)
                .last(String.format(" limit %s,%s", 0, 5))
        );

        return ResponseData.successResponse(conversion(homeRoom));
    }

    @Override
    public ResponseData<?> getRoomList(GetRoomListRequest commonPage, HttpServletRequest request) {
        QueryWrapper<Room> wrapper = new QueryWrapper<>();
        wrapper.eq("deleted", 0);

        if (commonPage.getType() != null) {
            wrapper.eq("type", commonPage.getType());
        }
        if (commonPage.getRecommend() != null) {
            wrapper.eq("recommend", commonPage.getRecommend());
        }
        if (commonPage.getShowHome() != null) {
            wrapper.eq("show_home", commonPage.getShowHome());
        }

        long count = roomMapper.selectCount(wrapper);

        wrapper.orderByDesc("hot_num");
        int jumpNum = (commonPage.getPageNum() - 1) * commonPage.getPageSize();
        wrapper.last(String.format(" limit %s,%s", jumpNum, commonPage.getPageSize()));

        List<Room> rooms = roomMapper.selectList(wrapper);

        List<GetRoomListResponse> result = rooms.stream().map(b -> {
            User user = userMapper.selectById(b.getUserId());
            Category category = categoryMapper.selectById(b.getCategoryId());
            String categoryName = null;
            if (category != null) {
                categoryName = category.getSportsName();
            }
            Match match = matchMapper.selectById(b.getMatchId());
            String matchName = "未关联比赛";
            String matchShortName = "未关联比赛";
            if (match != null) {
                matchName = match.getCompetitionName();
                matchShortName = match.getCompetitionShortName();
            }
            return GetRoomListResponse.builder()
                    .anchorId(b.getUserId())
                    .anchorName(user.getUserName())
                    .announcement(b.getAnnouncement())
                    .categoryId(b.getCategoryId())
                    .categoryName(categoryName)
                    .createTime(DateUtil.format(b.getCreateTime(), DateUtil.YMDHMS_))
                    .hotNum(b.getHotNum())
                    .huanXinRoomId(b.getHuanXinRoomId())
                    .likeNum(b.getLikeNum())
                    .liveUrl(b.getLiveUrl())
                    .matchId(b.getMatchId())
                    .matchName(matchName)
                    .matchShortName(matchShortName)
                    .recommend(b.getRecommend())
                    .roomCover(ConstantValue.completeAddress(b.getRoomCover()))
                    .roomDesc(b.getRoomDesc())
                    .roomName(b.getRoomName())
                    .roomNum(b.getRoomNum())
                    .showHome(b.getShowHome())
                    .showHot(b.getShowHot())
                    .type(b.getType())
                    .useCutImg(b.getUseCutImg())
                    .qqNum(b.getQqNum())
                    .wxNum(b.getWxNum())
                    .qqGroupNum(b.getQqGroupNum())
                    .enableFakeChat(b.getEnableFakeChat())
                    .build();
        }).collect(Collectors.toList());

        return ResponseData.successResponse(new ResultPage<>(commonPage.getPageNum(), commonPage.getPageSize(), (int) count, result));
    }

    @Override
    @Transactional
    public ResponseData<?> editRoomById(EditRoomRequest editRoomRequest, HttpServletRequest request) {
        Room room = roomMapper.selectById(editRoomRequest.getRoomId());
        if (room == null) {
            return ResponseData.fail400Response("roomId 错误未找到房间");
        }
        if (editRoomRequest.getType() != null) {
            if (editRoomRequest.getType() == 1) {
                List<Room> rooms = roomMapper.selectList(Wrappers.<Room>lambdaQuery()
                        .eq(Room::getType, 1)
                        .eq(Room::getDeleted, 0)
                        .eq(Room::getUserId, room.getUserId())
                );
                if (rooms.size() != 0 && !rooms.get(0).getHuanXinRoomId().equals(room.getHuanXinRoomId())) {
                    return ResponseData.fail400Response("一个主播同时只能开播一个房间");
                }
            }
            room.setType(editRoomRequest.getType());
        }
        if (editRoomRequest.getDeleted() != null) {
            room.setDeleted(editRoomRequest.getDeleted());
            Match match = matchMapper.selectById(room.getMatchId());
            if (match != null) {
                RoomHostRecord record = roomHostRecordMapper.selectOne(Wrappers.<RoomHostRecord>lambdaQuery()
                        .eq(RoomHostRecord::getRoomId, room.getHuanXinRoomId())
                        .eq(RoomHostRecord::getMatchTime, match.getMatchTime())
                        .eq(RoomHostRecord::getHostUserId, room.getUserId())
                );
                if (editRoomRequest.getDeleted() == 1 && record != null) {
                    roomHostRecordMapper.deleteById(record.getId());
                }
            }
        }
        if (editRoomRequest.getRecommend() != null) {
            room.setRecommend(editRoomRequest.getRecommend());
        }
        if (editRoomRequest.getShowHome() != null) {
            room.setShowHome(editRoomRequest.getShowHome());
        }
        if (editRoomRequest.getShowHot() != null) {
            room.setShowHot(editRoomRequest.getShowHot());
        }
        if (editRoomRequest.getHotNum() != null) {
            room.setHotNum(editRoomRequest.getHotNum());
        }
        if (StringUtils.isNotBlank(editRoomRequest.getLiveUrl())) {
            room.setLiveUrl(editRoomRequest.getLiveUrl());
        }
        if (StringUtils.isNotBlank(editRoomRequest.getRoomCover())) {
            room.setRoomCover(editRoomRequest.getRoomCover());
        }
        if (StringUtils.isNotBlank(editRoomRequest.getRoomName())) {
            room.setRoomName(editRoomRequest.getRoomName());
        }
        if (StringUtils.isNotBlank(editRoomRequest.getRoomDesc())) {
            room.setRoomDesc(editRoomRequest.getRoomDesc());
        }
        if (StringUtils.isNotBlank(editRoomRequest.getAnnouncement())) {
            room.setAnnouncement(editRoomRequest.getAnnouncement());
        }
        if (StringUtils.isNotBlank(editRoomRequest.getQqNum())) {
            room.setQqNum(editRoomRequest.getQqNum());
        }
        if (StringUtils.isNotBlank(editRoomRequest.getWxNum())) {
            room.setWxNum(editRoomRequest.getWxNum());
        }
        if (StringUtils.isNotBlank(editRoomRequest.getQqGroupNum())) {
            room.setQqGroupNum(editRoomRequest.getQqGroupNum());
        }

        room.setEnableFakeChat(editRoomRequest.getEnableFakeChat());
        if (StringUtils.isNotBlank(editRoomRequest.getMatchId())) {
            Match match = matchMapper.selectById(editRoomRequest.getMatchId());
            if (match == null) {
                return ResponseData.fail400Response("MatchId 错误");
            }
            room.setMatchId(editRoomRequest.getMatchId());

            RoomHostRecord record = roomHostRecordMapper.selectOne(Wrappers.<RoomHostRecord>lambdaQuery()
                    .eq(RoomHostRecord::getRoomId, room.getHuanXinRoomId())
                    .eq(RoomHostRecord::getMatchTime, match.getMatchTime())
                    .eq(RoomHostRecord::getHostUserId, room.getUserId())
            );
            if (record == null) {
                roomHostRecordMapper.insert(RoomHostRecord.builder()
                        .matchTime(match.getMatchTime())
                        .endTime(DateUtil.getTmEndTime())
                        .startTime(new Date())
                        .hostUserId(room.getUserId())
                        .matchId(match.getId())
                        .roomId(room.getHuanXinRoomId())
                        .build());
            } else {
                record.setEndTime(DateUtil.getTmEndTime());
                record.setMatchId(match.getId());
                record.setHostUserId(room.getUserId());
                record.setRoomId(room.getHuanXinRoomId());
                record.setStartTime(new Date());
                record.setMatchTime(match.getMatchTime());
                roomHostRecordMapper.updateById(record);
            }
        }
        roomMapper.updateById(room);
        return ResponseData.successResponse();
    }

    @Override
    public ResponseData<?> getRobotMsgList(CommonPage commonPage, HttpServletRequest request) {
        long count = robotMsgMapper.selectCount(Wrappers.<RobotMsg>lambdaQuery()
                .eq(RobotMsg::getDeleted, 0)
        );
        int jumpNum = (commonPage.getPageNum() - 1) * commonPage.getPageSize();
        List<RobotMsg> robotMsgs = robotMsgMapper.selectList(Wrappers.<RobotMsg>lambdaQuery()
                .eq(RobotMsg::getDeleted, 0)
                .orderByDesc(RobotMsg::getCreateTime)
                .last(String.format(" limit %s,%s", jumpNum, commonPage.getPageSize()))
        );

        return ResponseData.successResponse(new ResultPage<>(commonPage.getPageNum(), commonPage.getPageSize(), (int) count, robotMsgs));
    }

    @Override
    public ResponseData<?> editRobotMsg(EditRobotMsgRequest editRobotMsgRequest, HttpServletRequest request) {
        if(editRobotMsgRequest.getId() == null || editRobotMsgRequest.getId() == 0){
            robotMsgMapper.insert(RobotMsg.builder()
                    .fakeMsg(editRobotMsgRequest.getFakeMsg())
                    .build());
        } else {
            RobotMsg robotMsg = robotMsgMapper.selectById(editRobotMsgRequest.getId());
            if(robotMsg != null){
                robotMsg.setFakeMsg(editRobotMsgRequest.getFakeMsg());
                robotMsg.setDeleted(editRobotMsgRequest.getDeleted());
                robotMsgMapper.updateById(robotMsg);
            }
        }
        return ResponseData.successResponse();
    }


    @Override
    public Room getVacancyRoom() {
        //获取补缺主播
        List<User> users = userMapper.selectList(Wrappers.<User>lambdaQuery()
                .eq(User::getDeleted, 0)
                .eq(User::getType, 1)
                .eq(User::getRole, 2)
        );
        if (users.size() == 0) {
            return null;
        }

        //获取不在直播状态的补缺主播的房间
        List<Room> rooms = roomMapper.selectList(Wrappers.<Room>lambdaQuery()
                .eq(Room::getDeleted, 0)
                .eq(Room::getType, 0)
                .in(Room::getUserId, users.stream().map(BaseStringIdEntity::getId).collect(Collectors.toList()))
        );
        if (rooms.size() > 0) {
            return rooms.get(0);
        } else {
            return null;
        }
    }

    @Override
    @Transactional
    public void createVacancyRoom(String roomName, Integer type, String liveUrl) {
        String userId = IdGen.uuid();
        userService.createAccount(CreateAccountRequest.builder()
                .userId(userId)
                .channelId("0")
                .nickName(IdGen.randomLong(6))
                .role(2)
                .sex(0)
                .type(1)
                .build());

        liveService.createRoom(CreateRoomRequest.builder()
                .deleted(0)
                .roomCutImg("https://zhibo-1305013790.cos.ap-nanjing.myqcloud.com/2021-05-27/sd-1-3526839-screenshot-16-41-41-1280x720.jpg")
                .matchId(null)
                .useCutImg(0)
                .roomAnnouncement("欢迎来到主播房间，欢迎下载APP")
                .roomName(roomName)
                .type(type)
                .build(), userId, liveUrl);
    }

    private List<GetHotLiveResponse> conversion(List<Room> rooms) {
        return rooms.stream().map(room -> {
            User user = userMapper.selectById(room.getUserId());

            return GetHotLiveResponse.builder()
                    .roomId(room.getHuanXinRoomId())
                    .roomName(room.getRoomName())
                    .hotNum(room.getHotNum())
                    .matchCutImg(ConstantValue.completeAddress(room.getRoomCover()))
                    .anchorName(user.getUserName())
                    .roomStatus(room.getType())
                    .recommend(room.getRecommend())
                    .liveUrl(room.getLiveUrl())
                    .anchorAvatar(ConstantValue.completeAddress(user.getUserIcon()))
                    .build();
        }).collect(Collectors.toList());
    }

}
